package com.bcxin.tenant.apis.impls;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.PlatformOperateLogRpcProvider;
import com.bcxin.api.interfaces.tenants.DepartmentRpcProvider;
import com.bcxin.api.interfaces.tenants.TenantDepartmentScheduledDeleteRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.operatelog.PlatformOperateLogRequest;
import com.bcxin.tenant.domain.entities.DepartmentEntity;
import com.bcxin.tenant.domain.entities.EmployeeEntity;
import com.bcxin.tenant.domain.entities.PlatformOperateLogEntity;
import com.bcxin.tenant.domain.entities.TenantDepartmentScheduledDeleteEntity;
import com.bcxin.tenant.domain.repositories.DepartmentRepository;
import com.bcxin.tenant.domain.repositories.EmployeeRepository;
import com.bcxin.tenant.domain.repositories.TenantDepartmentScheduledDeleteRepository;
import com.bcxin.tenant.domain.repository.impls.PlatformOperateLogJpaRepository;
import com.bcxin.tenant.domain.services.DepartmentService;
import com.bcxin.tenant.domain.services.EmployeeService;
import com.bcxin.tenant.domain.services.commands.BatchDeleteDepartmentCommand;
import com.bcxin.tenant.domain.services.commands.BatchLeaveEmployeeCommand;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.beans.BeanUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * description：定时删除部门表
 * author：linchunpeng
 * date：2024/12/13
 */
@Slf4j
@DubboService(version = ApiConstant.VERSION, validation = "true", timeout = 120 * 1000, retries = 0)
public class TenantDepartmentScheduledDeleteRpcProviderImpl implements TenantDepartmentScheduledDeleteRpcProvider {

    private final UnitWork unitWork;

    private final TenantDepartmentScheduledDeleteRepository tenantDepartmentScheduledDeleteRepository;
    private final DepartmentRepository departmentRepository;
    private final DepartmentService departmentService;
    private final EmployeeService employeeService;
    private final EmployeeRepository employeeRepository;
    private final CacheV2Provider cacheV2Provider;

    public TenantDepartmentScheduledDeleteRpcProviderImpl(UnitWork unitWork,
                                                          TenantDepartmentScheduledDeleteRepository tenantDepartmentScheduledDeleteRepository,
                                                          DepartmentRepository departmentRepository,
                                                          DepartmentService departmentService,
                                                          EmployeeService employeeService,
                                                          EmployeeRepository employeeRepository,
                                                          CacheV2Provider cacheV2Provider) {
        this.unitWork = unitWork;
        this.tenantDepartmentScheduledDeleteRepository = tenantDepartmentScheduledDeleteRepository;
        this.departmentRepository = departmentRepository;
        this.departmentService = departmentService;
        this.employeeService = employeeService;
        this.employeeRepository = employeeRepository;
        this.cacheV2Provider = cacheV2Provider;
    }

    /**
     * description：定时删除部门
     * author：linchunpeng
     * date：2024/12/13
     */
    @Override
    public void executeDelete() {
        try {
            log.info("定时删除部门");
            Date nowTime = new Date();
            //先查询出：删除时间 < 当前时间 && 实际删除时间是null 的数据
            List<TenantDepartmentScheduledDeleteEntity> departmentDeleteEntityList = tenantDepartmentScheduledDeleteRepository.findByScheduledDeleteTimeIsLessThanEqualAndActualDeleteTimeIsNull(nowTime);
            if (CollectionUtil.isNotEmpty(departmentDeleteEntityList)) {
                for (TenantDepartmentScheduledDeleteEntity departmentDeleteEntity : departmentDeleteEntityList) {
                    //查询部门
                    Optional<DepartmentEntity> departmentEntityOptional = departmentRepository.findById(departmentDeleteEntity.getDepartmentId());
                    if (departmentEntityOptional.isPresent()) {
                        DepartmentEntity departmentEntity = departmentEntityOptional.get();
                        String organizationId = departmentEntity.getOrganization().getId();
                        //查询自己和子部门列表
                        List<DepartmentEntity> departmentEntityList = departmentRepository.findBySelfAndChild(departmentDeleteEntity.getOrganizationId(), departmentEntity.getIndexTree().concat("%"));
                        if (CollectionUtil.isNotEmpty(departmentEntityList)) {
                            //按照层级排序
                            departmentEntityList.sort((department1, department2) -> department2.getIndexTree().length() - department1.getIndexTree().length());
                            //循环删除
                            this.unitWork.executeTran(() -> {
                                //执行部门人员批量离职操作
                                List<String> leaveIdList = employeeRepository.findIdListByDepartmentIndexTree(departmentEntity.getIndexTree().concat("%"));
                                if (CollectionUtil.isNotEmpty(leaveIdList)) {
                                    employeeService.dispatch(BatchLeaveEmployeeCommand.create(leaveIdList, organizationId, new Date(), "项目到期终止，人员自动离职"));
                                }
                                //删除部门以及子部门
                                List<String> deleteDepartmentIdList = departmentEntityList.stream().map(DepartmentEntity::getId).collect(Collectors.toList());
                                for (String deleteDepartmentId : deleteDepartmentIdList) {
                                    departmentService.dispatch(BatchDeleteDepartmentCommand.create(organizationId, Collections.singletonList(deleteDepartmentId)));
                                }
                                //清理缓存
                                cacheV2Provider.delFuzzy(String.format("tenant:department:cache:%s*", organizationId));
                                //设置实际删除时间
                                departmentDeleteEntity.setActualDeleteTime(nowTime);
                                tenantDepartmentScheduledDeleteRepository.save(departmentDeleteEntity);
                            });
                        }
                    }
                }
            }
        } catch (Exception e) {
            log.error("定时删除部门异常：{}", e.getMessage(), e);
        }
    }

}
