package com.bcxin.tenant.apis.impls;

import cn.hutool.core.util.ReUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.utils.RedisUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.SuperviseDepartmentRpcProvider;
import com.bcxin.api.interfaces.commons.responses.SuperviseDepartmentResponse;
import com.bcxin.tenant.domain.entities.SuperviseDepartmentEntity;
import com.bcxin.tenant.domain.repositories.SuperviseDepartmentRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * description: 监管机构
 * author: linchunpeng
 * date:  2023-05-10 9:46
 */
@Slf4j
@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class SuperviseDepartmentRpcProviderImpl implements SuperviseDepartmentRpcProvider {


    private final SuperviseDepartmentRepository superviseDepartmentRepository;

    private final RedisUtil redisUtil;

    private final static String SUPERVISE_DEPARTMENT_REDIS_KEY = "bcxin:tenant:platform:supervise:%s:%s";

    /**
     * 行业代码到区域ID的映射配置
     * key: 行业代码, value: 区域ID映射规则（Map<原始区域前缀, 目标区域ID>）
     * 如果行业不在映射表中，使用原始 regionId
     */
    private static final Map<String, Map<String, String>> INDUSTRY_REGION_MAPPING = new HashMap<String, Map<String, String>>() {{
        // 群防群治行业（06）
        put("06", new HashMap<String, String>() {{
            put("11", "010000"); // 北京地区映射到 010000
        }});
        // 内保行业（04）
        put("04", new HashMap<String, String>() {{
            put("11", "020000"); // 北京地区映射到 020000
        }});
    }};

    public SuperviseDepartmentRpcProviderImpl(SuperviseDepartmentRepository superviseDepartmentRepository, RedisUtil redisUtil){
        this.superviseDepartmentRepository = superviseDepartmentRepository;
        this.redisUtil = redisUtil;
    }

    /**
     * description: 根据区域id获取监管机构树形结构
     * author: linchunpeng
     * date:  2023-05-10 9:37
     */
    @Override
    public SuperviseDepartmentResponse findByRegionId(String regionId, String industryCode) {
        if (StringUtils.isBlank(regionId) || StringUtils.isBlank(industryCode)) {
            return null;
        }

        // 根据行业代码获取区域映射规则
        String actualRegionId = getActualRegionId(regionId, industryCode);

        // 使用组合的缓存key（包含行业代码和区域ID）
        String cacheKey = String.format(SUPERVISE_DEPARTMENT_REDIS_KEY, actualRegionId, industryCode);
        Object redisCache = redisUtil.get(cacheKey);
        if (redisCache != null) {
            return JSONObject.parseObject(redisCache.toString(), SuperviseDepartmentResponse.class);
        }

        //根据区域和行业代码查询监管机构（使用实际映射后的区域ID）
        String regionStartWith = ReUtil.replaceAll(actualRegionId, "0+$", "");
        List<SuperviseDepartmentEntity> departmentList = superviseDepartmentRepository.findByRegionIdStartWithAndIndustryCode(regionStartWith, industryCode);

        if (CollectionUtils.isNotEmpty(departmentList)) {
            //是否只返回两个层级
            boolean isReturnSecond  = regionId.startsWith("11");
            String root = "";
            //监管机构集合
            Map<String, SuperviseDepartmentResponse> departmentMap = new HashMap<>();
            for (SuperviseDepartmentEntity entity : departmentList) {
                departmentMap.put(entity.getId(), new SuperviseDepartmentResponse(entity.getId(), entity.getName(), entity.getSuperior(), entity.getIndexCode(), null));
                if (StringUtils.isBlank(entity.getSuperior())) {
                    //根节点id
                    root = entity.getId();
                }
            }
            //循环组装子节点
            for (SuperviseDepartmentEntity entity : departmentList) {
                SuperviseDepartmentResponse node = departmentMap.get(entity.getId());
                //判断是否只返回两个层级
                if (isReturnSecond) {
                    //取出indexCode判断层级
                    String indexCode = node.getIndexCode();
                    if (indexCode.split("___").length > 2) {
                        continue;
                    }
                }

                SuperviseDepartmentResponse superior = departmentMap.get(entity.getSuperior());
                if (superior != null) {
                    if (CollectionUtils.isEmpty(superior.getChildren())) {
                        superior.setChildren(new ArrayList<>());
                    }
                    superior.getChildren().add(node);
                }
            }
            SuperviseDepartmentResponse result = departmentMap.get(root);
            //放入缓存（使用组合的缓存key）
            redisUtil.set(cacheKey, JSONObject.toJSONString(result), 3600);
            return result;
        }
        return null;
    }

    /**
     * 根据行业代码和原始区域ID获取实际使用的区域ID
     * @param regionId 原始区域ID
     * @param industryCode 行业代码
     * @return 实际使用的区域ID
     */
    private String getActualRegionId(String regionId, String industryCode) {
        // 获取该行业的区域映射规则
        Map<String, String> regionMapping = INDUSTRY_REGION_MAPPING.get(industryCode);
        
        if (regionMapping != null) {
            // 遍历映射规则，查找匹配的区域前缀
            for (Map.Entry<String, String> entry : regionMapping.entrySet()) {
                String regionPrefix = entry.getKey();
                if (regionId.startsWith(regionPrefix)) {
                    String mappedRegionId = entry.getValue();
                    log.info("行业代码 {} - 区域 {} 映射到监管机构区域：{}", industryCode, regionId, mappedRegionId);
                    return mappedRegionId;
                }
            }
        }
        
        // 如果没有匹配的映射规则，使用原始区域ID
        return regionId;
    }

    /**
     * description: 清除redis缓存
     * author: linchunpeng
     * date:  2023-05-10 18:19
     */
    @Override
    public void clearRedis(String regionId) {
        // 清除所有行业的缓存（遍历所有可能的行业代码）
        for (String industryCode : INDUSTRY_REGION_MAPPING.keySet()) {
            String actualRegionId = getActualRegionId(regionId, industryCode);
            String cacheKey = String.format(SUPERVISE_DEPARTMENT_REDIS_KEY, actualRegionId, industryCode);
            redisUtil.del(cacheKey);
        }
        // 也清除原始区域ID的缓存（兼容旧格式）
        String oldCacheKey = String.format("bcxin:tenant:platform:supervise:%s", regionId);
        redisUtil.del(oldCacheKey);
    }
}
