package com.bcxin.tenant.apis.translates.impls;

import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.api.interfaces.tenants.responses.DepartmentGetResponse;
import com.bcxin.api.interfaces.tenants.responses.DepartmentTreeGetResponse;
import com.bcxin.tenant.apis.translates.DepartmentTranslate;
import com.bcxin.Infrastructures.enums.DepartImPermissionType;
import com.bcxin.tenant.domain.repositories.dtos.DepartmentDto;
import com.bcxin.tenant.domain.snapshots.DepartImAllowedDepartSnapshot;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

@Component
public class DepartmentTranslateImpl implements DepartmentTranslate {
    private final JsonProvider jsonProvider;

    public DepartmentTranslateImpl(JsonProvider jsonProvider) {
        this.jsonProvider = jsonProvider;
    }

    @Override
    public DepartmentGetResponse translate(DepartmentDto departmentDto) {

        Collection<String> allowedDepartIds = null;
        DepartImAllowedDepartSnapshot snapshot = this.jsonProvider.toObject(DepartImAllowedDepartSnapshot.class, departmentDto.getPermissionConfig());
        if (snapshot != null) {
            allowedDepartIds = snapshot.getDepartIds();
        }

        return new DepartmentGetResponse(departmentDto.getId(),
                departmentDto.getName(),
                departmentDto.getParentId(),
                departmentDto.getDisplayOrder(),

                departmentDto.getTotalMember(),
                translate2DepartIMPermissionType(departmentDto.getPermissionType()),
                allowedDepartIds,
                departmentDto.getLevel());
    }

    @Override
    public Collection<DepartmentTreeGetResponse> translate2TreeGetResponse(Collection<DepartmentDto> departments) {
        if (CollectionUtils.isEmpty(departments)) {
            return Collections.EMPTY_LIST;
        }

        int minLevel = departments.stream().map(ii -> ii.getLevel()).min(Integer::compare).get();

        Collection<DepartmentTreeGetResponse> rootResponses =
                departments.stream()
                        .filter(ii -> ii.getLevel() == minLevel)
                        .map(ii -> {
                            return translate2DepartmentTreeGetResponse(departments, ii, jsonProvider);
                        })
                        .sorted((ix, it) -> {
                            return ix.getDisplayOrder() - it.getDisplayOrder();
                        }).collect(Collectors.toList());
        Collection<String> allProcessedDepartIds = rootResponses.stream().flatMap(ii -> ii.getAllDepartIds().stream())
                .collect(Collectors.toList());
        Collection<DepartmentDto> unProcessedDepartments =
                departments.stream().filter(ii -> !allProcessedDepartIds.contains(ii.getId()))
                        .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(unProcessedDepartments)) {
            rootResponses.addAll(translate2TreeGetResponse(unProcessedDepartments));
        }

        return rootResponses;
    }

    private static Collection<DepartmentTreeGetResponse> buildChildNodes(JsonProvider jsonProvider, DepartmentDto parent, Collection<DepartmentDto> departments) {
        return
                departments.stream().filter(ii -> parent.getId().equals(ii.getParentId()))
                        .map(ii -> translate2DepartmentTreeGetResponse(departments,ii,jsonProvider)).collect(Collectors.toList());
    }


    private static com.bcxin.api.interfaces.enums.DepartImPermissionType translate2DepartIMPermissionType(DepartImPermissionType permissionType) {
        switch (permissionType) {
            case JustOnDepart:
                return com.bcxin.api.interfaces.enums.DepartImPermissionType.JustOnDepart;
            case Whole:
                return com.bcxin.api.interfaces.enums.DepartImPermissionType.Whole;
            case Special:
                return com.bcxin.api.interfaces.enums.DepartImPermissionType.Special;
            case JustSelf:
                return com.bcxin.api.interfaces.enums.DepartImPermissionType.JustSelf;
            case OnDepartAndSub:
                return com.bcxin.api.interfaces.enums.DepartImPermissionType.OnDepartAndSub;
            default:
                throw new NotSupportTenantException(String.format("非法类型(%s)", permissionType));
        }
    }


    private static DepartmentTreeGetResponse translate2DepartmentTreeGetResponse(
            Collection<DepartmentDto> departments,
            DepartmentDto selectedDepartment,JsonProvider jsonProvider) {
        DepartmentDto ii = selectedDepartment;
        String config = ii.getPermissionConfig();
        DepartImAllowedDepartSnapshot snapshot = jsonProvider.toObject(DepartImAllowedDepartSnapshot.class, config);

        Collection<DepartmentTreeGetResponse> children = buildChildNodes(jsonProvider, ii, departments);
        Collection<DepartmentTreeGetResponse.DataItem> dataItems = null;
        if (snapshot != null && !CollectionUtils.isEmpty(snapshot.getDepartIds())) {
            dataItems = departments.stream().filter(ix -> snapshot.getDepartIds().stream().anyMatch(di -> ix.getId().equalsIgnoreCase(di)))
                    .map(ix -> DepartmentTreeGetResponse.DataItem.create(ix.getId(), ix.getName()))
                    .collect(Collectors.toList());
        }

        DepartmentTreeGetResponse item = DepartmentTreeGetResponse.create(ii.getId(), ii.getName(), ii.getParentId(),
                ii.getDisplayOrder(), ii.getTotalMember(), ii.getLevel(),
                translate2DepartIMPermissionType(ii.getPermissionType()),
                snapshot == null ? null : snapshot.getDepartIds(), children,
                dataItems);

        return item;
    }
}
