package com.bcxin.tenant.open.dubbo.writer.providers.translates;

import com.bcxin.tenant.open.document.domains.documents.*;
import com.bcxin.tenant.open.document.domains.repositories.*;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdDispatchDataScopeDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdEmployeeDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdSecurityStationDocumentRepository;
import com.bcxin.tenant.open.document.domains.utils.GeoPointUtils;
import com.bcxin.tenant.open.domains.entities.*;
import com.bcxin.tenant.open.domains.readers.RdCompositedReader;
import com.bcxin.tenant.open.domains.repositories.AttendanceRepository;
import com.bcxin.tenant.open.domains.services.commands.CreateRoomCommandResponse;
import com.bcxin.tenant.open.domains.services.commands.InviteRoomUserCommandResponse;
import com.bcxin.tenant.open.domains.utils.RdCompanyEntityUtils;
import com.bcxin.tenant.open.domains.utils.RdEmployeeEntityUtils;
import com.bcxin.tenant.open.domains.utils.RdSecurityStationEntityUtil;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.requests.CreateRoomWriterRpcRequest;
import com.bcxin.tenant.open.jdks.requests.InviteRoomUsersWriterRpcRequest;
import com.bcxin.tenant.open.jdks.requests.OrgPursePointRequest;
import com.bcxin.tenant.open.jdks.responses.*;
import com.redis.om.spring.search.stream.EntityStream;
import com.redis.om.spring.search.stream.SearchStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.data.geo.Point;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import java.util.stream.Collectors;

@Component
public class DataTranslateImpl implements DataTranslate {
    private static final Logger logger = LoggerFactory.getLogger(DataTranslateImpl.class);
    private final BeanFactory beanFactory;
    private final EntityStream entityStream;

    private final RdDispatchDataScopeDocumentRepository dispatchDataScopeDocumentRepository;
    private final RdCompanyDocumentRepository companyDocumentRepository;
    private final JsonProvider jsonProvider;

    public DataTranslateImpl(BeanFactory beanFactory, EntityStream entityStream,
                             RdDispatchDataScopeDocumentRepository dispatchDataScopeDocumentRepository,
                             RdCompanyDocumentRepository companyDocumentRepository, JsonProvider jsonProvider) {
        this.beanFactory = beanFactory;
        this.entityStream = entityStream;
        this.dispatchDataScopeDocumentRepository = dispatchDataScopeDocumentRepository;
        this.companyDocumentRepository = companyDocumentRepository;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public Collection<RdEmployeeDocument> translate(Collection<RdEmployeeEntity> employees,
                                                    boolean isFlushStation,
                                                    RdEmployeeDocumentRepository employeeDocumentRepository,
                                                    AttendanceRepository attendanceRepository,
                                                    RdCompositedReader compositedReader) {
        if (CollectionUtils.isEmpty(employees)) {
            return Collections.EMPTY_LIST;
        }

        String[] selectedTenantUserIds =
                employees.stream()
                        .filter(ii -> !BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(ii.getSecurityStationId()))
                        .map(ii -> ii.getTenantUserId())
                        .filter(ii -> StringUtils.hasLength(ii))
                        .distinct().toArray(ix -> new String[ix]);
        Collection<RdCommunityUserDocument> communityUserDocuments = new ArrayList<>();
        if (selectedTenantUserIds != null && selectedTenantUserIds.length > 0) {
            EntityStream entityStream =
                    beanFactory.getBean(EntityStream.class);
            SearchStream<RdCommunityUserDocument> searchStream =
                    entityStream.of(RdCommunityUserDocument.class);

            searchStream =
                    searchStream
                            .filter(RdCommunityUserDocument$.TENANT_USER_ID.in(selectedTenantUserIds));
            communityUserDocuments =
                    searchStream.collect(Collectors.toList());
        }

        /**
         * 内保单位需要可以调度该保安人员
         */
        Collection<RdDispatchDataScopeDocument> dataScopeDocuments = new ArrayList<>();
        Collection<RdCompanyDocument> proprietorCompanyDocuments = new ArrayList<>();
        Collection<String> stationIds =
                employees.stream().map(ii->ii.getSecurityStationId())
                        .filter(ii->StringUtils.hasLength(ii) && !StringUtil.isEqual(BusinessConstants.DefaultEmptyValue,ii))
                .distinct().collect(Collectors.toList());
        if(!CollectionUtils.isEmpty(stationIds)) {
            dataScopeDocuments =
                    beanFactory.getBean(RdDispatchDataScopeDocumentRepository.class)
                            .findAllById(stationIds);
            if(!CollectionUtils.isEmpty(dataScopeDocuments)) {
                Collection<String> comIds =
                        dataScopeDocuments.stream()
                                .filter(ix -> ix.getScopes() != null)
                                .flatMap(ix -> ix.getScopes().stream())
                                .collect(Collectors.toList());

                if(!CollectionUtils.isEmpty(comIds)) {
                    proprietorCompanyDocuments =
                            beanFactory.getBean(RdCompanyDocumentRepository.class)
                                    .findAllById(comIds);
                }
            }
        }

        Collection<RdCommunityUserDocument> finalCommunityUserDocuments = communityUserDocuments;
        Collection<RdEmployeeDocument> documents = new ArrayList<>();
        for (RdEmployeeEntity ii : employees) {
            RdEmployeeDocument document = RdEmployeeEntityUtils.toDocument(ii,
                    isFlushStation,
                    employeeDocumentRepository,
                    attendanceRepository, compositedReader,
                    beanFactory, finalCommunityUserDocuments,
                    dataScopeDocuments,
                    proprietorCompanyDocuments
            );

            if (document != null) {
                documents.add(document);
            }
        }

        return documents;
    }

    @Override
    public Collection<RdCompanyDocument> translate2Company(Collection<RdCompanyEntity> companies,
                                                           RdCompanyDocumentRepository companyDocumentRepository,
                                                           RdCompositedReader compositedReader) {
        if (CollectionUtils.isEmpty(companies)) {
            return Collections.EMPTY_LIST;
        }

        Collection<RdCompanyDocument> documents =
                companies.stream().map(ii -> {
                    return RdCompanyEntityUtils.toDocument(
                            ii, companyDocumentRepository,
                            compositedReader,
                            entityStream);
                }).collect(Collectors.toList());

        return documents;
    }

    @Override
    public Collection<RdSecurityStationDocument> translate2Stations(
            Collection<RdSecurityStationEntity> securityStations,
            RdSecurityStationDocumentRepository securityStationDocumentRepository,
            RdCompositedReader compositedReader,
            Collection<StationDeviceEntity> stationDevices) {
        if (CollectionUtils.isEmpty(securityStations)) {
            return Collections.EMPTY_LIST;
        }

        Collection<String> stationIds = securityStations.stream().map(ii -> ii.getId())
                .distinct().collect(Collectors.toList());
        Collection<RdDispatchDataScopeDocument> dataScopeDocuments
                = this.dispatchDataScopeDocumentRepository.findAllById(stationIds);

        Collection<RdCompanyDocument> proprietorCompanyDocuments = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dataScopeDocuments)) {
            Collection<String> comIds =
                    dataScopeDocuments.stream().filter(ix -> ix.getScopes() != null)
                            .flatMap(ix -> ix.getScopes().stream())
                            .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(comIds)) {
                proprietorCompanyDocuments = this.companyDocumentRepository.findAllById(comIds);
            }
        }

        Collection<RdCompanyDocument> finalProprietorCompanyDocuments = proprietorCompanyDocuments;
        return securityStations.stream().map(ii -> {
            return RdSecurityStationEntityUtil.toDocument(ii, false,
                    securityStationDocumentRepository, entityStream, dataScopeDocuments,
                    finalProprietorCompanyDocuments,stationDevices);
        }).filter(ii -> ii != null).collect(Collectors.toList());
    }

    @Override
    public RdDeviceDeskPermissionResponse translate(RdDeviceDeskPermissionDocument document) {
        if (document == null) {
            return null;
        }

        Collection<RdDeviceDeskPermissionResponse.RdDeviceDeskServiceScopeResponse>
                deviceDeskServiceScopeResponses = Collections.EMPTY_LIST;
        if (!CollectionUtils.isEmpty(document.getServiceScopes())) {
            deviceDeskServiceScopeResponses =
                    document.getServiceScopes().stream().map(ii ->
                            RdDeviceDeskPermissionResponse.RdDeviceDeskServiceScopeResponse.create(
                                    ii.getId(),
                                    ii.getName(),
                                    ii.getNumber(),
                                    ii.getBeginDate(),
                                    ii.getEndDate(),
                                    ii.getBoundEmployeeId(),
                                    ii.getAssignedSuperviseDepartId(),
                                    ii.getIpAddress(),
                                    ii.getOrgId(),
                                    ii.getDesc(),
                                    ii.getIr()
                            )).collect(Collectors.toList());
        }

        RdDeviceDeskPermissionResponse deskPermissionResponse =
                RdDeviceDeskPermissionResponse.create(deviceDeskServiceScopeResponses);

        return deskPermissionResponse;
    }

    @Override
    public OrgPurseValidateResponse translate2OrgPurses(OrgPursePointRequest request, Collection<OrgPurseDocument> orgPurses) {
        if (CollectionUtils.isEmpty(request.getOrganizationIds())) {
            return null;
        }

        Collection<OrgPurseValidateResponse.OrgPurseItemResponse> itemResponses =
                request.getOrganizationIds().stream().distinct()
                        .map(orgId -> {
                            Optional<OrgPurseDocument> selectedOrgPurseOptional
                                    = orgPurses.stream().filter(ii -> ii.getId().equalsIgnoreCase(orgId))
                                    .findFirst();
                            if (!selectedOrgPurseOptional.isPresent() || selectedOrgPurseOptional.get().getPoints() == null) {
                                return OrgPurseValidateResponse.OrgPurseItemResponse.create(orgId, false);
                            }

                            OrgPurseDocument orgPurse = selectedOrgPurseOptional.get();
                            return OrgPurseValidateResponse.OrgPurseItemResponse.create(orgId,
                                    orgPurse.isDispatchable()
                            );
                        }).collect(Collectors.toList());

        return OrgPurseValidateResponse.create(itemResponses);
    }

    @Override
    public PoliceIncidentOverviewResponse translate2PoliceIncidentsDetailReaderResponse(PoliceIncidentsEntity pie) {
        {
            Map<String, Object> items = new HashMap<>();
            if (StringUtils.hasLength(pie.getExtendJsonInfo())) {
                items = this.jsonProvider.toObject(Map.class, pie.getExtendJsonInfo());
            }

            PoliceIncidentOverviewResponse res =
                    PoliceIncidentOverviewResponse.create(
                            String.valueOf(pie.getId()),
                            pie.getCategory(),
                            pie.getTenantUserId(),
                            pie.getEmpName(),
                            pie.getTenantEmployeeId(),
                            pie.getEmpIdNumber(),
                            pie.getOrganizationId(),
                            pie.getOrganizationName(),
                            pie.getSecurityStationId(),
                            pie.getSecurityStationName(),
                            pie.getSuperviseDepartId(),
                            pie.getSuperviseDepartName(),
                            pie.getCode(),
                            pie.getName(),
                            pie.getEmpName(),
                            pie.getSceneUrl(),
                            pie.getContact(),
                            pie.getPoliceEventStatus(),
                            pie.getPoliceEventLevel(), pie.getAddress(),
                            pie.getReason(), pie.getPoliceEventType(),
                            pie.getLonLat().getLat(), pie.getLonLat().getLon(),
                            pie.getActionTaken(), pie.getActionTendency(),
                            pie.getProcessedSuggestion(),
                            pie.getProcessedLocaleGuidance(),
                            pie.getProcessedUserId(),
                            pie.getProcessedUserName(),
                            pie.getProcessedTime(),
                            pie.getCreatedTime(),
                            pie.getSecurityStationAddress(),
                            pie.getDescription(),
                            pie.getLocation(),
                            pie.getProjectId(),
                            pie.getProjectName(),
                            items,
                            pie.getPoints()
                    );

            return res;
        }
    }

    @Override
    public CreateRoomWriterRpcResponse translate(
            CreateRoomCommandResponse commandResponse,
            CreateRoomWriterRpcRequest request,
            Collection<OrgPurseEntity> orgPurses) {
        Collection<CreateRoomWriterRpcResponse.RoomUserResponse> roomUsers = getRoomUserResponses(commandResponse, request, orgPurses);

        return CreateRoomWriterRpcResponse.create(commandResponse == null ? -1 : commandResponse.getRoomNo(),
                roomUsers);
    }

    private Collection<CreateRoomWriterRpcResponse.RoomUserResponse> getRoomUserResponses(CreateRoomCommandResponse commandResponse, CreateRoomWriterRpcRequest request, Collection<OrgPurseEntity> orgPurses) {
        Collection<CreateRoomWriterRpcResponse.RoomUserResponse> roomUsers =
                request.getRoomUsers().stream().map(ix ->
                        {
                            long points = 0l;
                            boolean dispatchable = false;
                            if (ix.isSponsor()) {
                                points = 0l;
                                dispatchable = true;
                            } else {
                                Optional<OrgPurseEntity> orgPurseOptional =
                                        orgPurses.stream()
                                                .filter(ii -> ii.getOrganizationId()
                                                        .equalsIgnoreCase(ix.getOrganizationId()))
                                                .findFirst();
                                if (orgPurseOptional.isPresent()) {
                                    points = orgPurseOptional.get().getPoints();
                                    dispatchable = orgPurseOptional.get().IsDispatchable();
                                }
                            }
                            CreateRoomWriterRpcResponse.RoomUserResponse roomUser =
                                    CreateRoomWriterRpcResponse.RoomUserResponse.create(
                                            ix.getEmployeeId(),
                                            ix.getOrganizationId(),
                                            ix.getTencentUserId(),
                                            ix.getSecurityStationId(),
                                            commandResponse == null ? "invalid" : commandResponse.getCid(ix.getEmployeeId()),
                                            ix.getSuperviseDepartId(),
                                            ix.isSponsor(),
                                            points, dispatchable);

                            return roomUser;
                        }
                ).collect(Collectors.toList());
                
        return roomUsers;
    }

    private Collection<InviteRoomUsersWriterRpcResponse.InviteRoomUserResponse> getInviteRoomUserResponses(InviteRoomUserCommandResponse commandResponse, InviteRoomUsersWriterRpcRequest request, Collection<OrgPurseEntity> orgPurses) {
        Collection<InviteRoomUsersWriterRpcResponse.InviteRoomUserResponse> roomUsers =
                request.getRoomUsers().stream().map(ix ->
                        {
                            long points = 0l;
                            boolean dispatchable = false;
                            Optional<OrgPurseEntity> orgPurseOptional =
                                    orgPurses.stream()
                                            .filter(ii -> ii.getOrganizationId()
                                                    .equalsIgnoreCase(ix.getOrganizationId()))
                                            .findFirst();
                            if (orgPurseOptional.isPresent()) {
                                points = orgPurseOptional.get().getPoints();
                                dispatchable = orgPurseOptional.get().IsDispatchable();
                            }
                            InviteRoomUsersWriterRpcResponse.InviteRoomUserResponse roomUser =
                                    InviteRoomUsersWriterRpcResponse.InviteRoomUserResponse.create(
                                            ix.getEmployeeId(),
                                            commandResponse == null ? null : commandResponse.getEmployeeName(ix.getEmployeeId()),
                                            ix.getOrganizationId(),
                                            ix.getTencentUserId(),
                                            ix.getSecurityStationId(),
                                            commandResponse == null ? "invalid" : commandResponse.getCid(ix.getEmployeeId()),
                                            ix.getSuperviseDepartId(),
                                            points, dispatchable);

                            return roomUser;
                        }
                ).collect(Collectors.toList());

        return roomUsers;
    }

    @Override
    public InviteRoomUsersWriterRpcResponse translate(InviteRoomUserCommandResponse commandResponse, InviteRoomUsersWriterRpcRequest request, Collection<OrgPurseEntity> orgPurses) {
        Collection<InviteRoomUsersWriterRpcResponse.InviteRoomUserResponse> roomUsers = getInviteRoomUserResponses(commandResponse, request, orgPurses);

        return InviteRoomUsersWriterRpcResponse.create(commandResponse == null ? -1 : commandResponse.getRoomNo(),
                roomUsers);
    }

    @Override
    public Collection<RdCommunityUserDocument> translate(Collection<RdCommunityUserEntity> communityUsers) {
        if (CollectionUtils.isEmpty(communityUsers)) {
            return Collections.EMPTY_LIST;
        }

        return communityUsers.stream().map(ii ->
                RdCommunityUserDocument.create(ii.getId(), ii.getTenantUserId(), ii.getSuperviseDepartId(), ii.getSuperviseDepartName())
        ).collect(Collectors.toList());
    }

    @Override
    public Collection<RdExamSiteDocument> translate2ExamSites(Collection<RdExamSiteEntity> examSites) {
        if (CollectionUtils.isEmpty(examSites)) {
            return Collections.EMPTY_LIST;
        }

        return examSites.stream().map(ii -> {
            RdExamSiteDocument document = new RdExamSiteDocument();
            document.setId(ii.getId());
            document.setName(ii.getName());

            document.setOrganizationId(ii.getOrganizationId());
            document.setCompanyName(ii.getCompanyName());
            document.setAddress(ii.getAddress());
            document.setCountOfRoom(ii.getCountOfRoom());
            document.setLonLat(GeoPointUtils.translate(ii.getLonLat()));
            document.setCapacityOfPerson(ii.getCapacityOfPerson());
            document.setStatus(ii.isStatus());
            document.setSuperviseDepartName(ii.getSuperviseDepartName());
            document.setSuperviseDepartId(ii.getSuperviseDepartId());

            document.setLastSyncTime(ii.getLastSyncTime());
            document.setRegion(ii.getRegionName());

            return document;
        }).collect(Collectors.toList());
    }

    @Override
    public Collection<RdExamSiteRoomDocument> translate2ExamSiteRooms(Collection<RdExamSiteRoomEntity> examSiteRooms) {
        if (CollectionUtils.isEmpty(examSiteRooms)) {
            return Collections.EMPTY_LIST;
        }

        return examSiteRooms.stream().map(ii -> {
            RdExamSiteRoomDocument document = new RdExamSiteRoomDocument();
            document.setId(ii.getId());
            document.setName(ii.getName());
            document.setLonLat(GeoPointUtils.translate(ii.getLonLat()));
            document.setExamSiteId(ii.getExamSiteId());
            document.setExamSiteName(ii.getExamSiteName());

            document.setCapacityOfPerson(ii.getCapacityOfPerson());
            document.setScopeRange(ii.getScopeRange());
            document.setStatus(ii.isStatus());

            document.setSuperviseDepartId(ii.getSuperviseDepartId());
            document.setSuperviseDepartName(ii.getSuperviseDepartName());

            document.setLastSyncTime(ii.getLastSyncTime());
            document.setOrganizationId(ii.getOrganizationId());
            document.setCompanyName(ii.getCompanyName());

            return document;
        }).collect(Collectors.toList());
    }

    @Override
    public Collection<RdSystemExamInfoDocument> translate2SystemExamInfo(Collection<RdSystemExamInfoEntity> systemExamInfos) {
        if (CollectionUtils.isEmpty(systemExamInfos)) {
            return Collections.EMPTY_LIST;
        }


        return systemExamInfos.stream().map(ii -> {
            RdSystemExamInfoDocument document = new RdSystemExamInfoDocument();
            document.setId(ii.getId());
            document.setBeginTime(LocalDateTime.ofInstant(ii.getBeginTime().toInstant(), ZoneId.systemDefault()));
            document.setEndTime(LocalDateTime.ofInstant(ii.getEndTime().toInstant(), ZoneId.systemDefault()));
            document.setBeginTimeOfTs(ii.getBeginTime().getTime());
            document.setEndTimeOfTs(ii.getEndTime().getTime());

            document.setExamType(ii.getExamType());
            document.setExamSiteRoomId(ii.getExamSiteRoomId());
            document.setTeacherIds(ii.getTeacherIds());
            document.setName(ii.getName());
            document.setLastSyncTime(ii.getLastSyncTime());

            return document;
        }).collect(Collectors.toList());
    }

    @Override
    public Collection<RdProjectDocument> translate(Collection<RdProjectEntity> projectEntities, RdProjectDocumentRepository projectDocumentRepository) {
        if (CollectionUtils.isEmpty(projectEntities)) {
            return Collections.EMPTY_LIST;
        }

        String[] empIds =
                projectEntities.stream().flatMap(ii->BusinessUtil.getSplitValues(ii.getPrincipalId()).stream())
                .distinct().toArray(ii->new String[ii]);
        Collection<RdEmployeeDocument> employeeDocuments = new ArrayList<>();
        if(empIds.length>0) {
            employeeDocuments = this.entityStream.of(RdEmployeeDocument.class)
                    .filter(RdEmployeeDocument$.ID.in(empIds)).collect(Collectors.toSet());
        }

        Collection<RdEmployeeDocument> finalEmployeeDocuments = employeeDocuments;
        Collection<RdProjectDocument> documents =
                projectEntities.stream().map(ii -> {
                    RdProjectDocument document = RdProjectDocument.create(ii.getId(), ii.getName());
                    document.setPrincipalId(ii.getPrincipalId());
                    String principalNames = BusinessConstants.DefaultEmptyValue;
                    String principalTelephones = BusinessConstants.DefaultEmptyValue;
                    Collection<String> principalIds = BusinessUtil.getSplitValues(ii.getPrincipalId());

                    if(!CollectionUtils.isEmpty(principalIds)) {
                        Collection<RdEmployeeDocument> selectedEmployments =
                                finalEmployeeDocuments.stream().filter(ix -> principalIds.contains(ix.getId()))
                                        .collect(Collectors.toList());

                        principalNames = selectedEmployments.stream().map(ix -> ix.getName()).collect(Collectors.joining(","));
                        principalTelephones = selectedEmployments.stream().map(ix -> ix.getContact()).collect(Collectors.joining(","));
                    }

                    document.setPrincipalTelephone(principalNames);
                    document.setPrincipalName(principalTelephones);


                    SimpleDateFormat dateFormat = new SimpleDateFormat();
                    document.setEndDate(null);
                    if(ii.getEndDate()!=null)
                    {
                        document.setEndDate(dateFormat.format(ii.getEndDate()));
                    }

                    document.setBeginDate(null);
                    if(ii.getBeginDate()!=null)
                    {
                        document.setBeginDate(dateFormat.format(ii.getBeginDate()));
                    }

                    document.setOrganizationId(ii.getOrganizationId());
                    document.setCompanyName(ii.getCompanyName());
                    document.setState(ii.getState());
                    document.setCountOfMembers(ii.getCountOfMembers());
                    document.setCountOfStations(ii.getCountOfStations());
                    document.setLonLat(GeoPointUtils.translate(ii.getLonLat()));

                    document.addScopePermissions(ii.getId());
                    document.addScopePermissions(ii.getOrganizationId());
                    document.addScopePermissions(ii.getSuperviseDepartId());
                    document.addScopePermissions(ii.getSuperviseDomainId());

                    document.updateWord(ii.getName(),ii.getAddress());
                    return document;
                }).collect(Collectors.toList());

        return documents;
    }
    
    @Override
    public ProjectResponse translate(RdProjectEntity ii, EntityStream entityStream,boolean forDetail) {
        if (ii == null) {
            return null;
        }

        String principalName = "";
        if(!BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(ii.getPrincipalId())) {
            String[] principalIds = ii.getPrincipalId().split(";");
            Collection<RdEmployeeDocument> employeeDocuments =
                    this.entityStream.of(RdEmployeeDocument.class)
                            .filter(RdEmployeeDocument$.ID.in(principalIds)).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(employeeDocuments)) {
                principalName = employeeDocuments.stream()
                        .map(ix -> ix.getName())
                        .collect(Collectors.joining(";"));
            }
        }

        long countOfStations = ii.getCountOfStations();
        if(forDetail) {
            countOfStations =
                    entityStream.of(RdSecurityStationDocument.class)
                            .filter(RdSecurityStationDocument$.SCOPE_PERMISSIONS.in(new String[]{
                                    ii.getId()
                            })).count();
        }

        return ProjectResponse.create(
                ii.getPkId(), ii.getId(), ii.getName(),ii.getType(), principalName,
                ii.getOrganizationId(),
                ii.getCompanyName(), ii.getAddress(),
                ii.getCountOfMembers(), ii.getCountOfTeams(), ii.getBeginDate(), ii.getEndDate(), countOfStations);
    }

    @Override
    public Collection<ProjectTeamResponse> translateProjectTeams(Collection<RdProjectTeamGroupEntity> teamGroups) {
        if (CollectionUtils.isEmpty(teamGroups)) {
            return Collections.EMPTY_LIST;
        }

        var rootTeams = teamGroups.stream()
                .filter(ii -> !StringUtils.hasLength(ii.getParentId()))
                .collect(Collectors.toList());

        Collection<ProjectTeamResponse> data = new HashSet<>();
        for (var item : rootTeams) {
            var team = ProjectTeamResponse.create(item.getId(), item.getName(), item.getOrganizationId(), item.getProjectId());
            generateProjectTeams(team, teamGroups);

            data.add(team);
        }

        return data;
    }

     @Override
    public Collection<RdDeviceDocument> translate2DeviceDocuments(Collection<RdDeviceEntity> devices
                                                                  ) {
        if (CollectionUtils.isEmpty(devices)) {
            return Collections.EMPTY_LIST;
        }

        String[] deviceIds =
                devices.stream().map(ii -> ii.getId()).collect(Collectors.toSet())
                        .toArray(ix -> new String[ix]);

        Collection<RdDeviceDocument> existsDocuments =
                this.entityStream.of(RdDeviceDocument.class).filter(RdDeviceDocument$.ID.in(deviceIds))
                        .collect(Collectors.toList());

        Collection<String> stationIds = devices.stream().map(ii -> ii.getSecurityStationId())
                .distinct().collect(Collectors.toList());
        Collection<RdDispatchDataScopeDocument> dataScopeDocuments
                = this.dispatchDataScopeDocumentRepository.findAllById(stationIds);

        final Collection<RdCompanyDocument> proprietorCompanyDocuments = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dataScopeDocuments)) {
            Collection<String> comIds =
                    dataScopeDocuments.stream().filter(ix -> ix.getScopes() != null)
                            .flatMap(ix -> ix.getScopes().stream())
                            .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(comIds)) {
                proprietorCompanyDocuments.addAll(this.companyDocumentRepository.findAllById(comIds));
            }
        }

        Collection<RdSecurityStationDocument> stationDocuments =  new ArrayList<>();
        if(!CollectionUtils.isEmpty(stationIds)) {
            stationDocuments =
                    this.entityStream.of(RdSecurityStationDocument.class)
                            .filter(RdSecurityStationDocument$.ID.in(stationIds.toArray(ix -> new String[ix])))
                            .collect(Collectors.toList());
        }

         Collection<RdSecurityStationDocument> finalStationDocuments = stationDocuments;
         return devices.stream().map(ii -> {
            RdDeviceDocument document = existsDocuments.stream().filter(ix -> ix.getId().equalsIgnoreCase(ii.getId()))
                    .findFirst().orElse(null);
            if (document == null) {
                document = RdDeviceDocument.create(ii.getId());
            }

            document.rest();
            document.updateWord(ii.getDispatchNo(), ii.getDeviceNo());
            document.setDispatchNo(ii.getDispatchNo());
            document.setDeviceNo(ii.getDeviceNo());
            document.setDeviceModel(ii.getDeviceModel());
            document.setOrganizationId(ii.getOrganizationId());
            document.setOrganizationName(ii.getOrganizationName());
            document.setSecurityStationId(ii.getSecurityStationId());
            document.setSecurityStationName(ii.getSecurityStationName());
            document.setDeviceType(ii.getDeviceType());
            document.changeStatus(ii.getDeviceStatus());
            document.setDeviceUid(ii.getDeviceUid());

            if(document.getLonLat()==null || document.getLonLat().getX()==BusinessConstants.INVALID_LON_LAT) {
                if (StringUtils.hasLength(ii.getLon())) {
                    document.setLonLat(new Point(Double.parseDouble(ii.getLon()), Double.parseDouble(ii.getLat())));
                }
            }

            document.setLastSyncTime(ii.getLastSyncTime());

            document.addScopePermissions(ii.getSecurityDepartId(), ii.getOrganizationId(), ii.getSecurityStationId());

             /**
              * 添加项目Id
              */
            if(StringUtils.hasLength(ii.getSecurityStationId())) {
                Optional<RdSecurityStationDocument> stationDocumentOptional =
                        finalStationDocuments.stream().filter(ix -> ix.getId().equalsIgnoreCase(ii.getSecurityStationId()))
                                .findFirst();
                if (stationDocumentOptional.isPresent()) {
                    document.addScopePermissions(stationDocumentOptional.get().getProjectId());
                }
            }

            /**
             * 添加内保单位信息; 保证可以被内保单位进行查看
             */
            if (!CollectionUtils.isEmpty(dataScopeDocuments)) {
                Collection<String> scopeIds =
                        dataScopeDocuments.stream()
                                .filter(it -> StringUtil.isEqual(it.getId(), ii.getId()) && !CollectionUtils.isEmpty(it.getScopes()))
                                .flatMap(it -> it.getScopes().stream())
                                .distinct()
                                .collect(Collectors.toList());

                if (!CollectionUtils.isEmpty(scopeIds)) {
                    document.addScopePermissions(scopeIds.toArray(ix -> new String[ix]));

                    /**
                     * 所属的内保单位; 正常是一家
                     * 内保暂时不需要查询保安公司数据
                     */
                    if (!CollectionUtils.isEmpty(proprietorCompanyDocuments)) {
                        Collection<RdCompanyDocument> perCompanies
                                = proprietorCompanyDocuments.stream().filter(it -> scopeIds.contains(it.getId()))
                                .collect(Collectors.toList());
                        if (!CollectionUtils.isEmpty(perCompanies)) {
                            document.addScopePermissions(
                                    perCompanies.stream().map(ix -> ix.getSuperviseDepartId())
                                            .filter(ix -> !StringUtil.isEmpty(ix))
                                            .collect(Collectors.toList())
                                            .toArray(ix -> new String[ix])
                            );
                            RdCompanyDocument firstCom = perCompanies.stream().findFirst().get();
                            document.assignProprietor(firstCom.getId(), firstCom.getName());
                            document.addResourceTypes(ResourceType.ProprietorManaged.name());
                        }
                    }
                }
            }

            return document;

        }).collect(Collectors.toSet());
    }


    private void generateProjectTeams(ProjectTeamResponse parent,Collection<RdProjectTeamGroupEntity> entities) {
        Collection<RdProjectTeamGroupEntity> children = entities.stream().filter(ii->parent.getId().equalsIgnoreCase(ii.getParentId()))
                .collect(Collectors.toList());

        for (var item : children) {
            var child =
                    ProjectTeamResponse.create(item.getId(), item.getName(), item.getOrganizationId(), item.getProjectId());

            parent.AddChild(child);
            generateProjectTeams(child, entities);
        }
    }
}
