package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdEmployeeDocument;
import com.bcxin.tenant.open.document.domains.documents.RdEmployeeDocument$;
import com.bcxin.tenant.open.document.domains.repositories.RdEmployeeDocumentRepository;
import com.bcxin.tenant.open.domains.entities.UserPurseEntity;
import com.bcxin.tenant.open.domains.repositories.UserPurseRepository;
import com.bcxin.tenant.open.domains.services.RdEmployeeService;
import com.bcxin.tenant.open.domains.services.UserPurseService;
import com.bcxin.tenant.open.domains.services.commands.CreateUserPurseTransactionCommand;
import com.bcxin.tenant.open.domains.services.commands.UpdateLatLonCommand;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.enums.DutySignInType;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.jdks.UserWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.DonatePointRequest;
import com.bcxin.tenant.open.jdks.requests.SyncUpdateGeoRequest;
import com.bcxin.tenant.open.jdks.requests.UpdateGeoRequest;
import com.bcxin.tenant.open.jdks.responses.UserPurseResponse;
import com.redis.om.spring.search.stream.EntityStream;
import com.redis.om.spring.search.stream.SearchStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.geo.Point;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 写入当前用户的信息
 */
@DubboService
public class UserWriterRpcProviderImpl implements UserWriterRpcProvider {
    private static Logger logger = LoggerFactory.getLogger(UserWriterRpcProviderImpl.class);

    private final RdEmployeeService rdEmployeeService;
    private final EntityStream entityStream;
    private final RdEmployeeDocumentRepository employeeDocumentRepository;

    private final UserPurseRepository userPurseRepository;
    private final UserPurseService userPurseService;

    public UserWriterRpcProviderImpl(RdEmployeeService rdEmployeeService,
                                     EntityStream entityStream,
                                     RdEmployeeDocumentRepository employeeDocumentRepository, UserPurseRepository userPurseRepository, UserPurseService userPurseService) {
        this.rdEmployeeService = rdEmployeeService;
        this.entityStream = entityStream;
        this.employeeDocumentRepository = employeeDocumentRepository;
        this.userPurseRepository = userPurseRepository;
        this.userPurseService = userPurseService;
    }

    /**
     * 更新坐标的时候; 先更新redis -> 定期更新人的坐标（以及智能人事）
     * @param request
     */
    @Override
    public void updateLonLat(UpdateGeoRequest request) {
        TenantEmployeeContext.TenantUserModel employeeModel =
                TenantContext.getInstance().getUserContext().get();
        UpdateLatLonCommand command =
                UpdateLatLonCommand.create(employeeModel.getEmployeeId(), request.getLatitude(), request.getLongitude());
        this.rdEmployeeService.dispatch(command);
    }

    @Override
    public void switchMarkSwitch(DutySignInType dutySignInType) {

    }

    /**
     * 更新保安人员的最新坐标及产生电子围栏的预警记录
     * @param request
     */
    @Override
    public void batchUpdateLonLat(Collection<SyncUpdateGeoRequest> request) {
        if (CollectionUtils.isEmpty(request)) {
            throw new NoFoundTenantException();
        }

        Collection<String> employeeIds
                = request.stream()
                .filter(ii->ii.getEmployeeIds()!=null)
                .flatMap(ix -> ix.getEmployeeIds().stream())
                .distinct().collect(Collectors.toList());

        Collection<RdEmployeeDocument> documents = getAndUpdateEmployeeLonLatDocuments(request);

        Collection<String> stationIds =
                documents.stream()
                        .map(ix -> ix.getSecurityStationId())
                        .collect(Collectors.toList());


    }

    @Override
    public UserPurseResponse getById(String id) {
        UserPurseEntity userPurse = this.userPurseRepository.getById(id);
        if (userPurse == null) {
            return UserPurseResponse.create(0);
        }

        return UserPurseResponse.create(userPurse.getPoints());
    }

    @Override
    public void donatePoints(DonatePointRequest request) {
        this.userPurseService.dispatch(CreateUserPurseTransactionCommand.CreateUserPurseTransactionsCommand.create(
                Stream.of(
                        CreateUserPurseTransactionCommand.create(
                                request.getReferenceType(),
                                request.getReferenceNumber(),
                                request.getTenantUserId(),
                                request.getNote(),
                                request.getPoints()
                        )
                ).collect(Collectors.toList())
        ));
    }

    private Collection<RdEmployeeDocument> getAndUpdateEmployeeLonLatDocuments(Collection<SyncUpdateGeoRequest> request) {
        if (CollectionUtils.isEmpty(request)) {
            return Collections.EMPTY_LIST;
        }

        String[] employeeIds = request.stream()
                .filter(ii->ii.getEmployeeIds()!=null)
                .map(ix -> ix.getEmployeeIds()).toArray(size -> new String[size]);
        SearchStream<RdEmployeeDocument> employeeDocumentSearchStream =
                this.entityStream.of(RdEmployeeDocument.class);
        employeeDocumentSearchStream =
                employeeDocumentSearchStream.filter(RdEmployeeDocument$.ID.in(employeeIds));

        Collection<RdEmployeeDocument> documents = employeeDocumentSearchStream.collect(Collectors.toList());

        if (CollectionUtils.isEmpty(documents)) {
            return Collections.EMPTY_LIST;
        }

        try {
            Collection<RdEmployeeDocument> changedDocuments = new ArrayList<>();
            request.forEach(rt -> {
                Collection<RdEmployeeDocument> selectedEmployeeDocuments
                        = documents.stream().filter(ix ->
                        rt.getEmployeeIds().stream().anyMatch(ik->ik.equalsIgnoreCase(ix.getTenantEmployeeId()))
                        )
                        .collect(Collectors.toList());

                for (RdEmployeeDocument dt : selectedEmployeeDocuments) {
                    Point point = new Point(rt.getLongitude(), rt.getLatitude());
                    dt.changeLonLat(point);

                    changedDocuments.add(dt);
                }
            });

            if (!changedDocuments.isEmpty()) {
                this.employeeDocumentRepository.saveAll(changedDocuments);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            logger.error("更新最新保安人员信息发送异常", ex);
        }

        return documents;
    }
}
