package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.entities.TenantAccountEntity;
import com.bcxin.tenant.open.domains.repositories.TenantAccountRepository;
import com.bcxin.tenant.open.domains.services.TenantAccountService;
import com.bcxin.tenant.open.domains.services.commands.CreateLogoSettingCommand;
import com.bcxin.tenant.open.domains.services.commands.CreateReadOperationInstructionCommand;
import com.bcxin.tenant.open.domains.services.commands.CreateTenantAccountCommand;
import com.bcxin.tenant.open.domains.components.CacheProvider;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.enums.DispatchAccountType;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.TenantAccountWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.LogoSettingRequest;
import com.bcxin.tenant.open.jdks.requests.SosSettingRequest;
import com.bcxin.tenant.open.jdks.responses.TenantAccountSettingResponse;
import org.apache.dubbo.config.annotation.DubboService;


@DubboService
public class TenantAccountWriterRpcProviderImpl implements TenantAccountWriterRpcProvider {
    private static final String TENANT_ACCOUNT_SETTING_FORMAT = "dispatch:tenant_account_setting:%s";

    private final TenantAccountRepository tenantAccountRepository;
    private final TenantAccountService accountService;
    private final CacheProvider cacheProvider;

    public TenantAccountWriterRpcProviderImpl(TenantAccountRepository tenantAccountRepository,
                                              TenantAccountService accountService,
                                              CacheProvider cacheProvider) {
        this.tenantAccountRepository = tenantAccountRepository;
        this.accountService = accountService;
        this.cacheProvider = cacheProvider;
    }

    @Override
    public void doSosSettingPatch(SosSettingRequest request) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        this.cacheProvider.delete(
                getTenantAccountCacheKey(userModel.getAccountType(), userModel.getTencentUserId())
        );

        this.accountService.dispatch(
                CreateTenantAccountCommand.create(
                        userModel.getAccountType(),
                        request.getSosEnabled(),
                        request.getSosKey(),
                        userModel.getTencentUserId(),
                        userModel.getEmployeeId(),
                        userModel.getName()
                )
        );
    }

    @Override
    public void doLogoSettingPatch(LogoSettingRequest request) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权有效调度台用户");
        }

        this.cacheProvider.delete(getTenantAccountCacheKey(userModel.getAccountType(), userModel.getTencentUserId()));
        this.accountService.dispatch(
                CreateLogoSettingCommand.create(
                        request.getLogo(), request.getTitle(),
                        userModel.getTencentUserId(),
                        userModel.getAccountType(),
                        userModel.getEmployeeId(), userModel.getName(),
                        request.getAlarmBell()
                        )
        );
    }

    @Override
    public void doReadOperationInstruction() {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        this.cacheProvider.delete(
                getTenantAccountCacheKey(userModel.getAccountType(), userModel.getTencentUserId())
        );

        this.accountService.dispatch(CreateReadOperationInstructionCommand.create(
                userModel.getTencentUserId(),
                userModel.getEmployeeId(), userModel.getName(), userModel.getAccountType())
        );
    }

    @Override
    public TenantAccountSettingResponse getById(DispatchAccountType accountType, String deviceNo) {
        TenantAccountSettingResponse data =
                this.cacheProvider.get(getTenantAccountCacheKey(accountType, deviceNo), () -> {
                    TenantAccountEntity accountEntity =
                            tenantAccountRepository.getByAccount(accountType, deviceNo);
                    if (accountEntity == null) {
                        return null;
                    }

                    return TenantAccountSettingResponse.create(accountEntity.getSosEnabled(),
                            accountEntity.getSosKey(), accountEntity.getLogo(),
                            accountEntity.getTitle(),
                            accountEntity.getReadInstructionTime(),
                            accountEntity.getAlarmBell());
                }, TenantAccountSettingResponse.class);

        if (data == null) {
            data = TenantAccountSettingResponse.create(false, null, null,
                    null, null, null);
        }

        return data;
    }

    private static String getTenantAccountCacheKey(DispatchAccountType accountType, String employeeId) {
        return String.format(TENANT_ACCOUNT_SETTING_FORMAT, employeeId);
    }
}
