package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdSystemExamInfoDocument;
import com.bcxin.tenant.open.document.domains.repositories.RdSystemExamInfoDocumentRepository;
import com.bcxin.tenant.open.domains.entities.RdSystemExamInfoEntity;
import com.bcxin.tenant.open.domains.repositories.RdSystemExamInfoRepository;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.utils.DateUtils;
import com.bcxin.tenant.open.jdks.SystemExamInfoRpcProvider;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.function.Function;
import java.util.stream.Collectors;

@DubboService
public class SystemExamInfoRpcProviderImpl implements SystemExamInfoRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(SystemExamInfoRpcProviderImpl.class);
    private static final int PAGE_SIZE = 500;
    private final RdSystemExamInfoRepository rdSystemExamInfoRepository;
    private final RdSystemExamInfoDocumentRepository rdSystemExamInfoDocumentRepository;
    private final DataTranslate dataTranslate;

    public SystemExamInfoRpcProviderImpl(RdSystemExamInfoRepository rdSystemExamInfoRepository,
                                         RdSystemExamInfoDocumentRepository rdSystemExamInfoDocumentRepository,
                                         DataTranslate dataTranslate) {
        this.rdSystemExamInfoRepository = rdSystemExamInfoRepository;
        this.rdSystemExamInfoDocumentRepository = rdSystemExamInfoDocumentRepository;
        this.dataTranslate = dataTranslate;
    }


    @Override
    public int flush2Redis(Collection<String> ids) {

        return this.doExecuteFlush(pageIndex -> {
            return this.rdSystemExamInfoRepository.getByPage(ids, pageIndex, PAGE_SIZE);
        }, ids);
    }

    private int doExecuteFlush(Function<Integer, Collection<RdSystemExamInfoEntity>> extractExamInfos, Collection<String> ids) {
        int pageIndex = 1;
        int totalCount = 0;

        Collection<RdSystemExamInfoEntity> rdSystemExamInfoEntities = extractExamInfos.apply(pageIndex);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }

        while (!CollectionUtils.isEmpty(rdSystemExamInfoEntities)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = rdSystemExamInfoEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<RdSystemExamInfoDocument> documents =
                    this.dataTranslate.translate2SystemExamInfo(rdSystemExamInfoEntities);

            if (CollectionUtils.isEmpty(documents)) {
                return totalCount;
            }

            Collection<RdSystemExamInfoDocument> updatedDocuments
                    = documents.stream().collect(Collectors.toList());

            this.rdSystemExamInfoDocumentRepository.saveAll(updatedDocuments);

            totalCount += documents.size();
            pageIndex++;
            if (documents.size() < PAGE_SIZE) {
                break;
            }

            rdSystemExamInfoEntities = extractExamInfos.apply(pageIndex);

        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.rdSystemExamInfoDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }
}
