package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.*;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RoomDocumentRepository;
import com.bcxin.tenant.open.domains.BillPaymentRuleConfig;
import com.bcxin.tenant.open.domains.entities.OrgPurseEntity;
import com.bcxin.tenant.open.domains.repositories.OrgPurseRepository;
import com.bcxin.tenant.open.domains.repositories.RoomRepository;
import com.bcxin.tenant.open.domains.services.RoomService;
import com.bcxin.tenant.open.domains.services.commands.*;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.enums.DispatchReasonType;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DispatchReasonType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.RoomRpcProvider;
import com.bcxin.tenant.open.jdks.requests.CreateRoomWriterRpcRequest;
import com.bcxin.tenant.open.jdks.requests.InviteRoomUsersWriterRpcRequest;
import com.bcxin.tenant.open.jdks.requests.RoomReplyRequest;
import com.bcxin.tenant.open.jdks.requests.TRTCActionWriterRpcRequest;
import com.bcxin.tenant.open.jdks.responses.CreateRoomWriterRpcResponse;
import com.bcxin.tenant.open.jdks.responses.InviteRoomUsersWriterRpcResponse;
import com.bcxin.tenant.open.jdks.responses.RoomResponse;
import com.redis.om.spring.search.stream.EntityStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@DubboService
public class RoomRpcProviderImpl implements RoomRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(RoomRpcProviderImpl.class);

    private final RoomService roomService;
    private final OrgPurseRepository orgPurseRepository;
    private final DataTranslate dataTranslate;
    private final RoomDocumentRepository roomDocumentRepository;
    private final RdCompanyDocumentRepository rdCompanyDocumentRepository;
    private final RoomRepository roomRepository;

    private final EntityStream entityStream;
    private final KafkaTemplate kafkaTemplate;
    private final JsonProvider jsonProvider;
    private final BillPaymentRuleConfig billPaymentRuleConfig;


    public RoomRpcProviderImpl(RoomService roomService,
                               OrgPurseRepository orgPurseRepository,
                               DataTranslate dataTranslate,
                               RoomDocumentRepository roomDocumentRepository,
                               RdCompanyDocumentRepository rdCompanyDocumentRepository,
                               RoomRepository roomRepository,
                               EntityStream entityStream,
                               KafkaTemplate kafkaTemplate,
                               JsonProvider jsonProvider,
                               BillPaymentRuleConfig billPaymentRuleConfig) {
        this.roomService = roomService;
        this.orgPurseRepository = orgPurseRepository;
        this.dataTranslate = dataTranslate;
        this.roomDocumentRepository = roomDocumentRepository;
        this.rdCompanyDocumentRepository = rdCompanyDocumentRepository;
        this.roomRepository = roomRepository;
        this.entityStream = entityStream;
        this.kafkaTemplate = kafkaTemplate;
        this.jsonProvider = jsonProvider;
        this.billPaymentRuleConfig = billPaymentRuleConfig;
    }

    @Override
    public CreateRoomWriterRpcResponse create(CreateRoomWriterRpcRequest request) {
        TenantEmployeeContext.TenantUserModel tenantUserModel = TenantContext.getInstance().getUserContext().get();
        if (tenantUserModel == null) {
            throw new UnAuthorizedTenantException();
        }

        if (request.getDeskType() == null) {
            throw new BadTenantException("参数有误, 调度台类型不能为空");
        }

        if (CollectionUtils.isEmpty(request.getRoomUsers())) {
            throw new BadTenantException("参数有误, 参与会议的人员不能为空");
        }
        if (request.getRoomUsers().stream().filter(ii -> ii.isSponsor()).count() != 1) {
            throw new BadTenantException("参数有误, 发起会议的人只能是一个人");
        }

        Collection<String> organizationIds =
                request.getRoomUsers().stream().map(ix -> ix.getOrganizationId())
                        .collect(Collectors.toList());

        boolean isPaymentOwner = billPaymentRuleConfig.isPaymentOwner(tenantUserModel.getOrgInstitutional(), tenantUserModel.getSuperviseRegionCode());
        if (isPaymentOwner) {
            organizationIds.add(tenantUserModel.getOrganizationId());
        }


        Collection<OrgPurseEntity> orgPurses = this.orgPurseRepository.getByOrganizationIds(organizationIds);
        if (isPaymentOwner) {
            if (!request.getRoomUsers().stream().filter(ix -> ix.isSponsor())
                    .allMatch(ix ->
                            orgPurses.stream().anyMatch(ir -> ir.getOrganizationId()
                                    .equalsIgnoreCase(ix.getOrganizationId()) && ir.IsDispatchable())
                    )) {
                throw new BadTenantException("您当前企业的联动值余额不足, 请联系管理员进行充值");
            }
        }

        CreateRoomCommandResponse commandResponse = null;
        if (billPaymentRuleConfig.ignoreCheckUserPoints(isPaymentOwner) ||
                request.getRoomUsers().stream().filter(ix -> !ix.isSponsor())
                        .allMatch(ix ->
                                orgPurses.stream().anyMatch(ir -> ir.getOrganizationId()
                                        .equalsIgnoreCase(ix.getOrganizationId()) && ir.IsDispatchable())
                        )
        ) {
            Collection<CreateRoomCommand.CreateRoomUserCommand> roomUserCommands =
                    request.getRoomUsers().stream().map(ii -> CreateRoomCommand.CreateRoomUserCommand.create(
                            ii.getEmployeeId(),
                            ii.getOrganizationId(),
                            ii.getTencentUserId(),
                            ii.getSecurityStationId(),
                            ii.getSuperviseDepartId(), ii.isSponsor()
                    )).collect(Collectors.toList());

            commandResponse = this.roomService.dispatch(
                    CreateRoomCommand.create(
                            request.getDeskType(),
                            request.getCommunicatedType(),
                            request.getYardmanType(),
                            request.getReferenceType() == null ? DispatchReasonType.Normal : request.getReferenceType(),
                            request.getReferenceNumber(),
                            roomUserCommands)
            );
        } else {
            Collection<String> notPointsOrganizationIds =
                    orgPurses.stream().filter(ix -> !ix.IsDispatchable())
                            .map(ix -> ix.getOrganizationId())
                            .collect(Collectors.toList());

            Collection<RdCompanyDocument> companyDocuments =
                    this.rdCompanyDocumentRepository.findAllById(notPointsOrganizationIds);
            if (!CollectionUtils.isEmpty(companyDocuments)) {
                throw new BadTenantException(String.format("如下企业(%s)联动值不足, 无法进行调度",
                        companyDocuments.stream().map(ix -> ix.getName()).collect(Collectors.joining(",")))
                );
            }
        }

        return this.dataTranslate.translate(commandResponse, request, orgPurses);
    }

    @Override
    public InviteRoomUsersWriterRpcResponse inviteUsers(long roomId, InviteRoomUsersWriterRpcRequest request) {
        if (CollectionUtils.isEmpty(request.getRoomUsers())) {
            throw new BadTenantException("参数有误, 参与会议的人员不能为空");
        }

        Collection<String> organizationIds =
                request.getRoomUsers().stream().map(ix -> ix.getOrganizationId())
                        .collect(Collectors.toList());

        Collection<OrgPurseEntity> orgPurses
                = this.orgPurseRepository.getByOrganizationIds(organizationIds);

        InviteRoomUserCommandResponse commandResponse = null;
        if (request.getRoomUsers().stream()
                .allMatch(ix ->
                        orgPurses.stream().anyMatch(ir -> ir.getOrganizationId().equalsIgnoreCase(ix.getOrganizationId()) && ir.IsDispatchable()))
        ) {
            Collection<InviteRoomUserCommand.InviteUserCommandItem> roomUserCommands =
                    request.getRoomUsers().stream().map(ii ->
                            InviteRoomUserCommand.InviteUserCommandItem.create(
                            ii.getEmployeeId(),
                            ii.getOrganizationId(),
                            ii.getTencentUserId(),
                            ii.getSecurityStationId(),
                            ii.getSuperviseDepartId()
                    )).collect(Collectors.toList());

            commandResponse = this.roomService.dispatch(InviteRoomUserCommand.create(roomId, roomUserCommands));
        } else {
            Collection<String> notPointsOrganizationIds =
                    orgPurses.stream().filter(ix -> !ix.IsDispatchable())
                            .map(ix -> ix.getOrganizationId())
                            .collect(Collectors.toList());

            Collection<RdCompanyDocument> companyDocuments =
                    this.rdCompanyDocumentRepository.findAllById(notPointsOrganizationIds);
            if (!CollectionUtils.isEmpty(companyDocuments)) {
                throw new BadTenantException(String.format("如下企业(%s)联动值不足, 无法进行调度",
                        companyDocuments.stream().map(ix -> ix.getName()).collect(Collectors.joining(",")))
                );
            }
        }

        return this.dataTranslate.translate(commandResponse, request, orgPurses);
    }

    @Override
    public RoomResponse get(Long id) {
        Optional<RoomDocument> roomDocumentOptional = this.roomDocumentRepository.findById(id);
        if (!roomDocumentOptional.isPresent()) {
            return null;
        }
        RoomDocument document = roomDocumentOptional.get();

        Collection<RoomResponse.RoomUserResponse> roomUsers =
                document.getRoomUsers().stream().map(ix -> RoomResponse.RoomUserResponse.create(
                        ix.getEmployeeId(),
                        ix.getEmployeeName(),
                        ix.getOrganizationId(), ix.getOrganizationName(),
                        ix.getTencentUserId(), ix.getSecurityStationId(),
                        ix.getSecurityStationName(),
                        ix.getSuperviseDepartId(),
                        ix.isSponsor(),
                        ix.getCid()
                )).collect(Collectors.toList());

        return RoomResponse.create(document.getId(),
                document.getActivated() == null ? false : document.getActivated().booleanValue(),
                roomUsers);
    }

    @Override
    public void close(Long id) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        this.roomService.dispatch(
                CloseRoomCommand.create(id, userModel.getOrganizationId())
        );
    }

    @Override
    public Collection<Long> getReadyForCalculatedRoomIds(int pageSize) {
        return this.roomRepository.getReadyForCalculatedRoomIds(pageSize);
    }

    @Override
    public void removeUser(long roomId, String tencentUserId) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }
        roomService.dispatch(RemoveUserCommand.create(roomId, tencentUserId));
    }

    @Override
    public Collection<String> getTrtcSendToSessionSpecialIds(long roomId, TRTCActionWriterRpcRequest request) {
        if (request.getAction() == null || request.getTencentUserIds() == null) {
            throw new BadTenantException("请求参数错误!");
        }

        Optional<RoomDocument> roomDocumentOptional = roomDocumentRepository.findById(roomId);
        if (!roomDocumentOptional.isPresent()) {
            throw new NoFoundTenantException("房间号无效！");
        }

        Collection<RoomDocument.RoomUserDocument> roomUsers = roomDocumentOptional.get().getRoomUsers();
        Collection<String> specialTencentIds = null;
        switch (request.getAction()) {
            case OPENWORD, STOPWORD, CAMON, CAMOFF -> {
                if (request.getTencentUserIds().length == 0) {
                    specialTencentIds = roomUsers.stream().filter(ix -> !ix.isSponsor()).map(ix -> ix.getTencentUserId()).collect(Collectors.toList());
                } else {
                    specialTencentIds = Arrays.stream(request.getTencentUserIds())
                            .filter(ix -> roomUsers.stream().anyMatch(ii -> ii.getTencentUserId().equals(ix))).collect(Collectors.toList());
                }
            }
            case HANDUP, CHEESE -> {
                String tencentUserId = request.getTencentUserIds()[0];
                if (!roomUsers.stream().anyMatch(ix -> ix.getTencentUserId().equals(tencentUserId))) {
                    throw new NoFoundTenantException("该人员不在房间内");
                }
                specialTencentIds =
                        roomUsers.stream().filter(ix -> ix.isSponsor()).map(ix -> ix.getTencentUserId()).collect(Collectors.toList());
            }
            case WORDALLOWED, WORDDENY, CAMALLOWED, CAMDENY -> {
                //不支持全量允许和拒绝
                if (request.getTencentUserIds().length == 0) {
                    throw new BadTenantException("参数错误，当前允许/拒绝请求没有指定用户");
                }
                specialTencentIds = Arrays.stream(request.getTencentUserIds())
                        .filter(ix -> roomUsers.stream().anyMatch(ii -> ii.getTencentUserId().equals(ix))).collect(Collectors.toList());
            }
        }
        return specialTencentIds;
    }

    @Override
    public void dispatch(RoomReplyRequest request) {
        String content = this.jsonProvider.getJson(request);
        String roomId = String.valueOf(request.getId());

        int partition = Math.abs(roomId.hashCode() % KafkaConstants.PARTITION_COUNT);

        this.kafkaTemplate.send(
                KafkaConstants.TOPIC_DISPATCH_ROOM_EMPLOYEE_ACTION,
                partition,
                roomId,
                content
        );
    }
}
