package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.criterias.RollCallPlanSearchCriteria;
import com.bcxin.tenant.open.domains.entities.RollCallPlanEntity;
import com.bcxin.tenant.open.domains.pojo.RollCallPlanTimerPoJo;
import com.bcxin.tenant.open.domains.repositories.RollCallPlanRepository;
import com.bcxin.tenant.open.domains.services.RollCallPlanService;
import com.bcxin.tenant.open.domains.services.commands.CreateRollCallPlanCommand;
import com.bcxin.tenant.open.domains.services.commands.DeleteRollCallPlanCommand;
import com.bcxin.tenant.open.domains.services.commands.UpdateRollCallPlanCommand;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.RepetitiveRule;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.RollCallPlanRpcProvider;
import com.bcxin.tenant.open.jdks.requests.RollCallPlanRequest;
import com.bcxin.tenant.open.jdks.requests.RollCallPlanSearchRequest;
import com.bcxin.tenant.open.jdks.requests.ScheduledRollCallPlanSearchRequest;
import com.bcxin.tenant.open.jdks.responses.RollCallPlanDetailResponse;
import com.bcxin.tenant.open.jdks.responses.RollCallPlanResponse;
import com.bcxin.tenant.open.jdks.responses.RollCallPlanScheduledResponse;
import org.apache.dubbo.config.annotation.DubboService;

import java.sql.Time;
import java.util.Collection;
import java.util.Date;
import java.util.stream.Collectors;

@DubboService
public class RollCallPlanRpcProviderImpl implements RollCallPlanRpcProvider {
    private final RollCallPlanService rollCallPlanService;
    private final RollCallPlanRepository rollCallPlanRepository;
    private final JsonProvider jsonProvider;

    public RollCallPlanRpcProviderImpl(RollCallPlanService rollCallPlanService,
                                       RollCallPlanRepository rollCallPlanRepository,
                                       JsonProvider jsonProvider) {
        this.rollCallPlanService = rollCallPlanService;
        this.rollCallPlanRepository = rollCallPlanRepository;
        this.jsonProvider = jsonProvider;
    }


    @Override
    public void create(RollCallPlanRequest request) {
        this.rollCallPlanService.dispatch(
                CreateRollCallPlanCommand.create(
                        request.getName(),
                        request.getBeginDate(),
                        request.getEndDate(),
                        request.getRepetitiveRules(),
                        request.getSuperviseDepartId(),
                        request.getSuperviseDepartName(),
                        request.getStationTypes(),
                        request.getTimers()
                )
        );
    }

    @Override
    public void update(long id, RollCallPlanRequest request) {
        this.rollCallPlanService.dispatch(
                UpdateRollCallPlanCommand.create(
                        id,
                        request.getName(),
                        request.getBeginDate(),
                        request.getEndDate(),
                        request.getRepetitiveRules(),
                        request.getSuperviseDepartId(),
                        request.getSuperviseDepartName(),
                        request.getStationTypes(),
                        request.getTimers()
                )
        );
    }

    @Override
    public void delete(long id) {
        this.rollCallPlanService.dispatch(DeleteRollCallPlanCommand.create(id));
    }

    @Override
    public EntityCollection<RollCallPlanResponse> search(RollCallPlanSearchRequest request) {
        EntityCollection<RollCallPlanEntity> rollCallPlanCollection =
                this.rollCallPlanRepository.search(RollCallPlanSearchCriteria.create(request.getPageIndex(),request.getPageSize(),request.getName(), request.getSuperviseDepartIds(), request.getStationTypes()));

        Collection<RollCallPlanResponse> responses =
                rollCallPlanCollection.getData().stream().map(ii -> {
                    return RollCallPlanResponse.create(ii.getId(), ii.getName(), ii.getSuperviseDepartName(),
                            jsonProvider.toObjects(String.class, ii.getStationTypes()),
                            ii.getBeginDate(), ii.getEndDate(),
                            ii.getCountOfInstances()
                            );
                        }

                ).collect(Collectors.toSet());

        return EntityCollection.create(responses, rollCallPlanCollection.getPageSize(), rollCallPlanCollection.getTotalCount());
    }

    @Override
    public RollCallPlanDetailResponse get(Long id) {
        RollCallPlanEntity rollCallPlan = this.rollCallPlanRepository.getById(id);
        if (rollCallPlan == null) {
            return null;
        }

        Collection<Time> timers = null;
        if (!StringUtil.isEmpty(rollCallPlan.getPeriodsOfTimes())) {
            timers = this.jsonProvider.toObjects(RollCallPlanTimerPoJo.class, rollCallPlan.getPeriodsOfTimes())
                    .stream().map(ii -> ii.getValue()).collect(Collectors.toList());
        }

        RollCallPlanDetailResponse response
                = RollCallPlanDetailResponse.create(rollCallPlan.getId(),
                rollCallPlan.getName(),
                rollCallPlan.getBeginDate(),
                rollCallPlan.getEndDate(),
                RepetitiveRule.getSelectedRepetitiveRules(rollCallPlan.getRepetitiveRule()),
                rollCallPlan.getSuperviseDepartId(),
                rollCallPlan.getSuperviseDepartName(),
                this.jsonProvider.toObjects(String.class, rollCallPlan.getStationTypes()),
                timers);

        return response;
    }

    @Override
    public Collection<RollCallPlanScheduledResponse> getScheduledRollCallPlans(ScheduledRollCallPlanSearchRequest request) {
        Collection<RollCallPlanEntity> callPlans =
                rollCallPlanRepository.getScheduledRollCallPlans(
                request.getScheduledTime(),
                request.getRepetitiveRule()
        );

        return callPlans.stream().map(ii -> {
            Collection<String> stationTypes = this.jsonProvider.toObjects(String.class, ii.getStationTypes());
            Collection<RollCallPlanTimerPoJo> timerPoJos = this.jsonProvider.toObjects(RollCallPlanTimerPoJo.class, ii.getPeriodsOfTimes());

            RollCallPlanScheduledResponse ds =
                    RollCallPlanScheduledResponse.create(
                            ii.getId(),
                            ii.getRepetitiveRule(),
                            ii.getSuperviseDepartId(),
                            stationTypes,
                            timerPoJos.stream()
                                    .filter(ix -> ix.getValue() != null)
                                    .map(ix -> ix.getValue())
                                    .collect(Collectors.toList()),
                            ii.getOrganizationId(),
                            ii.getOrganizationInstitutional(),
                            ii.getOrganizationName(),
                            ii.getCreatorId(),
                            ii.getCreatorName()
                    );

            return ds;
        }).collect(Collectors.toList());
    }
}
