package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.criterias.RollCallEmployeeCriteria;
import com.bcxin.tenant.open.domains.criterias.RollCallEmployeeGroupCriteria;
import com.bcxin.tenant.open.domains.dtos.RollCallEmployeeGroupSearchDTO;
import com.bcxin.tenant.open.domains.entities.RollCallEmployeeEntity;
import com.bcxin.tenant.open.domains.pojo.SecurityDepartPoJo;
import com.bcxin.tenant.open.domains.pojo.SecurityStationPoJo;
import com.bcxin.tenant.open.domains.repositories.RollCallEmployeeRepository;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.RollCallEmployeeReaderRpcProvider;
import com.bcxin.tenant.open.jdks.requests.RollCallEmployeeGroupSearchRequest;
import com.bcxin.tenant.open.jdks.requests.RollCallEmployeeSearchRequest;
import com.bcxin.tenant.open.jdks.responses.RollCallEmployeeGroupSearchResponse;
import com.bcxin.tenant.open.jdks.responses.RollCallEmployeeSearchResponse;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@DubboService
public class RollCallEmployeeReaderRpcProviderImpl implements RollCallEmployeeReaderRpcProvider {
    private final RollCallEmployeeRepository rollCallEmployeeRepository;
    private final JsonProvider jsonProvider;

    public RollCallEmployeeReaderRpcProviderImpl(RollCallEmployeeRepository rollCallEmployeeRepository,
                                                 JsonProvider jsonProvider) {
        this.rollCallEmployeeRepository = rollCallEmployeeRepository;
        this.jsonProvider = jsonProvider;
    }


    @Override
    public EntityCollection<RollCallEmployeeGroupSearchResponse> search(RollCallEmployeeGroupSearchRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        String[] organizationIds = null;
        if(!userModel.isSuperviseDepartRole()) {
            organizationIds = userModel.getNoSuperviseScopePermissions();
        }

        RollCallEmployeeGroupCriteria criteria =
                RollCallEmployeeGroupCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getRollCallPlanId(),
                        organizationIds,
                        request.getStationName(),
                        request.getSuperviseDepartIds(),
                        request.getStationTypes(),
                        request.getActionEmployeeName(),
                        request.getBeginDate(),
                        request.getEndDate(),
                        request.getCallTypes(),
                        userModel.getOrganizationId()
                );

        EntityCollection<RollCallEmployeeGroupSearchDTO> rollCallEmployeeEntityEntityCollection =
                this.rollCallEmployeeRepository.search(criteria);

        Collection<RollCallEmployeeGroupSearchResponse> data =
                rollCallEmployeeEntityEntityCollection.getData().stream().map(ii -> {
                                    return RollCallEmployeeGroupSearchResponse.create(
                                            ii.getRollCallId(),ii.getStationId(),
                                            ii.getStationName(), ii.getStationTypes(),
                                            ii.getSuperviseDepartName(), ii.getOrganizationName(),
                                            ii.getCreatedTime(), ii.getCountOfMembers(), ii.getCountOfSuccess(), ii.getCountOfFailed(),
                                            ii.getActionEmployeeName()
                                    );
                                }
                        )
                        .collect(Collectors.toList());

        return EntityCollection.create(
                data, rollCallEmployeeEntityEntityCollection.getPageSize(),
                rollCallEmployeeEntityEntityCollection.getTotalCount());
    }

    @Override
    public Collection<RollCallEmployeeSearchResponse> search(RollCallEmployeeSearchRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        String actionOrganizationId = null;
        String calledEmployeeId = null;
        if(request.isOnlyForBeingCalledRecord()) {
            if (request.getBeginDate() == null || request.getEndDate() == null) {
                throw new ArgumentTenantException("过滤日期的值不能为空");
            }

            calledEmployeeId = userModel.getEmployeeId();
        }else {
            if (!StringUtils.hasLength(request.getRollCallId())) {
                throw new ArgumentTenantException("点名RollCallId不能为空");
            }

            actionOrganizationId = userModel.getOrganizationId();
        }


        RollCallEmployeeCriteria criteria
                = RollCallEmployeeCriteria.create(
                request.getPageIndex(),
                request.getPageSize(),
                calledEmployeeId,
                request.getBeginDate(),
                request.getEndDate(),
                request.getRollCallId(),
                request.getCallTypes(),
                request.getCallStatuses(),
                actionOrganizationId,
                request.getSecurityStationId()
        );

        Collection<RollCallEmployeeEntity> employeeEntityEntityCollection =
                this.rollCallEmployeeRepository.search(criteria);

        Collection<RollCallEmployeeSearchResponse> data
                = employeeEntityEntityCollection.stream().map(ii -> {

            SecurityStationPoJo poJo = this.jsonProvider.toObject(SecurityStationPoJo.class, ii.getSecurityStationJson());
            SecurityDepartPoJo departPoJo = this.jsonProvider.toObject(SecurityDepartPoJo.class, ii.getSecurityDepartJson());

            return RollCallEmployeeSearchResponse.create(
                    ii.getRollCallId(), ii.getCreatedTime(),
                    ii.getCallType(),
                    ii.getStatus(), ii.getActionOrganizationName(),
                    poJo.getId(),
                    poJo.getName(),
                    departPoJo.getId(),
                    ii.getCalledEmployeeId(),
                    ii.getCalledEmployeeName(),
                    ii.getCalledTenantUserId(),
                    ii.getCalledTencentUserId(),
                    ii.getCalledOrganizationId(),
                    ii.getCalledOrganizationName(),
                    poJo.isMaster()
            );
        }).collect(Collectors.toList());

        return data;
    }
}
