package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.dtos.RdSyncEventWrapperDTO;
import com.bcxin.tenant.open.domains.entities.RdSyncEventEntity;
import com.bcxin.tenant.open.domains.enums.EventSourceType;
import com.bcxin.tenant.open.domains.repositories.RdSyncEventRepository;
import com.bcxin.tenant.open.domains.repositories.RdSyncEventRepositoryImpl;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.enums.ProcessedStatus;
import com.bcxin.tenant.open.jdks.CompanyWriterRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeWriterRpcProvider;
import com.bcxin.tenant.open.jdks.RdSyncEventRpcProvider;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.function.Consumer;
import java.util.stream.Collectors;

@DubboService
public class RdSyncEventRpcProviderImpl implements RdSyncEventRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(RdSyncEventRepositoryImpl.class);
    private final RdSyncEventRepository syncEventRepository;
    private final EmployeeWriterRpcProvider employeeWriterRpcProvider;
    private final CompanyWriterRpcProvider companyWriterRpcProvider;
    private final RdSecurityStationWriterRpcProviderImpl securityStationWriterRpcProvider;

    public RdSyncEventRpcProviderImpl(RdSyncEventRepository syncEventRepository,
                                      EmployeeWriterRpcProvider employeeWriterRpcProvider,
                                      CompanyWriterRpcProvider companyWriterRpcProvider,
                                      RdSecurityStationWriterRpcProviderImpl securityStationWriterRpcProvider) {
        this.syncEventRepository = syncEventRepository;
        this.employeeWriterRpcProvider = employeeWriterRpcProvider;
        this.companyWriterRpcProvider = companyWriterRpcProvider;
        this.securityStationWriterRpcProvider = securityStationWriterRpcProvider;
    }

    @Override
    public int process() {
        Collection<RdSyncEventEntity> syncEvents =
                this.syncEventRepository.getAllPending(100);

        if(CollectionUtils.isEmpty(syncEvents)) {
            return 0;
        }

        doExecuteSync(syncEvents, EventSourceType.Employee, ids -> {
            this.employeeWriterRpcProvider.flush2Redis(ids,false);
        });

        doExecuteSync(syncEvents, EventSourceType.Company, ids -> {
            this.companyWriterRpcProvider.flush2Redis(ids, DispatchDataType.Company);
        });

        doExecuteSync(syncEvents, EventSourceType.Station, ids -> {
            this.securityStationWriterRpcProvider.flush2Redis(ids);
        });

        return syncEvents.size();
    }

    private void doExecuteSync(Collection<RdSyncEventEntity> syncEvents,
                               EventSourceType sourceType,
                               Consumer<Collection<String>> consumer) {
        Collection<RdSyncEventEntity> selectedSyncEvents = syncEvents.stream()
                .filter(ix -> ix.getSourceType() == sourceType)
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(selectedSyncEvents)) {
            Collection<String> elementIds
                    = selectedSyncEvents.stream().map(ix -> ix.getId()).collect(Collectors.toList());
            try {
                consumer.accept(elementIds);

                syncEventRepository.process(RdSyncEventWrapperDTO.create(
                        selectedSyncEvents.stream()
                                .map(ix -> RdSyncEventWrapperDTO.RdSyncEventItemDTO.create(ix.getPkId(), ProcessedStatus.Done, "Done"))
                                .collect(Collectors.toList())
                ));

                logger.error("成功: 自动({})刷新同步{}({})到redis",
                        sourceType,
                        elementIds.stream()
                                .map(ix -> String.format("'%s'", ix))
                                .collect(Collectors.joining(",")),
                        sourceType);
            } catch (Exception ex) {
                logger.error("异常: 自动()刷新同步{}({})到redis发生异常:{}",
                        sourceType,
                        elementIds.stream()
                                .map(ix -> String.format("'%s'", ix))
                                .collect(Collectors.joining(",")),
                        sourceType,
                        ex.toString());

                ex.printStackTrace();
            }
        }
    }
}
