package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdSecurityStationDocument;
import com.bcxin.tenant.open.document.domains.repositories.RdSecurityStationDocumentRepository;
import com.bcxin.tenant.open.document.domains.utils.GeoPointUtils;
import com.bcxin.tenant.open.domains.entities.RdSecurityStationEntity;
import com.bcxin.tenant.open.domains.entities.StationDeviceEntity;
import com.bcxin.tenant.open.domains.readers.RdCompositedReader;
import com.bcxin.tenant.open.domains.repositories.RdSecurityStationRepository;
import com.bcxin.tenant.open.domains.repositories.StationDeviceRepository;
import com.bcxin.tenant.open.domains.services.SecurityStationService;
import com.bcxin.tenant.open.domains.services.commands.CreateSyncStationCommand;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.utils.DateUtils;
import com.bcxin.tenant.open.jdks.CompanyWriterRpcProvider;
import com.bcxin.tenant.open.jdks.SecurityStationWriterRpcProvider;
import com.redis.om.spring.search.stream.EntityStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.geo.Point;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.*;
import java.util.stream.Collectors;

@DubboService
public class RdSecurityStationWriterRpcProviderImpl implements SecurityStationWriterRpcProvider {
    private final static Logger logger = LoggerFactory.getLogger(RdSecurityStationWriterRpcProviderImpl.class);

    private final RdSecurityStationRepository securityStationRepository;
    private final RdSecurityStationDocumentRepository securityStationDocumentRepository;
    private final DataTranslate dataTranslate;
    private final CompanyWriterRpcProvider companyWriterRpcProvider;
    private final RdCompositedReader compositedReader;
    private final SecurityStationService securityStationService;

    private final StationDeviceRepository deviceRepository;


    public RdSecurityStationWriterRpcProviderImpl(
            RdSecurityStationRepository securityStationRepository,
            RdSecurityStationDocumentRepository securityStationDocumentRepository,
            DataTranslate dataTranslate, EntityStream entityStream,
            CompanyWriterRpcProvider companyWriterRpcProvider,
            RdCompositedReader compositedReader,
            SecurityStationService securityStationService, StationDeviceRepository deviceRepository) {
        this.securityStationRepository = securityStationRepository;
        this.securityStationDocumentRepository = securityStationDocumentRepository;
        this.dataTranslate = dataTranslate;
        this.compositedReader = compositedReader;
        this.companyWriterRpcProvider = companyWriterRpcProvider;
        this.securityStationService = securityStationService;
        this.deviceRepository = deviceRepository;
    }

    @Override
    public int flush2Redis(Collection<String> ids) {
        int pageIndex = 1;
        int pageSize = 500;

        int totalCount = 0;

        Collection<RdSecurityStationEntity> securityStationEntities =
                this.securityStationRepository.getByPage(ids, pageIndex, pageSize);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }

        while (!CollectionUtils.isEmpty(securityStationEntities)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = securityStationEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<StationDeviceEntity> stationDevices = this.deviceRepository.getByStationIds(ids);
            Collection<RdSecurityStationDocument> documents =
                    this.dataTranslate.translate2Stations(securityStationEntities,
                            this.securityStationDocumentRepository,
                            this.compositedReader,
                            stationDevices);

            if (CollectionUtils.isEmpty(documents)) {
                return totalCount;
            }


            Date now = DateUtils.getToday();
            Collection<RdSecurityStationDocument> deletedDocuments
                    = documents.stream()
                    .filter(ii -> ii.getBeginDate().after(now) || ii.getEndDate().before(now) ||
                            !(ii.getSiteState().equalsIgnoreCase("已完善") || ii.getSiteState().equalsIgnoreCase("生效"))
                    ).collect(Collectors.toList());

            Collection<RdSecurityStationDocument> updatedDocuments
                    = documents.stream()
                    .filter(ii -> {
                                boolean flag = (ii.getSiteState().equalsIgnoreCase("已完善") ||
                                        ii.getSiteState().equalsIgnoreCase("生效"));
                                if (!flag) {
                                    return false;
                                }

                                Timestamp endDate = DateUtils.getSpecialDayEndTimestamp(ii.getEndDate());
                                return ii.getBeginDate().before(DateUtils.getTodayEndTimestamp()) && endDate.after(now);
                            }
                    )
                    .collect(Collectors.toList());
            if (deletedDocuments != null && !deletedDocuments.isEmpty()) {
                this.securityStationDocumentRepository.deleteAll(deletedDocuments);
                logger.error("删除如下驻勤内容:{};",
                        deletedDocuments.stream()
                                .map(ix -> ix.getId()).collect(Collectors.joining(";"))
                );
            }

            this.securityStationDocumentRepository.saveAll(updatedDocuments);
            logger.info("正在刷新驻勤信息数据到RedisJson:pageIndex={}; count={};", pageIndex, documents.size());

            /**
             * 针对无效坐标的情况, 这边做double check
             */
            Collection<RdSecurityStationDocument> invalidStationDocuments =
                    updatedDocuments.stream().filter(ii -> {
                        Point lcn = ii.getLonLat();
                        if (lcn == null) {
                            return true;
                        }

                        if (Math.ceil(lcn.getX()) == lcn.getX() || Math.ceil(lcn.getY()) == lcn.getY()) {
                            return true;
                        }

                        return false;
                    }).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(invalidStationDocuments)) {
                Collection<String> securityIds =
                        invalidStationDocuments.stream()
                                .map(ii -> ii.getId())
                                .collect(Collectors.toList());

                Collection<RdSecurityStationDocument> expectedDocuments =
                        this.securityStationDocumentRepository.findAllById(securityIds);

                Collection<RdSecurityStationDocument> filteredInvalidStationDocuments =
                        invalidStationDocuments.stream()
                                .filter(ii -> expectedDocuments.stream().anyMatch(ix -> ix.getId().equalsIgnoreCase(ii.getId())))
                                .collect(Collectors.toList());

                if(!CollectionUtils.isEmpty(filteredInvalidStationDocuments)) {
                    for (RdSecurityStationDocument doc : filteredInvalidStationDocuments) {
                        doc.setLonLat(GeoPointUtils.translate(null));
                    }

                    this.securityStationDocumentRepository.saveAll(filteredInvalidStationDocuments);
                }
            }

            totalCount += documents.size();
            pageIndex++;
            if (documents.size() < pageSize) {
                break;
            }

            securityStationEntities = this.securityStationRepository.getByPage(ids, pageIndex, pageSize);

        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.securityStationDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }

    @Override
    public void flush2DbRedis(String content) {
        securityStationService.dispatch(CreateSyncStationCommand.create(content));
    }

    @Override
    public Collection<String> getInvalidStationIds(int pageSize) {
        Collection<String> ids =
                this.securityStationRepository.getInvalidStationIds(pageSize);

        return ids;
    }
}
