package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdCompanyDocument;
import com.bcxin.tenant.open.document.domains.documents.RdSecurityStationDocument;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdSecurityStationDocumentRepository;
import com.bcxin.tenant.open.domains.criterias.RdSecurityStationRailCriteria;
import com.bcxin.tenant.open.domains.entities.RdSecurityStationRailEntity;
import com.bcxin.tenant.open.domains.repositories.RdSecurityStationRailRepository;
import com.bcxin.tenant.open.domains.services.RdSecurityStationRailService;
import com.bcxin.tenant.open.domains.services.commands.CheckStationRailWarningCommand;
import com.bcxin.tenant.open.domains.services.commands.CreateRdSecurityStationRailCommand;
import com.bcxin.tenant.open.domains.services.commands.DeleteRdSecurityStationRailCommand;
import com.bcxin.tenant.open.domains.services.commands.UpdateRdSecurityStationRailCommand;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.valueTypes.RdSecurityStationRailSnapshootValueType;
import com.bcxin.tenant.open.jdks.RdSecurityStationRailWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.RdSecurityStationRailRequest;
import com.bcxin.tenant.open.jdks.requests.RdSecurityStationRailSearchRequest;
import com.bcxin.tenant.open.jdks.requests.SyncUpdateGeoRequest;
import com.bcxin.tenant.open.jdks.responses.RdSecurityStationRailSearchResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationRailDetailResponse;
import org.apache.commons.collections.CollectionUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.apache.dubbo.config.annotation.Method;
import org.springframework.util.StringUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@DubboService(methods = {@Method(name = "doCheck",async = false)})
public class RdSecurityStationRailWriterRpcProviderImpl implements RdSecurityStationRailWriterRpcProvider {
    private final RdSecurityStationRailService securityStationRailService;
    private final RdSecurityStationRailRepository rdSecurityStationRailRepository;
    private final RdSecurityStationDocumentRepository securityStationDocumentRepository;

    private final RdCompanyDocumentRepository companyDocumentRepository;
    private final JsonProvider jsonProvider;

    public RdSecurityStationRailWriterRpcProviderImpl(RdSecurityStationRailService securityStationRailService,
                                                      RdSecurityStationRailRepository rdSecurityStationRailRepository,
                                                      RdSecurityStationDocumentRepository securityStationDocumentRepository,
                                                      RdCompanyDocumentRepository companyDocumentRepository, JsonProvider jsonProvider) {
        this.securityStationRailService = securityStationRailService;
        this.rdSecurityStationRailRepository = rdSecurityStationRailRepository;
        this.securityStationDocumentRepository = securityStationDocumentRepository;
        this.companyDocumentRepository = companyDocumentRepository;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public long create(RdSecurityStationRailRequest request) {
        CreateRdSecurityStationRailCommand command =
                CreateRdSecurityStationRailCommand.create(
                request.getName(),
                request.getShapeType(),
                request.getShapedLocation(),
                request.getStationId(),
                request.isFromStationManagement(),
                request.getStationName(),
                request.getSuperviseDepartId(),
                request.getSuperviseDepartName(),
                request.getBeginTime(),
                request.getEndTime(),
                request.getRuleType(),
                request.getNote(),
                request.getOrganizationId());

        return this.securityStationRailService.dispatch(command);
    }

    @Override
    public void update(long id, RdSecurityStationRailRequest request) {
        UpdateRdSecurityStationRailCommand command
                = UpdateRdSecurityStationRailCommand.create(id,
                request.getName(), request.getShapeType(),
                request.getShapedLocation(),
                request.getStationId(),
                request.isFromStationManagement(),
                request.getStationName(),
                request.getSuperviseDepartId(),
                request.getSuperviseDepartName(),
                request.getBeginTime(),
                request.getEndTime(), request.getRuleType(),
                request.getNote());

        this.securityStationRailService.dispatch(command);
    }

    @Override
    public void delete(long id) {
        this.securityStationRailService.dispatch(DeleteRdSecurityStationRailCommand.create(id));
    }

    @Override
    public Collection<RdSecurityStationRailSearchResponse> search(RdSecurityStationRailSearchRequest request) {

        RdSecurityStationRailCriteria criteria= null;
        if(request.getSearchModel()==null){
           throw new BadTenantException("SearchModel不能为空");
        }
        switch (request.getSearchModel()){
            case Station -> {
                if(StringUtils.hasLength(request.getStationId())) {
                    criteria = RdSecurityStationRailCriteria.create(request.getStationId(), null,null,RdSecurityStationRailCriteria.SearchModel.Station);
                }else{
                    return Collections.EMPTY_LIST;
                }
            }
            case SecurityResource -> {
                criteria = RdSecurityStationRailCriteria.create(null,request.getSuperviseDepartIds(),null, RdSecurityStationRailCriteria.SearchModel.SecurityResource);
            }
            case Search -> {
                criteria = RdSecurityStationRailCriteria.create(null, null, request.getKeyWord(), RdSecurityStationRailCriteria.SearchModel.Search);
            }
        }
        Collection<RdSecurityStationRailEntity> entities =
                rdSecurityStationRailRepository.search(criteria);

        if (CollectionUtils.isEmpty(entities)) {
            return Collections.EMPTY_LIST;
        }


        List<RdSecurityStationDocument> documents = new ArrayList<>();
        if(request.getSearchModel()!= RdSecurityStationRailSearchRequest.SearchModel.SecurityResource) {
            List<String> ids = new ArrayList<>();
            for (RdSecurityStationRailEntity entity : entities) {
                ids.add(entity.getReferenceNumber());
            }

            documents = this.securityStationDocumentRepository.findAllById(ids);
        }
        List<RdSecurityStationDocument> finalDocuments = documents;
        return entities.stream().map(ix -> {
            String name = ix.getReferenceName();
            if(request.getSearchModel()!= RdSecurityStationRailSearchRequest.SearchModel.SecurityResource) {
                Optional<RdSecurityStationDocument> document =
                        finalDocuments.stream().filter(ii -> ii.getId().equalsIgnoreCase(ix.getReferenceNumber())).findFirst();

                if (document.isPresent()) {
                    name = document.get().getName();
                }
            }

            RdSecurityStationRailSearchResponse
                    response = RdSecurityStationRailSearchResponse.create(
                    request.getSearchModel(),
                    ix.getPkId(),
                    ix.getName(),
                    ix.getShapeType(),
                    this.jsonProvider.toObject(RdSecurityStationRailSnapshootValueType.ShapedLocationValueType.class, ix.getAreaJson()),
                    ix.getReferenceNumber(),
                    name,
                    ix.getBeginTime(),
                    ix.getEndTime(),
                    ix.getRuleType(),
                    ix.getNote(),
                    ix.getOwnerOrganizationId()
            );

            return response;
        }).collect(Collectors.toList());
    }

    @Override
    public void doCheck(Collection<SyncUpdateGeoRequest> syncUpdateGeoRequests) {
        if (CollectionUtils.isEmpty(syncUpdateGeoRequests)) {
            return;
        }

        Collection<CheckStationRailWarningCommand.StationEmployeeLocation> locations =
                syncUpdateGeoRequests.stream()
                        .filter(ix -> ix.getAddress() != null)
                        .map(ix ->
                                CheckStationRailWarningCommand.StationEmployeeLocation.create(
                                        ix.getEmployeeIds(),
                                        ix.getTimestamp(),
                                        ix.getLatitude(), ix.getLongitude(),
                                        ix.getAddress().getCountry(), ix.getAddress().getProvince(),
                                        ix.getAddress().getCity(), ix.getAddress().getDistrict(),
                                        ix.getAddress().getStreet(), ix.getAddress().getStreetNum(),
                                        ix.getAddress().getPoiName(),
                                        ix.getAddress().getCityCode())
                        ).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(locations)) {
            return;
        }

        this.securityStationRailService.dispatch(
                CheckStationRailWarningCommand.create(locations)
        );
    }
    @Override
    public SecurityStationRailDetailResponse get(Long id) {
        RdSecurityStationRailEntity rdSecurityStationRailEntity = rdSecurityStationRailRepository.getById(id);
        if (rdSecurityStationRailEntity == null) {
            return null;
        }
        String referenceNumber = rdSecurityStationRailEntity.getReferenceNumber();
        Optional<RdSecurityStationDocument> securityStationDocumentOptional =
                securityStationDocumentRepository.findById(referenceNumber);
        String companyName = null;
        long countOfSecurityMan = 0;
        String companyId = rdSecurityStationRailEntity.getOrganizationId();
        String stationName = rdSecurityStationRailEntity.getReferenceName();
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        if (securityStationDocumentOptional.isPresent()) {
            RdSecurityStationDocument document = securityStationDocumentOptional.get();
            companyName = document.getCompanyName();
            countOfSecurityMan = document.getCountOfSecurityMan();
            companyId = document.getOrganizationId();
            stationName = document.getName();
        }

        if(!StringUtils.hasLength(companyName)) {
            Optional<RdCompanyDocument> companyDocumentOptional = this.companyDocumentRepository.findById(companyId);
            if (companyDocumentOptional.isPresent()) {
                companyName = companyDocumentOptional.get().getName();
            }
        }

        RdSecurityStationRailSnapshootValueType.ShapedLocationValueType locationValueType =
                this.jsonProvider.toObject(RdSecurityStationRailSnapshootValueType.ShapedLocationValueType.class,
                        rdSecurityStationRailEntity.getAreaJson()
                );

        return SecurityStationRailDetailResponse.create(id,
                rdSecurityStationRailEntity.getName(),
                simpleDateFormat.format(rdSecurityStationRailEntity.getBeginTime()),
                simpleDateFormat.format(rdSecurityStationRailEntity.getEndTime()),
                countOfSecurityMan,
                referenceNumber,
                companyName,
                rdSecurityStationRailEntity.getCreatorName(),
                companyId,
                stationName,
                rdSecurityStationRailEntity.getShapeType(),
                rdSecurityStationRailEntity.getRuleType(),
                locationValueType
        );
    }
}
