package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdProjectDocument;
import com.bcxin.tenant.open.document.domains.repositories.RdProjectDocumentRepository;
import com.bcxin.tenant.open.document.domains.utils.GeoPointUtils;
import com.bcxin.tenant.open.domains.criterias.ProjectAdvanceSearchCriteria;
import com.bcxin.tenant.open.domains.entities.RdProjectEntity;
import com.bcxin.tenant.open.domains.entities.RdProjectTeamGroupEntity;
import com.bcxin.tenant.open.domains.repositories.RdProjectRepository;
import com.bcxin.tenant.open.domains.repositories.RdProjectTeamGroupRepository;
import com.bcxin.tenant.open.domains.repositories.RdProjectTeamMemberRepository;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.open.infrastructures.utils.DateUtils;
import com.bcxin.tenant.open.jdks.ProjectWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.ProjectAdvanceSearchRequest;
import com.bcxin.tenant.open.jdks.responses.ProjectResponse;
import com.bcxin.tenant.open.jdks.responses.ProjectTeamResponse;
import com.redis.om.spring.search.stream.EntityStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.geo.Point;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.stream.Collectors;

@DubboService
public class ProjectWriterRpcProviderImpl implements ProjectWriterRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(ProjectWriterRpcProvider.class);

    private final DataTranslate dataTranslate;
    private final RdProjectDocumentRepository projectDocumentRepository;
    private final RdProjectRepository projectRepository;

    private final RdProjectTeamGroupRepository projectTeamGroupRepository;
    private final RdProjectTeamMemberRepository projectTeamMemberRepository;

    private final EntityStream entityStream;

    public ProjectWriterRpcProviderImpl(DataTranslate dataTranslate,
                                        RdProjectDocumentRepository projectDocumentRepository,
                                        RdProjectRepository projectRepository,
                                        RdProjectTeamGroupRepository projectTeamGroupRepository,
                                        RdProjectTeamMemberRepository projectTeamMemberRepository, EntityStream entityStream) {
        this.dataTranslate = dataTranslate;
        this.projectDocumentRepository = projectDocumentRepository;
        this.projectRepository = projectRepository;
        this.projectTeamGroupRepository = projectTeamGroupRepository;
        this.projectTeamMemberRepository = projectTeamMemberRepository;
        this.entityStream = entityStream;
    }


    @Override
    public int flush2Redis(Collection<String> ids) {
        int pageIndex = 1;
        int pageSize = 500;

        int totalCount = 0;

        Collection<RdProjectEntity> projectEntities =
                this.projectRepository.getByPage(ids, pageIndex, pageSize);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }

        while (!CollectionUtils.isEmpty(projectEntities)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = projectEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<RdProjectDocument> documents =
                    this.dataTranslate.translate(projectEntities,
                            this.projectDocumentRepository);

            if (CollectionUtils.isEmpty(documents)) {
                return totalCount;
            }


            Date now = DateUtils.getToday();
            Collection<RdProjectDocument> deletedDocuments
                    = documents.stream()
                    .filter(ii ->
                            //ii.getBeginDate().after(now) || ii.getEndDate().before(now) || todo:2024-12-15
                            (StringUtils.hasLength(ii.getState()) && ii.getState().equalsIgnoreCase("停用"))
                    ).collect(Collectors.toList());

            Collection<RdProjectDocument> updatedDocuments
                    = documents.stream()
                    .filter(ii -> {
                                boolean flag = (StringUtils.hasLength(ii.getState()) || !"停用".equalsIgnoreCase(ii.getState()));
                                if (!flag) {
                                    return false;
                                }

                                return true;
                                //todo:2024-12-15
                                //Timestamp endDate = DateUtils.getSpecialDayEndTimestamp(ii.getEndDate());
                                //return ii.getBeginDate().before(DateUtils.getTodayEndTimestamp()) && endDate.after(now);
                            }
                    )
                    .collect(Collectors.toList());
            if (deletedDocuments != null && !deletedDocuments.isEmpty()) {
                this.projectDocumentRepository.deleteAll(deletedDocuments);
                logger.error("删除如下项目内容:{};",
                        deletedDocuments.stream()
                                .map(ix -> ix.getId()).collect(Collectors.joining(";"))
                );
            }

            this.projectDocumentRepository.saveAll(updatedDocuments);
            logger.info("正在刷新临保项目信息数据到RedisJson:pageIndex={}; count={};", pageIndex, documents.size());

            /**
             * 针对无效坐标的情况, 这边做double check
             */
            Collection<RdProjectDocument> invalidProjectDocuments =
                    updatedDocuments.stream().filter(ii -> {
                        Point lcn = ii.getLonLat();
                        if (lcn == null) {
                            return true;
                        }

                        if (Math.ceil(lcn.getX()) == lcn.getX() || Math.ceil(lcn.getY()) == lcn.getY()) {
                            return true;
                        }

                        return false;
                    }).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(invalidProjectDocuments)) {
                Collection<String> securityIds =
                        invalidProjectDocuments.stream()
                                .map(ii -> ii.getId())
                                .collect(Collectors.toList());

                Collection<RdProjectDocument> expectedDocuments =
                        this.projectDocumentRepository.findAllById(securityIds);

                Collection<RdProjectDocument> filteredInvalidStationDocuments =
                        invalidProjectDocuments.stream()
                                .filter(ii -> expectedDocuments.stream().anyMatch(ix -> ix.getId().equalsIgnoreCase(ii.getId())))
                                .collect(Collectors.toList());

                if (!CollectionUtils.isEmpty(filteredInvalidStationDocuments)) {
                    for (RdProjectDocument doc : filteredInvalidStationDocuments) {
                        doc.setLonLat(GeoPointUtils.translate(null));
                    }

                    this.projectDocumentRepository.saveAll(filteredInvalidStationDocuments);
                }
            }

            totalCount += documents.size();
            pageIndex++;
            if (documents.size() < pageSize) {
                break;
            }

            projectEntities = this.projectRepository.getByPage(ids, pageIndex, pageSize);

        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.projectDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }

    @Override
    public EntityCollection<ProjectResponse> search(ProjectAdvanceSearchRequest request) {
        var projectCollection =
                this.projectRepository.search(ProjectAdvanceSearchCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getName(),
                        request.getType(),
                        request.getStatus(),
                        request.getBeginDate(),
                        request.getEndDate()
                ));

        var response = projectCollection.getData().stream()
                .map(ii -> this.dataTranslate.translate(ii,this.entityStream,false))
                .collect(Collectors.toSet());

        return EntityCollection.create(response, request.getPageSize(), projectCollection.getTotalCount());
    }

    @Override
    public ProjectResponse get(String id) {
        var data = this.projectRepository.getByV5Id(id);
        if(data==null) {
            return null;
        }

        return this.dataTranslate.translate(data,this.entityStream,true);
    }

    @Override
    public Collection<ProjectTeamResponse> getTeams(String id) {
        if (!StringUtils.hasLength(id)) {
            throw new ArgumentTenantException("参数异常; 项目Id不能为空");
        }

        Collection<RdProjectTeamGroupEntity> teams = projectTeamGroupRepository.getTeams(id);

        return this.dataTranslate.translateProjectTeams(teams);
    }
}
