package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.criterias.ProjectRoadmapCriteria;
import com.bcxin.tenant.open.domains.dtos.ProjectRoadmapDTO;
import com.bcxin.tenant.open.domains.readers.RdCompositedReader;
import com.bcxin.tenant.open.domains.services.ProjectRoadmapService;
import com.bcxin.tenant.open.domains.services.commands.CreateProjectRoadmapCommand;
import com.bcxin.tenant.open.domains.services.commands.DeleteProjectRoadmapCommand;
import com.bcxin.tenant.open.domains.services.commands.UpdateProjectRoadmapCommand;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.ProjectRoadmapRpcProvider;
import com.bcxin.tenant.open.jdks.requests.ProjectRoadmapRequest;
import com.bcxin.tenant.open.jdks.requests.ProjectRoadmapSearchRequest;
import com.bcxin.tenant.open.jdks.responses.ProjectRoadmapResponse;
import org.apache.dubbo.config.annotation.DubboService;

import java.util.Collection;
import java.util.stream.Collectors;

@DubboService
public class ProjectRoadmapRpcProviderImpl implements ProjectRoadmapRpcProvider {
    private final ProjectRoadmapService projectRoadmapService;
    private final RdCompositedReader compositedReader;
    private final JsonProvider jsonProvider;

    public ProjectRoadmapRpcProviderImpl(ProjectRoadmapService projectRoadmapService,
                                         RdCompositedReader compositedReader, JsonProvider jsonProvider) {
        this.projectRoadmapService = projectRoadmapService;
        this.compositedReader = compositedReader;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public EntityCollection<ProjectRoadmapResponse> search(ProjectRoadmapSearchRequest request) {
        EntityCollection<ProjectRoadmapDTO> roadmapDTOs = this.compositedReader.search(
                ProjectRoadmapCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getName(),
                        request.getOrganizationName(),
                        request.getProjectIds()
                )
        );

        Collection<ProjectRoadmapResponse> responses =
                roadmapDTOs.getData().stream().map(ii -> {
                    Object roadmap = this.jsonProvider.toObject(Object.class, ii.getRoadmapJson());
                    Object roadmapImage = this.jsonProvider.toObject(Object.class, ii.getRoadmapImageJson());
                    return new ProjectRoadmapResponse(
                            ii.getId(),
                            ii.getName(),
                            roadmapImage,
                            ii.getProjectId(),
                            ii.getProjectName(),
                            roadmap, ii.getOrganizationId(), ii.getOrganizationName(), ii.getLastUpdatedTime(), ii.getCreatedTime()
                    );
                }).collect(Collectors.toList());

        return EntityCollection.create(responses, roadmapDTOs.getPageSize(), roadmapDTOs.getTotalCount());
    }

    @Override
    public void create(ProjectRoadmapRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }
        this.projectRoadmapService.dispatch(
                CreateProjectRoadmapCommand.create(
                        request.getName(), request.getRoadmapImage(),
                        request.getProjectId(),
                        request.getRoadmap(),
                        userModel.getOrganizationId(),
                        userModel.getOrgName(),
                        userModel.getEmployeeId()
                )
        );
    }

    @Override
    public void update(Long id, ProjectRoadmapRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        this.projectRoadmapService.dispatch(
                UpdateProjectRoadmapCommand.create(
                        id,
                        request.getName(), request.getRoadmapImage(),
                        request.getProjectId(),
                        request.getRoadmap(),
                        userModel.getEmployeeId()
                )
        );
    }

    @Override
    public void delete(Long id) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        this.projectRoadmapService.dispatch(
                DeleteProjectRoadmapCommand.create(id, userModel.getOrganizationId(), userModel.getEmployeeId())
        );
    }
}
