package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.components.SearchSpecialPermissionScopeComponent;
import com.bcxin.tenant.open.document.domains.dtos.SearchSpecialPermissionScopeDTO;
import com.bcxin.tenant.open.domains.criterias.PoliceIncidentRegionReportCriteria;
import com.bcxin.tenant.open.domains.criterias.PoliceIncidentSearchCriteria;
import com.bcxin.tenant.open.domains.criterias.PoliceIncidentsSearchCriteria;
import com.bcxin.tenant.open.domains.dtos.PoliceIncidentPendingStatisticsDTO;
import com.bcxin.tenant.open.domains.dtos.PoliceIncidentRegionReportDTO;
import com.bcxin.tenant.open.domains.entities.PoliceIncidentsEntity;
import com.bcxin.tenant.open.domains.repositories.PoliceIncidentsRepository;
import com.bcxin.tenant.open.domains.services.PoliceIncidentsService;
import com.bcxin.tenant.open.domains.services.commands.*;
import com.bcxin.tenant.open.domains.utils.RedisConstantUtils;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventCategory;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventLevel;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventStatus;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.ForbidTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.jdks.PoliceIncidentsRpcProvider;
import com.bcxin.tenant.open.jdks.SecurityStationReaderRpcProvider;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.CreatePoliceIncidentsRequest;
import com.bcxin.tenant.open.jdks.requests.CreateSosRequest;
import com.bcxin.tenant.open.jdks.requests.PoliceIncidentsSearchRequest;
import com.bcxin.tenant.open.jdks.requests.UpdatePoliceIncidentsRequest;
import com.bcxin.tenant.open.jdks.responses.PoliceIncidentOverviewResponse;
import com.bcxin.tenant.open.jdks.responses.PoliceIncidentRegionReportResponse;
import com.bcxin.tenant.open.jdks.responses.PoliceIncidentsLevelCountResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationDetailResponse;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.core.*;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 警情信息处理
 */
@DubboService
public class PoliceIncidentsRpcProviderImpl implements PoliceIncidentsRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(PoliceIncidentsRpcProviderImpl.class);

    private final PoliceIncidentsService policeIncidentsService;
    private final PoliceIncidentsRepository policeIncidentsRepository;
    private final DataTranslate dataTranslate;
    private final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;

    private final RedisTemplate<String, Object> redisTemplate;
    private final SearchSpecialPermissionScopeComponent specialPermissionScopeComponent;

    private final JsonProvider jsonProvider;

    public PoliceIncidentsRpcProviderImpl(PoliceIncidentsService policeIncidentsService,
                                          PoliceIncidentsRepository policeIncidentsRepository,
                                          DataTranslate dataTranslate,
                                          SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                          RedisTemplate<String, Object> redisTemplate,
                                          SearchSpecialPermissionScopeComponent specialPermissionScopeComponent, JsonProvider jsonProvider) {
        this.policeIncidentsService = policeIncidentsService;
        this.policeIncidentsRepository = policeIncidentsRepository;
        this.dataTranslate = dataTranslate;
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.redisTemplate = redisTemplate;
        this.specialPermissionScopeComponent = specialPermissionScopeComponent;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public void create(CreatePoliceIncidentsRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }
        CreatePoliceIncidentsCommand command =
                CreatePoliceIncidentsCommand.create(
                        request.getCategory(),
                        request.getName(),
                        request.getSceneUrl(), request.getContact(),
                        request.getPoliceEventLevel(),
                        request.getAddress(), request.getReason(), request.getActionTaken(),
                        request.getActionTendency(), request.getPoliceEventType(),
                        request.getLatitude(), request.getLongitude(), request.getDescription(),
                        request.getLocation(),
                        request.getItems()
                );

        this.policeIncidentsService.dispatch(command);
    }

    @Override
    public void updateInfo(UpdatePoliceIncidentsInfoRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }
        UpdatePoliceIncidentsInfoCommand command =
                UpdatePoliceIncidentsInfoCommand.create(
                        request.getId(),
                        request.getSceneUrl(),
                        request.getDescription()
                );

        this.policeIncidentsService.dispatch(command);
    }

    @Override
    public void update(UpdatePoliceIncidentsRequest request) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("暂无授权, 无法访问");
        }

        UpdatePoliceIncidentsCommand command =
                UpdatePoliceIncidentsCommand.create(
                 request.getId(), request.getPoliceEventStatus(),
                request.getActionTaken(), request.getActionTendency(),
                request.getProcessedSuggestion(),
                request.getProcessedLocaleGuidance(),
                        request.getPoints());
        this.policeIncidentsService.dispatch(command);
    }

    @Override
    public void updateStatus(Long id) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("暂无授权, 无法访问");
        }

        this.policeIncidentsService.dispatch(
                UpdatePoliceIncidentStatusCommand.create(
                        id, PoliceEventStatus.Over
                )
        );
    }

    @Override
    public void create(CreateSosRequest request) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();

        if (userModel == null) {
            throw new UnAuthorizedTenantException("当前用户未授权");
        }

        if (BusinessUtil.checkIfDesk(userModel.getAccountType())) {
            throw new UnAuthorizedTenantException("调度台账号不允许操作");
        }

        String stationId =
                userModel.getSecurityStationId();
        String stationAddress = null;
        String organizationName = null;
        String projectId = null;
        String projectName = null;

        if (StringUtils.hasLength(stationId)) {
            SecurityStationDetailResponse detailResponse =
                    this.securityStationReaderRpcProvider.get(stationId);
            if (detailResponse != null) {
                stationAddress = detailResponse.getAddress();
                organizationName = detailResponse.getCompanyName();
                projectId = detailResponse.getProjectId();
                projectName = detailResponse.getProjectName();
            }
        }

        this.policeIncidentsService.dispatch(
                Collections.singleton(CreateSsoPoliceIncidentsCommand.create(
                        userModel.getTencentUserId(),
                        userModel.getEmployeeId(),
                        userModel.getOrganizationId(),
                        organizationName,
                        projectId,
                        projectName,
                        userModel.getSecurityStationId(),
                        userModel.getSuperviseDepartId(),
                        userModel.getSecurityStationName(),
                        stationAddress,
                        request.getLatitude(),
                        request.getLongitude(),
                        request.getAddress(),
                        userModel.getSuperviseDepartName(),
                        userModel.getName(),
                        userModel.getTenantUserIdCard(),
                        ResourceType.toResourceTypes(userModel.getResourceTypesForApp())
                ))
        );
    }

    @Override
    public void create(Collection<CreateSpecialSosRequest> specialSosRequests) {
        if (CollectionUtils.isEmpty(specialSosRequests)) {
            return;
        }
        Collection<CreateSsoPoliceIncidentsCommand> batchSosCommand =
                specialSosRequests.stream().map(ii -> {
                    CreateSsoPoliceIncidentsCommand cmd =
                            CreateSsoPoliceIncidentsCommand.create(
                                    ii.getTencentUserId(),
                                    ii.getTenantEmployeeId(),
                                    ii.getOrganizationId(),
                                    ii.getOrganizationName(),
                                    ii.getProjectId(),
                                    ii.getProjectName(),
                                    ii.getSecurityStationId(),
                                    ii.getSuperviseDepartId(),
                                    ii.getSecurityStationName(),
                                    ii.getStationAddress(),
                                    ii.getLatitude(),
                                    ii.getLongitude(),
                                    ii.getAddress(),
                                    ii.getSuperviseDepartName(),
                                    ii.getTenantUserName(),
                                    ii.getTenantUserIdCard(),
                                    ResourceType.toResourceTypes(ii.getResourceTypes())
                            );

                    return cmd;
                }).collect(Collectors.toList());

        this.policeIncidentsService.dispatch(batchSosCommand);
    }

    @Override
    public void update(UpdateSosPoliceIncidentStatusRequest request) {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("暂无授权, 无法访问");
        }

        this.policeIncidentsService.dispatch(
                UpdateSosPoliceIncidentStatusCommand
                        .create(request.getId(),
                                userModel.getEmployeeId(),
                                userModel.getName()
                        )
        );
    }

    @Override
    public PoliceIncidentOverviewResponse get(Long id) {
        PoliceIncidentsEntity pie = this.policeIncidentsRepository.getById(id);
        if (pie == null) {
            throw new BadTenantException("警情上报记录不存在");
        }

        return this.dataTranslate.translate2PoliceIncidentsDetailReaderResponse(pie);
    }

    @Override
    public PoliceIncidentOverviewResponse layerGet(Long id) {
        PoliceIncidentsEntity pie = this.policeIncidentsRepository.getById(id);
        if (pie == null) {
            throw new BadTenantException("警情上报记录不存在");
        }

        if (PoliceEventStatus.None.equals(pie.getPoliceEventStatus())) {
            this.policeIncidentsService.dispatch(UpdatePoliceIncidentStatusCommand.create(id, PoliceEventStatus.Pending));
        }

        return this.dataTranslate.translate2PoliceIncidentsDetailReaderResponse(pie);
    }

    @Override
    public EntityCollection<PoliceIncidentOverviewResponse> search(PoliceIncidentsSearchRequest request) {
        TenantEmployeeContext.TenantUserModel userModel
                = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("暂无授权, 无法访问");
        }
        Collection<String> departIds
                = Arrays.stream(userModel.getAssignedSuperviseDepartIds())
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(request.getSuperviseDepartIds())) {
            departIds = departIds.stream()
                    .filter(ii -> request.getSuperviseDepartIds().contains(ii)).collect(Collectors.toList());

            if (CollectionUtils.isEmpty(departIds)) {
                return EntityCollection.create(Collections.EMPTY_LIST, 0, 0l);
            }
        }

        Collection<PoliceEventCategory> categories = null;
        if (request.isFromSos()) {
            categories = Collections.singleton(PoliceEventCategory.Sos);
        } else {
            categories = Arrays.stream(PoliceEventCategory.values()).collect(Collectors.toList())
                    .stream().filter(ii -> ii != PoliceEventCategory.Sos).collect(Collectors.toList());
            if (request.getCategory() != null) {
                categories = Collections.singleton(request.getCategory());
            }
        }

        Collection<String> relativeOrganizationIds = new HashSet<>();
        String[] noSuperviseScopePermissions = userModel.getNoSuperviseScopePermissions();
        if(noSuperviseScopePermissions!=null) {
            relativeOrganizationIds = Arrays.stream(noSuperviseScopePermissions).toList();
        }

        Collection<String> otherSpecialPermissionScopeIds = new ArrayList<>();
        SearchSpecialPermissionScopeDTO specialPermissionScope =
                this.specialPermissionScopeComponent.get(userModel);
        if (specialPermissionScope != null && specialPermissionScope.isApplySpecialPermissionScopeRule()) {
            otherSpecialPermissionScopeIds = specialPermissionScope.getScopeIds();
        }

        Collection<String> managedProjectIds = new HashSet<>();
        if(userModel.getNoSuperviseScopePermissions()!=null) {
            managedProjectIds =
                    Arrays.stream(noSuperviseScopePermissions)
                            .filter(ii -> BusinessConstants.isRelativeProjectIdStuffId(ii)).toList();
            if(!CollectionUtils.isEmpty(managedProjectIds)) {
                relativeOrganizationIds = Stream.of(userModel.getOrganizationId()).collect(Collectors.toSet());
            }
        }

        if(request.getLevel()==null) {
            request.setLevel(request.getPoliceEventLevel());
        }

        PoliceIncidentSearchCriteria criteria =
                PoliceIncidentSearchCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getDeskTypes(),
                        relativeOrganizationIds,
                        request.getSearchModel() == PoliceIncidentsSearchRequest.SearchModel.Personal ? userModel.getEmployeeId() : null,
                        userModel.isSuperviseDepartRole(),
                        categories,
                        request.getName(),
                        request.getCreatedStarTime(),
                        request.getCreatedEndTime(),
                        request.getType(),
                        request.getEmpName(),
                        request.getEmpIdNumber(),
                        request.getContact(),
                        request.getOrganizationName(),
                        request.getStationName(),
                        departIds,
                        request.getProcessedStarTime(),
                        request.getProcessedEndTime(),
                        request.getProcessedUserName(),
                        request.getStatus(),
                        otherSpecialPermissionScopeIds,
                        request.getProjectName(),
                        managedProjectIds,
                        request.getLevel(),
                        request.getReason(),
                        request.getRegionName()
                );

        criteria.setForExport(request.isForExport());
        EntityCollection<PoliceIncidentsEntity> policeIncidentsEntityEntityCollection =
                this.policeIncidentsRepository.search(criteria);

        Collection<PoliceIncidentOverviewResponse> details =
                policeIncidentsEntityEntityCollection.getData().stream().map(pie -> {
                    PoliceIncidentOverviewResponse piO = this.dataTranslate.translate2PoliceIncidentsDetailReaderResponse(pie);

                    return piO;
                }).collect(Collectors.toList());

        return EntityCollection.create(
                details,
                policeIncidentsEntityEntityCollection.getPageSize(),
                policeIncidentsEntityEntityCollection.getTotalCount()
        );
    }

    @Override
    public EntityCollection<PoliceIncidentOverviewResponse> getPendingByLevel(PoliceIncidentGetPendingRequest request) {
        Collection<PoliceEventCategory> categories =
                //Arrays.stream(PoliceEventCategory.values()).filter(ii -> ii != PoliceEventCategory.Sos)
                Arrays.stream(PoliceEventCategory.values())
                        .collect(Collectors.toList());

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        if ( userModel.getAssignedSuperviseDepartIds() == null || userModel.getAssignedSuperviseDepartIds().length == 0) {
            return EntityCollection.create(Collections.EMPTY_LIST, request.getPageSize(), 0l);
        }

        Collection<String> superviseDepartIds = Arrays.stream(userModel.getAssignedSuperviseDepartIds()).toList();
        EntityCollection<PoliceIncidentsEntity> policeIncidentsEntityEntityCollection =
                this.policeIncidentsRepository.search(
                        PoliceIncidentsSearchCriteria.create(
                                request.getPageIndex(),
                                request.getPageSize(),
                                superviseDepartIds,
                                request.getLevel(),
                                PoliceEventStatus.None,
                                categories
                        ));

        Collection<PoliceIncidentOverviewResponse> details = policeIncidentsEntityEntityCollection
                .getData().stream().map(pie -> {
                    return this.dataTranslate.translate2PoliceIncidentsDetailReaderResponse(pie);
                }).collect(Collectors.toList());

        return EntityCollection.create(details, request.getPageSize(), policeIncidentsEntityEntityCollection.getTotalCount());
    }

    @Override
    public PoliceIncidentsLevelCountResponse getPendingTotal(PoliceIncidentStatisticsRequest request) {
        if (request == null) {
            throw new BadTenantException("无效参数");
        }

        String orgInstitutional = null;
        Collection<String> superviseDepartIds = request.getSuperviseDepartIds();
        if (!request.isFromSocketRequest()) {
            TenantEmployeeContext.TenantUserModel userModel =
                    TenantContext.getInstance().getUserContext().get();
            if (userModel == null) {
                throw new UnAuthorizedTenantException(String.format("暂无授权, 无法访问:%s", request.isFromSocketRequest()));
            }

            superviseDepartIds = Arrays.stream(userModel.getAssignedSuperviseDepartIds()).toList();

            orgInstitutional = userModel.getOrgInstitutional();
        }

        if (!CollectionUtils.isEmpty(superviseDepartIds) || BusinessUtil.isSupervise(orgInstitutional)) {
            PoliceIncidentsLevelCountResponse countResponse = null;
            if (request.isFromRedis()) {
                Collection<String> finalSuperviseDepartIds = superviseDepartIds;
                countResponse =
                        this.redisTemplate.execute(new SessionCallback<PoliceIncidentsLevelCountResponse>() {
                            @Override
                            public <K, V> PoliceIncidentsLevelCountResponse execute(RedisOperations<K, V> operations) throws DataAccessException {
                                return executeGetSize(operations, finalSuperviseDepartIds);
                            }
                        });
            } else {
                Collection<PoliceIncidentPendingStatisticsDTO> statisticsDTOS =
                        this.policeIncidentsRepository.getPendingTotal(superviseDepartIds);

                AtomicInteger comonCount = new AtomicInteger();
                AtomicInteger moreCount = new AtomicInteger();
                AtomicInteger greatCount = new AtomicInteger();
                AtomicInteger greatsCount = new AtomicInteger();
                if (!CollectionUtils.isEmpty(statisticsDTOS)) {
                    Arrays.stream(PoliceEventLevel.values()).collect(Collectors.toList()).forEach(pel -> {
                        Optional<PoliceIncidentPendingStatisticsDTO> selectedPendingStatisticsOptional =
                                statisticsDTOS.stream().filter(ii -> ii.getLevel() == pel).findFirst();

                        switch (pel) {
                            case More -> {
                                if (selectedPendingStatisticsOptional.isPresent()) {
                                    moreCount.set(selectedPendingStatisticsOptional.get().getTotalCount());
                                }
                            }
                            case Comon -> {
                                if (selectedPendingStatisticsOptional.isPresent()) {
                                    comonCount.set(selectedPendingStatisticsOptional.get().getTotalCount());
                                }
                            }
                            case Great -> {
                                if (selectedPendingStatisticsOptional.isPresent()) {
                                    greatCount.set(selectedPendingStatisticsOptional.get().getTotalCount());
                                }
                            }
                            case Greats -> {
                                if (selectedPendingStatisticsOptional.isPresent()) {
                                    greatsCount.set(selectedPendingStatisticsOptional.get().getTotalCount());
                                }
                            }
                        }
                    });

                    countResponse = new PoliceIncidentsLevelCountResponse(comonCount.get(),
                            moreCount.get(), greatCount.get(), greatsCount.get(), false);
                }
            }

            return countResponse;
        }

        throw new ForbidTenantException("非监管调度台; 禁止调用警情上报的查询接口");
    }

    @Override
    public void delete(Long id) {
        this.policeIncidentsService.delete(id);
    }

    @Override
    public  Collection<PoliceIncidentRegionReportResponse> getReport(PoliceIncidentRegionReportRequest request) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MONTH, 1);
        calendar.set(Calendar.DAY_OF_MONTH, 1);

        Collection<PoliceIncidentRegionReportDTO> reports =
                this.policeIncidentsRepository.getReport(PoliceIncidentRegionReportCriteria.create(
                        request.getRegionName(),
                        calendar.getTime()
                ));

        return reports.stream().map(ii -> PoliceIncidentRegionReportResponse.create(
                ii.getAmount(),
                ii.getYearMonth()
        )).collect(Collectors.toList());
    }

    private PoliceIncidentsLevelCountResponse executeGetSize(RedisOperations operations, Collection<String> superviseDepartCodes) {
        int commonCount = 0;
        for (String commonKey : RedisConstantUtils.getPoliceIncidentsSuperviseDepartSetKeys(PoliceEventLevel.Comon, superviseDepartCodes)) {
            commonCount += operations.opsForSet().size(commonKey);
        }

        int greatCount = 0;
        for (String greatKey : RedisConstantUtils.getPoliceIncidentsSuperviseDepartSetKeys(PoliceEventLevel.Great, superviseDepartCodes)) {
            greatCount += operations.opsForSet().size(greatKey);
        }

        int moreCount = 0;
        for (String moreKey : RedisConstantUtils.getPoliceIncidentsSuperviseDepartSetKeys(PoliceEventLevel.More, superviseDepartCodes)) {
            moreCount += operations.opsForSet().size(moreKey);
        }

        int greatsCount = 0;
        for (String greatsKey : RedisConstantUtils.getPoliceIncidentsSuperviseDepartSetKeys(PoliceEventLevel.Greats, superviseDepartCodes)) {
            greatsCount += operations.opsForSet().size(greatsKey);
        }

        return PoliceIncidentsLevelCountResponse.create(
                commonCount,
                greatCount,
                moreCount,
                greatsCount, true);
    }
}
