package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdExamSiteDocument;
import com.bcxin.tenant.open.document.domains.repositories.RdExamSiteDocumentRepository;
import com.bcxin.tenant.open.domains.entities.RdExamSiteEntity;
import com.bcxin.tenant.open.domains.repositories.RdExamSiteRepository;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.utils.DateUtils;
import com.bcxin.tenant.open.jdks.ExamSiteWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.ExamSiteRoomSearchRequest;
import com.bcxin.tenant.open.jdks.responses.ExamSiteRoomOverviewResponse;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.function.Function;
import java.util.stream.Collectors;

@DubboService
public class ExamSiteRpcProviderImpl implements ExamSiteWriterRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(ExamSiteRpcProviderImpl.class);
    private static final int PAGE_SIZE = 500;
    private final RdExamSiteRepository rdExamSiteRepository;
    private final RdExamSiteDocumentRepository rdExamSiteDocumentRepository;
    private final DataTranslate dataTranslate;


    public ExamSiteRpcProviderImpl(RdExamSiteRepository rdExamSiteRepository,
                                   RdExamSiteDocumentRepository rdExamSiteDocumentRepository,
                                   DataTranslate dataTranslate) {
        this.rdExamSiteRepository = rdExamSiteRepository;
        this.rdExamSiteDocumentRepository = rdExamSiteDocumentRepository;
        this.dataTranslate = dataTranslate;
    }

    @Override
    public int flush2Redis(Collection<String> ids) {

        return this.doExecuteFlush(pageIndex -> {
            return this.rdExamSiteRepository.getByPage(ids, pageIndex, PAGE_SIZE);
        }, ids);
    }

    private int doExecuteFlush(Function<Integer, Collection<RdExamSiteEntity>> extractExamSiteRooms, Collection<String> ids) {
        int pageIndex = 1;
        int totalCount = 0;

        Collection<RdExamSiteEntity> examSiteEntities = extractExamSiteRooms.apply(pageIndex);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }

        while (!CollectionUtils.isEmpty(examSiteEntities)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = examSiteEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<RdExamSiteDocument> documents =
                    this.dataTranslate.translate2ExamSites(examSiteEntities);

            if (CollectionUtils.isEmpty(documents)) {
                return totalCount;
            }


            Date now = DateUtils.getToday();
            Collection<RdExamSiteDocument> deletedDocuments
                    = documents.stream()
                    .filter(ii -> !ii.isStatus()).collect(Collectors.toList());

            Collection<RdExamSiteDocument> updatedDocuments
                    = documents.stream().filter(ii -> ii.isStatus()).collect(Collectors.toList());

            if (deletedDocuments != null && !deletedDocuments.isEmpty()) {
                this.rdExamSiteDocumentRepository.deleteAll(deletedDocuments);
                logger.error("删除如下驻勤内容:{};",
                        deletedDocuments.stream()
                                .map(ix -> ix.getId()).collect(Collectors.joining(";"))
                );
            }

            this.rdExamSiteDocumentRepository.saveAll(updatedDocuments);

            totalCount += documents.size();
            pageIndex++;
            if (documents.size() < PAGE_SIZE) {
                break;
            }

            examSiteEntities = extractExamSiteRooms.apply(pageIndex);

        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.rdExamSiteDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }
}
