package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdEmployeeDocument;
import com.bcxin.tenant.open.document.domains.documents.RdEmployeeDocument$;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdEmployeeDocumentRepository;
import com.bcxin.tenant.open.domains.entities.RdEmployeeEntity;
import com.bcxin.tenant.open.domains.entities.TenantEmployeeAttendanceEntity;
import com.bcxin.tenant.open.domains.readers.RdCompositedReader;
import com.bcxin.tenant.open.domains.repositories.AttendanceRepository;
import com.bcxin.tenant.open.domains.repositories.RdEmployeeRepository;
import com.bcxin.tenant.open.domains.services.RdEmployeeService;
import com.bcxin.tenant.open.domains.services.commands.BatchUpdateEmployeeLocationCommand;
import com.bcxin.tenant.open.domains.services.commands.CleanNonUsedEmployeeCommand;
import com.bcxin.tenant.open.domains.services.commands.CreateSyncEmployeeCommand;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.enums.EmploymentStatus;
import com.bcxin.tenant.open.jdks.CompanyWriterRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeWriterRpcProvider;
import com.bcxin.tenant.open.jdks.HotCacheRpcProvider;
import com.bcxin.tenant.open.jdks.requests.EmployeeTrackRequest;
import com.bcxin.tenant.open.jdks.requests.HotCacheRequest;
import com.bcxin.tenant.open.jdks.requests.UpdateEmployeeLocationRequest;
import com.bcxin.tenant.open.jdks.requests.UpdateGeoEmployeeRequest;
import com.bcxin.tenant.open.jdks.responses.EmployeeTrackResponse;
import com.redis.om.spring.search.stream.EntityStream;
import com.redis.om.spring.search.stream.SearchStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@DubboService
public class EmployeeWriterRpcProviderImpl implements EmployeeWriterRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(EmployeeWriterRpcProviderImpl.class);
    private final RdEmployeeDocumentRepository employeeDocumentRepository;
    private final RdEmployeeRepository rdEmployeeRepository;
    private final DataTranslate dataTranslate;
    private final AttendanceRepository attendanceRepository;
    private final CompanyWriterRpcProvider companyWriterRpcProvider;
    private final RdCompositedReader compositedReader;
    private final RdEmployeeService employeeService;
    private final EntityStream entityStream;

    private final HotCacheRpcProvider hotCacheRpcProvider;

    public EmployeeWriterRpcProviderImpl(RdEmployeeDocumentRepository employeeDocumentRepository,
                                         RdEmployeeRepository rdEmployeeRepository,
                                         DataTranslate dataTranslate,
                                         AttendanceRepository attendanceRepository,
                                         RdCompanyDocumentRepository rdCompanyDocumentRepository,
                                         CompanyWriterRpcProvider companyWriterRpcProvider,
                                         RdCompositedReader compositedReader,
                                         RdEmployeeService employeeService,
                                         EntityStream entityStream,
                                         HotCacheRpcProvider hotCacheRpcProvider) {
        this.employeeDocumentRepository = employeeDocumentRepository;
        this.rdEmployeeRepository = rdEmployeeRepository;
        this.dataTranslate = dataTranslate;
        this.attendanceRepository = attendanceRepository;
        this.companyWriterRpcProvider = companyWriterRpcProvider;
        this.compositedReader = compositedReader;
        this.employeeService = employeeService;
        this.entityStream = entityStream;
        this.hotCacheRpcProvider = hotCacheRpcProvider;
    }

    @Override
    public int flush2Redis(Collection<String> ids, boolean clearRails) {
        return flush2Redis(ids, false, clearRails);
    }

    @Override
    public int flush2RedisByTenantUserIds(Collection<String> tenantUserIds) {
        if (CollectionUtils.isEmpty(tenantUserIds)) {
            return 0;
        }
        String[] tenantIdsArray = tenantUserIds.toArray(new String[tenantUserIds.size()]);

        SearchStream<RdEmployeeDocument> employeeDocumentSearchStream =
                this.entityStream.of(RdEmployeeDocument.class)
                        .filter(RdEmployeeDocument$.TENANT_USER_ID.in(tenantIdsArray));

        Collection<RdEmployeeDocument> selectedDocuments
                = employeeDocumentSearchStream
                .collect(Collectors.toList());
        /**
         * 当无数据的时候, 不要同步,避免刷新整张表到Redis
         */
        if (CollectionUtils.isEmpty(selectedDocuments)) {
            logger.error("非预期: 无法正常刷新到人员(tenant_user_ids={})的职员信息", tenantUserIds.stream().collect(Collectors.joining(";")));
            return 0;
        }

        Collection<String> employeeIds = selectedDocuments.stream()
                .map(ix -> ix.getTenantEmployeeId())
                .collect(Collectors.toList());

        return flush2Redis(employeeIds, false);
    }

    @Override
    public int flushStation2Redis(Collection<String> ids) {
        return flush2Redis(ids, true);
    }

    @Override
    public void update(UpdateGeoEmployeeRequest request) {

    }

    @Override
    public void flush2DbRedis(String content) {
        this.employeeService.dispatch(CreateSyncEmployeeCommand.create(content));
    }

    @Override
    public Collection<String> getStationIdsByEmployeeIds(Collection<String> employeeIds) {
        if (CollectionUtils.isEmpty(employeeIds)) {
            return Collections.emptyList();
        }

        String[] employeeIdsArray = employeeIds.toArray(new String[employeeIds.size()]);

        Collection<RdEmployeeDocument> documents
                = this.entityStream.of(RdEmployeeDocument.class)
                .filter(RdEmployeeDocument$.ID.in(employeeIdsArray))
                .collect(Collectors.toList());
        Collection<String> notFoundEmployees =
                employeeIds.stream().filter(ix ->
                                !documents.stream().anyMatch(ii -> ii.getTenantEmployeeId().equalsIgnoreCase(ix)))
                        .collect(Collectors.toList());

        Collection<String> stationIds = documents.stream()
                .map(ix -> ix.getSecurityStationId())
                .distinct()
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(notFoundEmployees)) {
            Collection<RdEmployeeEntity> employees =
                    rdEmployeeRepository.getAllByNoPkIds(notFoundEmployees);
            if (!CollectionUtils.isEmpty(employees)) {
                stationIds.addAll(employees.stream().map(ii -> ii.getSecurityStationId())
                        .filter(ix -> StringUtils.hasLength(ix))
                        .distinct().collect(Collectors.toList()));
            }
        }

        return stationIds;
    }

    @Override
    public Collection<EmployeeTrackResponse> getTrackLocations(EmployeeTrackRequest request) {
        Collection<TenantEmployeeAttendanceEntity> attendances =
                attendanceRepository.getEmployeeAttendances(request.getEmployeeId(), request.getBeginDate(), request.getEndDate());

        Collection<EmployeeTrackResponse> responses = attendances.stream().map(ix ->
                        EmployeeTrackResponse.create(ix.getLonLat().getLat(),
                                ix.getLonLat().getLon(),
                                ix.getCreatedTime()
                        )).sorted((ix, ir) -> ix.getCreatedTime().compareTo(ir.getCreatedTime()))
                .collect(Collectors.toList());

        return responses;
    }

    @Override
    public Collection<String> getCheckingEmployeeIds(int pageIndex, int pageSize) {
        return this.rdEmployeeRepository.getCheckingEmployeeIds(pageIndex, pageSize);
    }

    @Override
    public int clearNonUsedEmployees(int pageSize) {
        Collection<String> empIds = this.rdEmployeeRepository.getNonUsedRecords(pageSize);

        return clearNonUsedEmployees(empIds);
    }

    @Override
    public int clearNonUsedEmployees(Collection<String> ids) {
        this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Employee, ids));
        return employeeService.dispatch(CleanNonUsedEmployeeCommand.create(ids));
    }

    @Override
    public void batchUpdateLocations(UpdateEmployeeLocationRequest request) {
        BatchUpdateEmployeeLocationCommand command = BatchUpdateEmployeeLocationCommand.create(
                request.getLocations().stream().map(ii -> BatchUpdateEmployeeLocationCommand.EmployeeLocation.create(
                        ii.getId(),
                        ii.getSignInType(),
                        ii.getLon(),
                        ii.getLat()
                )).collect(Collectors.toList())
        );

        employeeService.dispatch(command);
    }

    private int flush2Redis(Collection<String> ids, boolean forceFlushStation, boolean clearRails) {
        int pageIndex = 1;
        int pageSize = 5000;

        int totalCount = 0;
        Collection<RdEmployeeEntity> employees = this.rdEmployeeRepository.getByPage(ids, pageIndex, pageSize);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }
        while (!CollectionUtils.isEmpty(employees)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = employees.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<RdEmployeeDocument> documents =
                    this.dataTranslate.translate(employees,
                            forceFlushStation,
                            employeeDocumentRepository,
                            attendanceRepository,
                            this.compositedReader);
            if (CollectionUtils.isEmpty(documents)) {
                return totalCount;
            }

            Collection<RdEmployeeDocument> deletedDocuments
                    = documents.stream().filter(ix -> ix.getStatus() == EmploymentStatus.OffJob)
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(deletedDocuments)) {
                this.employeeDocumentRepository.deleteAll(deletedDocuments);
            }

            Collection<RdEmployeeDocument> noDeletedDocuments
                    = documents.stream().filter(ix -> ix.getStatus() != EmploymentStatus.OffJob)
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(noDeletedDocuments)) {
                if (clearRails) {
                    noDeletedDocuments.forEach(dc -> {
                        dc.setSecurityStationRailIds(new HashSet<>());
                    });
                }

                this.employeeDocumentRepository.saveAll(noDeletedDocuments);
            }

            logger.info("完成刷新保安人员数据到RedisJson:pageIndex={}; count={};", pageIndex, documents.size());

            pageIndex++;

            if (employees.size() < pageSize) {
                break;
            }

            employees = this.rdEmployeeRepository.getByPage(ids, pageIndex, pageSize);
            totalCount += documents.size();
        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.employeeDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }
}
