package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.domains.criterias.DispatchLogDetailSearchCriteria;
import com.bcxin.tenant.open.domains.criterias.DispatchLogSearchCriteria;
import com.bcxin.tenant.open.domains.criterias.MyDispatchLogRecordCriteria;
import com.bcxin.tenant.open.domains.criterias.SponsorDispatchLogSearchCriteria;
import com.bcxin.tenant.open.domains.dtos.SponsorDispatchLogDTO;
import com.bcxin.tenant.open.domains.entities.DeviceCommunicatedLogDetailEntity;
import com.bcxin.tenant.open.domains.entities.DeviceCommunicatedLogEntity;
import com.bcxin.tenant.open.domains.repositories.DeviceCommunicatedLogDetailRepository;
import com.bcxin.tenant.open.domains.repositories.DeviceCommunicatedLogRepository;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.repositories.RepositoryBaseAbstract;
import com.bcxin.tenant.open.jdks.DispatchLogRpcProvider;
import com.bcxin.tenant.open.jdks.requests.SponsorDispatchLogSearchRequest;
import com.bcxin.tenant.open.jdks.requests.DispatchLogDetailSearchRequest;
import com.bcxin.tenant.open.jdks.requests.DispatchLogSearchRequest;
import com.bcxin.tenant.open.jdks.requests.MyDispatchLogRecordRequest;
import com.bcxin.tenant.open.jdks.responses.SponsorDispatchLogSearchResponse;
import com.bcxin.tenant.open.jdks.responses.DispatchLogDetailSearchResponse;
import com.bcxin.tenant.open.jdks.responses.DispatchLogSearchResponse;
import com.bcxin.tenant.open.jdks.responses.MyDispatchLogRecordResponse;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@DubboService
public class DispatchLogRpcProviderImpl extends RepositoryBaseAbstract implements DispatchLogRpcProvider {
    private final DeviceCommunicatedLogRepository communicatedLogRepository;
    private final DeviceCommunicatedLogDetailRepository deviceCommunicatedLogDetailRepository;

    public DispatchLogRpcProviderImpl(DeviceCommunicatedLogRepository communicatedLogRepository,
                                      DeviceCommunicatedLogDetailRepository deviceCommunicatedLogDetailRepository) {
        this.communicatedLogRepository = communicatedLogRepository;
        this.deviceCommunicatedLogDetailRepository = deviceCommunicatedLogDetailRepository;
    }

    @Override
    public EntityCollection<DispatchLogSearchResponse> search(DispatchLogSearchRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = this.getCurrentUser();

        Collection<String> callerDeviceNos = new ArrayList<>();
        if(!userModel.isSuperviseDepartRole()) {
            if (!CollectionUtils.isEmpty(userModel.getDeviceDeskServiceScopes())) {
                callerDeviceNos = userModel.getDeviceDeskServiceScopes()
                        .stream().filter(ix -> StringUtils.hasLength(ix.getNumber()))
                        .map(ix -> ix.getNumber())
                        .collect(Collectors.toList());
            }
        }

        Collection<String> superviseDepartIds =
                userModel.getMatchedSuperviseDepartIds(request.getSuperviseDepartIds());

        DispatchLogSearchCriteria criteria =
                DispatchLogSearchCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getDeskTypes(),
                        superviseDepartIds,
                        request.getCallerName(), request.getCalledCompanyName(),
                        request.getDispatchedDateFrom(), request.getDispatchedDateTo(),
                        callerDeviceNos,
                        request.getProjectName());

        criteria.setForExport(request.isForExport());
        EntityCollection<DeviceCommunicatedLogEntity> communicatedLogs =
                this.communicatedLogRepository.search(
                        criteria
                );

        Collection<DispatchLogSearchResponse> searchResponses =
                communicatedLogs.getData().stream().map(ii -> {
                    return DispatchLogSearchResponse.create(
                            ii.getPkId(),
                            ii.getId(),
                            ii.getCallerDeviceNo(),
                            ii.getCallerCompanyName(),
                            ii.getDomainId(),
                            ii.getBeginTime(),
                            ii.getEndTime(), ii.getCommunicatedType(),
                            ii.getCalledDomainId(),
                            ii.getCalledCompanyName(),
                            ii.getCreatedTime(),
                            ii.getDispatchTimeInMinutes(),
                            ii.getPoints(),
                            ii.isPaidByCaller(),
                            ii.getProjectId(),
                            ii.getProjectName()
                    );
                }).collect(Collectors.toList());

        return EntityCollection.create(searchResponses, request.getPageSize(), communicatedLogs.getTotalCount());
    }

    @Override
    public EntityCollection<DispatchLogDetailSearchResponse> search(DispatchLogDetailSearchRequest request) {
        DispatchLogDetailSearchCriteria criteria
                = new DispatchLogDetailSearchCriteria();
        criteria.setLogId(request.getLogId());
        criteria.setCalledOrganizationId(request.getCalledOrganizationId());

        EntityCollection<DispatchLogDetailSearchResponse> communicatedLogDetails =
                this.queryByDispatchLogDetailSearchCriteria(criteria);

        return communicatedLogDetails;
    }

    @Override
    public Collection<MyDispatchLogRecordResponse> search(MyDispatchLogRecordRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        Collection<DeviceCommunicatedLogDetailEntity> detailEntities =
                this.deviceCommunicatedLogDetailRepository.search(
                        MyDispatchLogRecordCriteria.create(userModel.getEmployeeId(), request.getBeginDate(), request.getEndDate())
                );

        return detailEntities.stream().map(ii -> MyDispatchLogRecordResponse.create(ii.getId(),
                        ii.getCallerUserName(), ii.getCallerCompanyName(), ii.getCommunicatedType(), ii.getBeginTime(), ii.getEndTime(), ii.getDispatchTimeInMinutes(),
                        ii.getCreatedTime()))
                .collect(Collectors.toList());
    }

    @Override
    public Collection<SponsorDispatchLogSearchResponse> search(SponsorDispatchLogSearchRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }
        SponsorDispatchLogSearchCriteria criteria =
                SponsorDispatchLogSearchCriteria.create(
                        Stream.of(userModel.getEmployeeId()).collect(Collectors.toList()),
                        request.getDateFrom(),
                        request.getDateTo()
                );
        Collection<SponsorDispatchLogDTO> communicatedLogs =
                this.communicatedLogRepository.search(
                        criteria
                );

        return communicatedLogs.stream().map(ii -> SponsorDispatchLogSearchResponse.create(
                String.valueOf(ii.getPkId()),
                ii.getId(),
                ii.getBeginTime(),
                ii.getEndTime(),
                ii.getCommunicatedType(),
                ii.getCallerCompanyName(),
                ii.getCallerUserName()
        )).collect(Collectors.toList());
    }

    @Override
    public Collection<DispatchLogDetailSearchResponse> getById(String id) {
        DispatchLogDetailSearchCriteria criteria = new DispatchLogDetailSearchCriteria();
        criteria.setLogId(id);
        criteria.setIgnoreCount(true);
        EntityCollection<DeviceCommunicatedLogDetailEntity> detailEntities = this.deviceCommunicatedLogDetailRepository.search(criteria);

        return detailEntities.getData().stream().map(ii -> DispatchLogDetailSearchResponse.create(
                ii.getPkId(),
                ii.getCalledUserName(),
                ii.getCalledDeviceNo(),
                ii.getBeginTime(),
                ii.getEndTime(),
                ii.getCommunicatedType(),
                ii.getDispatchTimeInMinutes(),
                ii.getPoints(),
                ii.getCalledCompanyName()
        )).collect(Collectors.toList());
    }

    private EntityCollection<DispatchLogDetailSearchResponse> queryByDispatchLogDetailSearchCriteria(
            DispatchLogDetailSearchCriteria criteria
    ) {
        EntityCollection<DeviceCommunicatedLogDetailEntity> communicatedLogDetails =
                this.deviceCommunicatedLogDetailRepository.search(criteria);

        Collection<DispatchLogDetailSearchResponse> searchResponses =
                communicatedLogDetails.getData().stream().map(ii -> {
                    return DispatchLogDetailSearchResponse.create(
                            ii.getPkId(),
                            ii.getCalledUserName(),
                            ii.getCalledDeviceNo(),
                            ii.getBeginTime(), ii.getEndTime(),
                            ii.getCommunicatedType(),
                            ii.getDispatchTimeInMinutes(),
                            ii.getPoints(),
                            ii.getCalledCompanyName());
                }).collect(Collectors.toList());

        return EntityCollection.create(searchResponses, criteria.getPageSize(), communicatedLogDetails.getTotalCount());
    }
}
