package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.*;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdDispatchDataScopeDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdEmployeeDocumentRepository;
import com.bcxin.tenant.open.document.domains.repositories.RdSecurityStationDocumentRepository;
import com.bcxin.tenant.open.domains.entities.RdDispatchDataScopeEntity;
import com.bcxin.tenant.open.domains.repositories.RdDispatchDataScopeRepository;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataScopeType;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.DispatchDataScopeRpcProvider;
import com.redis.om.spring.search.stream.EntityStream;
import com.redis.om.spring.search.stream.SearchStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@DubboService
public class DispatchDataScopeRpcProviderImpl implements DispatchDataScopeRpcProvider {
    private final RdDispatchDataScopeRepository dispatchDataScopeRepository;
    private final RdDispatchDataScopeDocumentRepository dispatchDataScopeDocumentRepository;

    private final RdSecurityStationDocumentRepository rdSecurityStationDocumentRepository;
    private final RdEmployeeDocumentRepository rdEmployeeDocumentRepository;
    private final RdCompanyDocumentRepository rdCompanyDocumentRepository;
    private final EntityStream entityStream;
    private final JsonProvider jsonProvider;

    public DispatchDataScopeRpcProviderImpl(RdDispatchDataScopeRepository dispatchDataScopeRepository,
                                            RdDispatchDataScopeDocumentRepository dispatchDataScopeDocumentRepository,
                                            RdSecurityStationDocumentRepository rdSecurityStationDocumentRepository,
                                            RdEmployeeDocumentRepository rdEmployeeDocumentRepository,
                                            RdCompanyDocumentRepository rdCompanyDocumentRepository,
                                            EntityStream entityStream, JsonProvider jsonProvider) {
        this.dispatchDataScopeRepository = dispatchDataScopeRepository;
        this.dispatchDataScopeDocumentRepository = dispatchDataScopeDocumentRepository;
        this.rdSecurityStationDocumentRepository = rdSecurityStationDocumentRepository;
        this.rdEmployeeDocumentRepository = rdEmployeeDocumentRepository;
        this.rdCompanyDocumentRepository = rdCompanyDocumentRepository;
        this.entityStream = entityStream;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public void flush2Redis(Collection<String> ids,DispatchDataScopeType scopeType) {
        int pageIndex = 1;
        int pageSize = 500;
        Collection<RdDispatchDataScopeEntity> dataScopeEntities =
                this.dispatchDataScopeRepository.getByPage(scopeType, ids, pageIndex, pageSize);
        Collection<String> extractIds = new ArrayList<>();
        while (!CollectionUtils.isEmpty(dataScopeEntities)) {
            Collection<RdDispatchDataScopeDocument> documents = new ArrayList<>();

            /**
             * 取出Redis JSON上的RdDispatchDataScopeDocument对象
             */
            Collection<RdDispatchDataScopeDocument> dataScopeDocuments = new HashSet<>();
            if (scopeType == DispatchDataScopeType.Proprietor) {
                Collection<String> pageIds =
                        dataScopeEntities.stream().map(ii -> ii.getId())
                                .collect(Collectors.toList());

                dataScopeDocuments =
                        this.dispatchDataScopeDocumentRepository.findAllById(pageIds);
            }

            /**
             * 必须设置为空
             */
            Collection<RdCompanyDocument> companyDocuments = new HashSet<>();
            if(scopeType==DispatchDataScopeType.Proprietor) {
                Set<String> organizationIds =
                        dataScopeEntities.stream()
                                .filter(ii -> ii.getScopeType() == DispatchDataScopeType.Proprietor)
                                .filter(ii -> !StringUtil.isEmpty(ii.getScope()))
                                .flatMap(ix ->
                                        Arrays.stream(ix.getScope().split(";|,"))
                                                .collect(Collectors.toList()).stream())
                                .collect(Collectors.toSet());

                Set<String> originalSelectedOrgIds =
                        dataScopeDocuments.stream()
                        .filter(ii -> ii.getScopes() != null && ii.getScopeType() == DispatchDataScopeType.Proprietor)
                        .flatMap(ii -> ii.getScopes().stream())
                        .collect(Collectors.toSet());
                if (!CollectionUtils.isEmpty(originalSelectedOrgIds)) {
                    organizationIds.addAll(originalSelectedOrgIds);
                }

                if (!CollectionUtils.isEmpty(organizationIds)) {
                    companyDocuments = this.rdCompanyDocumentRepository.findAllById(organizationIds);
                }
            }

            for (RdDispatchDataScopeEntity scope : dataScopeEntities) {
                Collection<String> originalScopes = new ArrayList<>();
                RdDispatchDataScopeDocument document
                        = dataScopeDocuments.stream()
                        .filter(ii -> StringUtil.isEqual(ii.getId(), scope.getId()))
                        .findFirst()
                        .orElse(null);
                if (document == null) {
                    document = RdDispatchDataScopeDocument.create(scope.getId(), scope.getScopeType());
                }

                String[] additionalScopes = null;
                if (StringUtils.hasLength(scope.getScope())) {
                    /**
                     * 企业(集团)的权限
                     */
                    if(scope.getScopeType()==DispatchDataScopeType.Company) {
                        Collection<String> comPermissions = this.jsonProvider.toObjects(String.class, scope.getScope());

                        additionalScopes = comPermissions.toArray(ii->new String[ii]);
                    }else {
                        additionalScopes = scope.getScope().split(";|,");
                    }

                    document.assignScopes(additionalScopes);
                    documents.add(document);
                } else {
                    document.assignScopes(null);
                }

                extractIds.add(document.getId());
                /**
                 * 针对内保单位; 我们需要调整驻勤点和对应人员的调度范围
                 * Id=驻勤点Id; scope为DOMAINID以;分割的列表
                 */
                if (scope.getScopeType() == DispatchDataScopeType.Proprietor) {
                    refreshProprietorDataSource(
                            scopeType,
                            companyDocuments,
                            document,
                            originalScopes,
                            additionalScopes,
                            extractIds);
                }
            }

            this.dispatchDataScopeDocumentRepository.saveAll(documents);
            pageIndex++;

            if (dataScopeEntities.size() >= pageSize) {
                dataScopeEntities =
                        this.dispatchDataScopeRepository.getByPage(scopeType, ids, pageIndex, pageSize);
            } else {
                dataScopeEntities.clear();
            }
        }

        if (!CollectionUtils.isEmpty(ids)) {
            Collection<String> deleteIds =
                    ids.stream().filter(ii -> !extractIds.contains(ii)).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(deleteIds)) {
                this.dispatchDataScopeDocumentRepository.deleteAllById(deleteIds);
            }
        }
    }

    @Override
    public Collection<String> getJsonResult(Collection<String> ids) {
        SearchStream<RdDispatchDataScopeDocument> dispatchDataScopeDocumentSearchStream =
                this.entityStream.of(RdDispatchDataScopeDocument.class);
        if (!CollectionUtils.isEmpty(ids)) {
            String[] selectedIds = ids.toArray(ix -> new String[ix]);

            dispatchDataScopeDocumentSearchStream =
                    dispatchDataScopeDocumentSearchStream.filter(RdDispatchDataScopeDocument$.ID.in(selectedIds));
        }

        return dispatchDataScopeDocumentSearchStream.collect(Collectors.toList())
                .stream().map(ix -> this.jsonProvider.getJson(ix)).collect(Collectors.toList());
    }

    @Override
    public Collection<String> getCompanyDataScopeById(String companyId) {
        Collection<RdDispatchDataScopeDocument> dataScopeDocuments =
                this.entityStream.of(RdDispatchDataScopeDocument.class)
                        .filter(RdDispatchDataScopeDocument$.ID.eq(companyId))
                        .collect(Collectors.toList());

        return dataScopeDocuments.stream().filter(ii -> ii != null)
                .flatMap(ii -> ii.getScopes().stream())
                .collect(Collectors.toSet());
    }

    @Override
    public Collection<String> getDataScopeById(String id, DispatchDataScopeType scopeType) {
        Collection<RdDispatchDataScopeEntity> dataScopeEntities =
                this.dispatchDataScopeRepository
                        .getByPage(scopeType,
                                Collections.singleton(String.format("%s_%s",scopeType.ordinal(),id)), 0, 10);

        Collection<String> permissionScopes =
                dataScopeEntities.stream().flatMap(ii -> {
                    Collection<String> scopes = new ArrayList<>();
                    if (StringUtils.hasLength(ii.getScope())) {
                        scopes = Arrays.stream(ii.getScope().split(";|,")).toList();
                    }

                    return scopes.stream();
                }).filter(ii -> StringUtils.hasLength(ii)).collect(Collectors.toList());

        return permissionScopes;
    }

    private String[] getProprietorOrgDepartIds(Collection<RdCompanyDocument> documents,
                                               String[] organizationIds) {

        if (organizationIds == null || organizationIds.length == 0) {
            return new String[0];
        }

        /**
         * 加入内保单位的监管归属列表
         */
        Collection<String> finalAdditionalScopes = Arrays.stream(organizationIds).toList();
        String[] orgScopePermissions
                = documents.stream()
                .filter(ii -> finalAdditionalScopes.contains(ii.getId()) &&
                        !StringUtil.isEmpty(ii.getSuperviseDepartId()))
                .map(ii -> ii.getSuperviseDepartId())
                .toArray(ix -> new String[ix]);

        return orgScopePermissions;
    }

    private Collection<RdEmployeeDocument> findEmployeeDocumentsByStationIds(Collection<String> stationIds) {
        if (CollectionUtils.isEmpty(stationIds)) {
            return Collections.EMPTY_LIST;
        }

        String[] sIds = stationIds.toArray(ix -> new String[ix]);

        return
                this.entityStream.of(RdEmployeeDocument.class)
                        .filter(RdEmployeeDocument$.SECURITY_STATION_ID.in(sIds))
                        .collect(Collectors.toList());
    }

    /**
     *  更改内保单位管理的驻勤点以及相关驻勤点的人的调度范围
     */
    private void refreshProprietorDataSource(
            DispatchDataScopeType scopeType,
            Collection<RdCompanyDocument> companyDocuments,
            RdDispatchDataScopeDocument document,
                                             Collection<String> originalScopes,
                                             String[] additionalScopes,
                                             Collection<String> extractStationIds) {
        if (scopeType != DispatchDataScopeType.Proprietor || CollectionUtils.isEmpty(extractStationIds)) {
            return;
        }

        if (document.getScopes() != null) {
            for (String sId : document.getScopes()) {
                originalScopes.add(sId);
            }

            Collection<String> originalSuperviseDepartIds =
                    companyDocuments.stream()
                            .filter(ii -> originalScopes.contains(ii.getId()))
                            .map(ii -> ii.getSuperviseDepartId())
                            .distinct().collect(Collectors.toList());
            originalScopes.addAll(originalSuperviseDepartIds);
        }

        /**
         * 加入内保单位的监管归属列表
         */
        String[] orgDepartIds = getProprietorOrgDepartIds(companyDocuments, additionalScopes);
        Collection<RdSecurityStationDocument> selectedSecurityDocuments
                = this.rdSecurityStationDocumentRepository.findAllById(extractStationIds);
        /**
         * 刷新驻勤点列表
         */
        if (!CollectionUtils.isEmpty(selectedSecurityDocuments)) {
            for (RdSecurityStationDocument dd : selectedSecurityDocuments) {
                /**
                 * 删除之前已经存在的数据范围权限
                 */
                if (!CollectionUtils.isEmpty(originalScopes)) {
                    dd.removeBusinessScopePermissions(originalScopes.toArray(ix -> new String[ix]));
                }

                /**
                 * 加入内保的企业列表
                 */
                dd.addScopePermissions(additionalScopes);
                dd.addScopePermissions(orgDepartIds);
                dd.addResourceTypes(ResourceType.ProprietorManaged.name());
            }

            this.rdSecurityStationDocumentRepository.saveAll(selectedSecurityDocuments);
        }

        /**
         * 刷新保安人员
         */
        Collection<RdEmployeeDocument> selectedEmployeeDocuments = findEmployeeDocumentsByStationIds(extractStationIds);

        /**
         * 刷新保安人员
         */
        if (!CollectionUtils.isEmpty(selectedEmployeeDocuments)) {
            for (RdEmployeeDocument dd : selectedEmployeeDocuments) {
                /**
                 * 删除之前已经存在的数据范围权限
                 */
                if (!CollectionUtils.isEmpty(originalScopes)) {
                    dd.removeBusinessScopePermissions(originalScopes.toArray(ix -> new String[ix]));
                }

                /**
                 * 加入内保的列表
                 */
                dd.addScopePermissions(additionalScopes);
                dd.addScopePermissions(orgDepartIds);
                dd.addResourceTypes(ResourceType.ProprietorManaged.name());
            }

            this.rdEmployeeDocumentRepository.saveAll(selectedEmployeeDocuments);
        }
    }
}
