package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdDeviceDocument;
import com.bcxin.tenant.open.document.domains.documents.RdDeviceDocument$;
import com.bcxin.tenant.open.document.domains.repositories.RdDeviceDocumentRepository;
import com.bcxin.tenant.open.domains.dtos.DeviceBatchUpdateDTO;
import com.bcxin.tenant.open.domains.dtos.SyncDataWrapperDTO;
import com.bcxin.tenant.open.domains.entities.RdDeviceEntity;
import com.bcxin.tenant.open.domains.entities.RdDeviceLocationHistoryEntity;
import com.bcxin.tenant.open.domains.repositories.RdDeviceLocationHistoryRepository;
import com.bcxin.tenant.open.domains.repositories.RdDeviceRepository;
import com.bcxin.tenant.open.domains.repositories.RdSyncDataRepository;
import com.bcxin.tenant.open.domains.services.DeviceLocationHistoryService;
import com.bcxin.tenant.open.domains.services.commands.BatchDeviceLocationHistoryCommand;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.jdks.DeviceWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.BatchDeviceLocationRequest;
import com.bcxin.tenant.open.jdks.requests.DeviceSyncRequest;
import com.bcxin.tenant.open.jdks.requests.DeviceTrackRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import com.bcxin.tenant.open.jdks.responses.DeviceTrackResponse;
import com.redis.om.spring.search.stream.EntityStream;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.geo.Point;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

@DubboService
public class DeviceWriterRpcProviderImpl implements DeviceWriterRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(DeviceWriterRpcProviderImpl.class);

    private final RdDeviceRepository rdDeviceRepository;
    private final DataTranslate dataTranslate;
    private final RdDeviceDocumentRepository rdDeviceDocumentRepository;
    private final EntityStream entityStream;

    private final DeviceLocationHistoryService deviceLocationHistoryService;

    private final RdDeviceLocationHistoryRepository deviceLocationHistoryRepository;
    private final JsonProvider jsonProvider;

    private final RdSyncDataRepository syncDataRepository;

    public DeviceWriterRpcProviderImpl(RdDeviceRepository rdDeviceRepository,
                                       DataTranslate dataTranslate,
                                       RdDeviceDocumentRepository rdDeviceDocumentRepository, EntityStream entityStream,
                                       DeviceLocationHistoryService deviceLocationHistoryService,
                                       RdDeviceLocationHistoryRepository deviceLocationHistoryRepository,
                                       JsonProvider jsonProvider, RdSyncDataRepository syncDataRepository) {
        this.rdDeviceRepository = rdDeviceRepository;
        this.dataTranslate = dataTranslate;
        this.rdDeviceDocumentRepository = rdDeviceDocumentRepository;
        this.entityStream = entityStream;
        this.deviceLocationHistoryService = deviceLocationHistoryService;
        this.deviceLocationHistoryRepository = deviceLocationHistoryRepository;
        this.jsonProvider = jsonProvider;
        this.syncDataRepository = syncDataRepository;
    }

    @Override
    public int flush2Redis(Collection<String> ids) {
        return doFlush2Redis(ids);
    }

    @Override
    public void batch(BatchDeviceLocationRequest request) {
        Collection<String> dispatchNos = request.getLocations().stream()
                .map(ii -> ii.getDispatchNo()).
                collect(Collectors.toSet());
        if(!CollectionUtils.isEmpty(dispatchNos)) {
            String[] selectedDispatchNos = dispatchNos.toArray(ix -> new String[ix]);
            Collection<RdDeviceDocument> documents =
                    this.entityStream.of(RdDeviceDocument.class)
                            .filter(RdDeviceDocument$.DISPATCH_NO.in(selectedDispatchNos))
                            .collect(Collectors.toList());

            if (!CollectionUtils.isEmpty(documents)) {
                for (RdDeviceDocument doc : documents) {
                    BatchDeviceLocationRequest.DeviceLocationRequest dlR =
                            request.getLocations().stream().filter(ii -> ii.getDispatchNo().equalsIgnoreCase(doc.getDeviceNo()))
                                    /**
                                     * 采用降序的方式; 取时间最大的值
                                     */
                                    .sorted((t1, t2) -> t1.getCreatedTimeTick() > t2.getCreatedTimeTick() ? -1 : 1)
                                    .findFirst().orElse(null);
                    if (dlR != null) {
                        doc.changeLonLat(new Point(dlR.getLon(), dlR.getLat()));
                    }
                }

                rdDeviceDocumentRepository.saveAll(documents);

                Collection<BatchDeviceLocationHistoryCommand.DeviceLocationHistoryItemCommand>
                        commandItems =
                        request.getLocations().stream()
                                .map(ii -> {
                                    RdDeviceDocument selectedDoc = documents.stream().filter(ix -> ix.getDispatchNo().equalsIgnoreCase(ii.getDispatchNo()))
                                            .findFirst().orElse(null);
                                    if (selectedDoc == null) {
                                        return null;
                                    }
                                    return
                                            BatchDeviceLocationHistoryCommand.DeviceLocationHistoryItemCommand.create(
                                                    selectedDoc.getId(),
                                                    ii.getDispatchNo(),
                                                    ii.getLon(), ii.getLat());
                                }).filter(ii -> ii != null)
                                .collect(Collectors.toSet());
                if (!CollectionUtils.isEmpty(commandItems)) {
                    this.deviceLocationHistoryService.batch(BatchDeviceLocationHistoryCommand.create(commandItems));

                    logger.error("提交经纬度信息到数据库:{}",
                            this.jsonProvider.getJson(commandItems)
                    );
                } else {
                    logger.error("根据deviceNo找不到设备信息:{};docs={}",
                            dispatchNos.stream().collect(Collectors.joining(",")),
                            this.jsonProvider.getJson(documents)
                    );
                }
            } else {
                logger.error("找不到设备信息:{}", dispatchNos.stream().collect(Collectors.joining(",")));
            }
        }
    }

    @Override
    public Collection<DeviceTrackResponse> getTrackLocations(DeviceTrackRequest request) {
        Collection<RdDeviceLocationHistoryEntity> deviceLocationHistories =
                this.deviceLocationHistoryRepository.getLocations(request.getDeviceId(), request.getBeginDate(), request.getEndDate());

        Collection<DeviceTrackResponse> responses = deviceLocationHistories.stream().map(ix ->
                        DeviceTrackResponse.create(ix.getLonLat().getLat(),
                                ix.getLonLat().getLon(),
                                ix.getCreatedTime()
                        )).sorted((ix, ir) -> ix.getCreatedTime().compareTo(ir.getCreatedTime()))
                .collect(Collectors.toList());

        return responses;
    }

    @Override
    public Collection<String> batchSync(Collection<String> ids) {
        this.syncDataRepository.reloadSyncData(SyncDataWrapperDTO.create(ids, DispatchDataType.HareWareDevice, null));

        rdDeviceRepository.getNoReadyDevices(ids);

        return null;
    }

    @Override
    public void batchUpdate(Collection<DeviceSyncRequest> requests) {
        Set<DeviceBatchUpdateDTO.DeviceItemDTO> deviceItems =
                requests.stream()
                        .map(ii -> DeviceBatchUpdateDTO.DeviceItemDTO.create(ii.getDeviceNo(), ii.getDeviceUid()))
                        .collect(Collectors.toSet());

        Collection<String> deviceIds =
                this.rdDeviceRepository.batchUpdate(DeviceBatchUpdateDTO.create(deviceItems));

        this.flush2Redis(deviceIds);
    }

    @Override
    public Collection<String> getDeviceNosByIds(Collection<String> ids) {
        Collection<RdDeviceEntity> devices = this.rdDeviceRepository.getByPage(ids, 0, ids.size());

        return devices.stream().map(ii -> ii.getDeviceNo()).collect(Collectors.toSet());
    }

    private int doFlush2Redis(Collection<String> ids) {
        int pageIndex = 1;
        int pageSize = 5000;

        int totalCount = 0;
        Collection<RdDeviceEntity> devices = this.rdDeviceRepository.getByPage(ids, pageIndex, pageSize);

        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }
        try {
            while (!CollectionUtils.isEmpty(devices)) {
                if (!CollectionUtils.isEmpty(notExistsIds)) {
                    Collection<String> existsIds = devices.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                    /**
                     * 使用Remove会抛null异常
                     */
                    notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
                }

                Collection<RdDeviceDocument> documents =
                        this.dataTranslate.translate2DeviceDocuments(devices);

                if (CollectionUtils.isEmpty(documents)) {
                    return totalCount;
                }

                this.rdDeviceDocumentRepository.saveAll(documents);
                logger.error("完成刷新设备数据到RedisJson:pageIndex={}; count={};", pageIndex, documents.size());

                pageIndex++;

                if (devices.size() < pageSize) {
                    break;
                }

                devices = this.rdDeviceRepository.getByPage(ids, pageIndex, pageSize);
                totalCount += documents.size();
            }
        } finally {
            /**
             * 清除不在需要的文档
             */
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                this.rdDeviceDocumentRepository.deleteAllById(notExistsIds);
            }
        }

        return totalCount;
    }
}
