package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.bcxin.tenant.open.document.domains.documents.RdCompanyDocument;
import com.bcxin.tenant.open.document.domains.dtos.RdCompanyCountOfSecurityManDTO;
import com.bcxin.tenant.open.document.domains.dtos.RdCompanyCountOfStationDTO;
import com.bcxin.tenant.open.document.domains.repositories.RdCompanyDocumentRepository;
import com.bcxin.tenant.open.document.domains.utils.GeoPointUtils;
import com.bcxin.tenant.open.domains.criterias.CompanyAdvanceCriteria;
import com.bcxin.tenant.open.domains.dtos.CompanyAdvanceDTO;
import com.bcxin.tenant.open.domains.entities.RdCompanyEntity;
import com.bcxin.tenant.open.domains.readers.RdCompositedReader;
import com.bcxin.tenant.open.domains.repositories.RdCompanyRepository;
import com.bcxin.tenant.open.domains.services.OrgPurseService;
import com.bcxin.tenant.open.domains.services.RdCompanyService;
import com.bcxin.tenant.open.domains.services.commands.CreateSyncCompanyCommand;
import com.bcxin.tenant.open.domains.services.commands.RefreshOrgPurseCacheCommand;
import com.bcxin.tenant.open.dubbo.writer.providers.translates.DataTranslate;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.utils.LonLatValueTypeUtils;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.infrastructures.valueTypes.LonLatValueType;
import com.bcxin.tenant.open.jdks.CompanyWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.CompanyOverviewRequest;
import com.bcxin.tenant.open.jdks.responses.CompanyAdvanceResponse;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@DubboService
public class CompanyWriterRpcProviderImpl implements CompanyWriterRpcProvider {
    private final static Logger logger = LoggerFactory.getLogger(CompanyWriterRpcProviderImpl.class);

    private final RdCompanyRepository companyRepository;
    private final RdCompanyDocumentRepository companyDocumentRepository;
    private final DataTranslate dataTranslate;
    private final RdCompositedReader compositedReader;
    private final RdCompanyService companyService;
    private final OrgPurseService orgPurseService;


    public CompanyWriterRpcProviderImpl(RdCompanyRepository companyRepository,
                                        RdCompanyDocumentRepository companyDocumentRepository,
                                        DataTranslate dataTranslate,
                                        RdCompositedReader compositedReader,
                                        RdCompanyService companyService, OrgPurseService orgPurseService) {

        this.companyRepository = companyRepository;
        this.companyDocumentRepository = companyDocumentRepository;
        this.dataTranslate = dataTranslate;
        this.compositedReader = compositedReader;
        this.companyService = companyService;
        this.orgPurseService = orgPurseService;
    }

    @Override
    public int flush2Redis(Collection<String> ids, DispatchDataType dataType) {
        int pageIndex = 1;
        int pageSize = 500;
        int totalCount = 0;

        /**
         * 消息队列出发的时候才刷新缓存: 针对ids不为空的情况
         */
        if (dataType == DispatchDataType.RefreshCompanyPoints && !CollectionUtils.isEmpty(ids)) {
            this.orgPurseService.dispatch(RefreshOrgPurseCacheCommand.create(ids,true));
        }

        Collection<RdCompanyEntity> companyEntities = this.companyRepository.getByPage(ids, pageIndex, pageSize);
        Collection<String> notExistsIds = new ArrayList<>();
        if (!CollectionUtils.isEmpty(ids)) {
            notExistsIds = ids.stream().toList();
        }

        while (!CollectionUtils.isEmpty(companyEntities)) {
            if (!CollectionUtils.isEmpty(notExistsIds)) {
                Collection<String> existsIds = companyEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());
                /**
                 * 使用Remove会抛null异常
                 */
                notExistsIds = notExistsIds.stream().filter(ii -> !existsIds.contains(ii)).collect(Collectors.toList());
            }

            Collection<RdCompanyDocument> documents =
                    this.dataTranslate.translate2Company(companyEntities, companyDocumentRepository, this.compositedReader);

            this.companyDocumentRepository.saveAll(documents);
            logger.info("正在刷新企业信息数据到RedisJson:pageIndex={}; count={};", pageIndex, documents.size());

            pageIndex++;
            if (documents.size() < pageSize) {
                break;
            }

            companyEntities = this.companyRepository.getByPage(ids, pageIndex, pageSize);
            totalCount += documents.size();
        }

        /**
         * 清除不在需要的文档
         */
        if (!CollectionUtils.isEmpty(notExistsIds)) {
            this.companyDocumentRepository.deleteAllById(notExistsIds);
        }

        return totalCount;
    }

    @Override
    public int flushSecurityMenCount2Redis(Collection<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return 0;
        }

        Collection<RdCompanyDocument> companyDocuments =
                this.companyDocumentRepository.findAllById(ids);
        if(CollectionUtils.isEmpty(companyDocuments)) {
            return 0;
        }

        Collection<RdCompanyCountOfSecurityManDTO> countOfSecurityManDTOS
                = this.compositedReader.getCountOfSecurityManByOrganizationIds(ids);

        companyDocuments.forEach(com -> {
            Optional<RdCompanyCountOfSecurityManDTO> countOfSecurityManDTOOptional
                    = countOfSecurityManDTOS.stream().filter(ii -> ii.getOrganizationId().equals(com.getOrganizationId()))
                    .findFirst();
            if (countOfSecurityManDTOOptional.isPresent()) {
                com.setCountOfSecurityMan(countOfSecurityManDTOOptional.get().getTotal());
            }
        });

        this.companyDocumentRepository.saveAll(companyDocuments);

        return companyDocuments.size();
    }

    @Override
    public int flushStationCount2Redis(Collection<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return 0;
        }

        Collection<RdCompanyDocument> companyDocuments =
                this.companyDocumentRepository.findAllById(ids);
        if(CollectionUtils.isEmpty(companyDocuments)) {
            return 0;
        }

        Collection<RdCompanyCountOfStationDTO> countOfStationDTOS
                = this.compositedReader.getCountOfStationByOrganizationIds(ids);

        companyDocuments.forEach(com -> {
            Optional<RdCompanyCountOfStationDTO> countOfStationDTOOptional
                    = countOfStationDTOS.stream().filter(ii -> ii.getOrganizationId().equals(com.getOrganizationId()))
                    .findFirst();
            if (countOfStationDTOOptional.isPresent()) {
                com.setCountOfSecurityMan(countOfStationDTOOptional.get().getTotal());
            }
        });

        return companyDocuments.size();
    }

    @Override
    public void flush2DbRedis(String content) {
        this.companyService.dispatch(CreateSyncCompanyCommand.create(content));
    }

    @Override
    public Collection<String> getCompanyIds(int pageIndex, int pageSize) {
        return this.companyRepository.getCompanyIds(pageIndex, pageSize);
    }

    @Override
    public EntityCollection<CompanyAdvanceResponse> search(CompanyOverviewRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }
        Collection<String> organizationIds = null;
        Collection<String> superviseDepartIds = null;
        if (!userModel.isSuperviseDepartRole()) {
            organizationIds = Arrays.stream(userModel.getNoSuperviseScopePermissions()).toList();
        } else {
            superviseDepartIds = Arrays.stream(userModel.getAssignedSuperviseDepartIds()).toList();
            if (!CollectionUtils.isEmpty(request.getSuperviseDepartIds())) {
                superviseDepartIds = superviseDepartIds.stream()
                        .filter(ii -> request.getSuperviseDepartIds().contains(ii)).collect(Collectors.toList());
            }
        }

        CompanyAdvanceCriteria criteria
                = CompanyAdvanceCriteria.create(
                        request.getPageIndex(),
                        request.getPageSize(),
                        request.getName(), request.getHasPoints(), request.getHasDesk(),
                request.getHasLocation(),
                organizationIds,
                superviseDepartIds,
                ResourceType.toResourceTypes(ResourceType.getResources(request.getDeskTypes()))
        );

        EntityCollection<CompanyAdvanceDTO> data =
                this.companyRepository.findByStations(criteria);

        Collection<CompanyAdvanceResponse> items =
                data.getData().stream().map(ii ->
                {
                    boolean hasLonLat = false;
                    Double latitude = null;
                    Double longitude = null;
                    boolean validLonlat = false;
                    if (!StringUtil.isEmpty(ii.getLonLat())) {
                        LonLatValueType lonLatValueType = LonLatValueTypeUtils.translate(ii.getLonLat());
                        if (lonLatValueType != null) {
                            latitude = lonLatValueType.getLat();
                            longitude = lonLatValueType.getLat();

                            if (GeoPointUtils.IsValidGeo(longitude, latitude)) {
                                validLonlat = true;
                            }
                        }

                        hasLonLat = true;
                    }

                    CompanyAdvanceResponse rp =
                            CompanyAdvanceResponse.create(
                                    ii.getId(), ii.getName(),
                                    ii.getLegal(),
                                    ii.getLegalContact(), ii.getPoints(),
                                    ii.getCountOfDesk(),ii.getCountOfProject(),
                                    latitude, longitude,
                                    hasLonLat, validLonlat);
                    rp.assignStationInfo(ii.getCountOfStation(),ii.getCountOfSecurityMan());
                    return rp;
                }
        ).collect(Collectors.toList());

        return EntityCollection.create(items, data.getPageSize(), data.getTotalCount());
    }
}
