package com.bcxin.tenant.open.dubbo.writer.providers.impls;

import com.alibaba.fastjson2.util.DateUtils;
import com.bcxin.tenant.open.document.domains.components.SearchSpecialPermissionScopeComponent;
import com.bcxin.tenant.open.document.domains.dtos.SearchSpecialPermissionScopeDTO;
import com.bcxin.tenant.open.domains.criterias.AttendanceAdvanceCriteria;
import com.bcxin.tenant.open.domains.criterias.AttendanceAdvanceDetailCriteria;
import com.bcxin.tenant.open.domains.criterias.AttendanceCriteria;
import com.bcxin.tenant.open.domains.criterias.MyAttendanceCriteria;
import com.bcxin.tenant.open.domains.dtos.AttendanceCompanyGroupDTO;
import com.bcxin.tenant.open.domains.dtos.AttendanceCompanyGroupDetailDTO;
import com.bcxin.tenant.open.domains.dtos.MyAttendanceRecordDTO;
import com.bcxin.tenant.open.domains.entities.TenantEmployeeAttendanceEntity;
import com.bcxin.tenant.open.domains.entities.valueTypes.GeoPointValueType;
import com.bcxin.tenant.open.domains.repositories.AttendanceRepository;
import com.bcxin.tenant.open.domains.services.AttendanceService;
import com.bcxin.tenant.open.domains.services.commands.BatchAttendanceCommitCommand;
import com.bcxin.tenant.open.domains.services.commands.CleanRecentlyAttendanceCommitCommand;
import com.bcxin.tenant.open.domains.services.commands.CreateAttendanceCommand;
import com.bcxin.tenant.open.domains.services.commands.ValidateAttendanceCommand;
import com.bcxin.tenant.open.domains.services.commands.results.ValidateAttendanceCommandResult;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.ReferenceType;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.AttendanceRpcProvider;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.wrappers.WrappedAttendanceWriterRequest;
import com.bcxin.tenant.open.jdks.responses.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.bcxin.tenant.open.domains.criterias.AttendanceCriteria.SearchModel.*;

/**
 * 警情信息处理
 */
@DubboService
public class AttendanceRpcProviderImpl implements AttendanceRpcProvider {
    private final AttendanceService attendanceService;
    private final AttendanceRepository attendanceRepository;
    private final SearchSpecialPermissionScopeComponent specialPermissionScopeComponent;

    private final JsonProvider jsonProvider;

    public AttendanceRpcProviderImpl(AttendanceService attendanceService,
                                     AttendanceRepository attendanceRepository,
                                     SearchSpecialPermissionScopeComponent specialPermissionScopeComponent, JsonProvider jsonProvider) {
        this.attendanceService = attendanceService;
        this.attendanceRepository = attendanceRepository;
        this.specialPermissionScopeComponent = specialPermissionScopeComponent;
        this.jsonProvider = jsonProvider;
    }

    /**
     * 新增警情
     *
     * @param wrapperRequest
     */
    @Override
    public void create(WrappedAttendanceWriterRequest wrapperRequest) {
        //TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        AttendanceWriterRequest request = wrapperRequest.getOriginalRequest();
        ReferenceType referenceType = request.getReferenceType();

        CreateAttendanceCommand command = CreateAttendanceCommand.create(
                wrapperRequest.getEmployeeId(),
                referenceType,
                wrapperRequest.getReferenceNumber(),
                request.getNote(),
                request.getAddress(), request.getRecordStatus(),
                GeoPointValueType.create(request.getLatitude(), request.getLongitude()),
                request.getAttachments(),
                request.getItems(),
                request.getHeaderOption()
        );

        this.attendanceService.dispatch(command);
    }

    @Override
    public AttendanceValidationResponse validate(AttendanceValidationRequest request) {
        ValidateAttendanceCommand command = ValidateAttendanceCommand.create(
                request.getReferenceType(),
                request.getReferenceNumber()
        );

        try {
            ValidateAttendanceCommandResult result = this.attendanceService.dispatch(command);

            return AttendanceValidationResponse.create(
                    result.getId(),
                    result.getName(), result.getAddress(), result.getLatitude(),
                    result.getLongitude(),
                    result.getPerformRange(), result.isEnableFaceValidation(),
                    result.isIgnoredPerformRangeLimited()
            );
        } catch (NoFoundTenantException ex) {
            throw new NoFoundTenantException(String.format("该职员(%s)所在的驻勤点无效(%s)", request.getEmployeeId(), command.getReferenceNumber()));
        }
    }

    @Override
    public Collection<AttendanceDetailReaderResponse> getByDay(String tenantEmployeeId, String createdDate) {
        if (StringUtils.isEmpty(tenantEmployeeId)) {
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            tenantEmployeeId = userModel.getEmployeeId();
        }
        List<TenantEmployeeAttendanceEntity> attendances = this.attendanceRepository.getByDay(tenantEmployeeId, createdDate);
        if (attendances == null) {
            throw new NoFoundTenantException();
        }
        Collection<AttendanceDetailReaderResponse> list = attendances.stream().map(pie -> {
            return AttendanceDetailReaderResponse.create(DateUtils.format(pie.getCreatedTime(), "yyyy-MM-dd HH:mm:ss"), pie.getRecordStatus().getTypeName(), pie.getReferenceType().getTypeName(), pie.getNote(), pie.getAddress());
        }).collect(Collectors.toList());
        return list;
    }

    @Override
    public EntityCollection<AttendanceEmpReaderResponse> search(AttendanceSearchRequest request) {
        int pageIndex = request.getPageIndex();
        if (pageIndex < 1) {
            pageIndex = 1;
        }
        AttendanceCriteria criteria = null;
        if (request.getSearchModel() == null) {
            throw new BadTenantException("SearchModel不能为空");
        }
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        switch (request.getSearchModel()) {
            case AdvanceSearch -> {
                /*
                Collection<String> superviseDepartIds = Arrays.stream(userModel.getAssignedSuperviseDepartIds()).toList();
                if (!CollectionUtils.isEmpty(request.getSuperviseDepartIds())) {
                    superviseDepartIds = superviseDepartIds.stream().filter(ii -> request.getSuperviseDepartIds().contains(ii)).collect(Collectors.toList());
                }*/

                Collection<String> superviseDepartIds =
                        userModel.getMatchedSuperviseDepartIds(request.getSuperviseDepartIds());
                Collection<String> otherSpecialPermissionScopeIds = new ArrayList<>();
                SearchSpecialPermissionScopeDTO specialPermissionScope =
                        this.specialPermissionScopeComponent.get(userModel);
                if (specialPermissionScope != null && specialPermissionScope.isApplySpecialPermissionScopeRule()) {
                    otherSpecialPermissionScopeIds = specialPermissionScope.getScopeIds();
                }

                Collection<String> relativeOrganizationIds = new HashSet<>();
                if (!userModel.isSuperviseDepartRole() && userModel.getNoSuperviseScopePermissions() != null) {
                    relativeOrganizationIds = Arrays.stream(userModel.getNoSuperviseScopePermissions()).toList();
                }

                Collection<String> managedProjectIds = new HashSet<>();
                if (userModel.getNoSuperviseScopePermissions() != null) {
                    managedProjectIds =
                            Arrays.stream(userModel.getNoSuperviseScopePermissions())
                                    .filter(ii -> BusinessConstants.isRelativeProjectIdStuffId(ii)).toList();

                    if (!CollectionUtils.isEmpty(managedProjectIds)) {
                        /**
                         * 针对赛演临保项目这种复杂的业务规则: 查询本公司 + 参与相关项目的数据 + 其中受限(通过: otherSpecialPermissionScopeIds控制)
                         */
                        relativeOrganizationIds = Stream.of(userModel.getOrganizationId()).collect(Collectors.toSet());
                    }
                }

                criteria = AttendanceCriteria.create(
                        request.getName(),
                        request.getCompanyName(),
                        request.getSecurityStationName(),
                        //userModel.getNoSuperviseScopePermissions(),
                        relativeOrganizationIds.toArray(ix->new String[ix]),
                        userModel.isSuperviseDepartRole(),
                        superviseDepartIds,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(),
                        request.getIdCardNo(),
                        request.getProjectIds(),
                        pageIndex,
                        request.getPageSize(),
                        AdvanceSearch);

                criteria.assignManagedProjectIds(managedProjectIds, otherSpecialPermissionScopeIds);
            }
            case Today -> {
                criteria = AttendanceCriteria.create(
                        request.getName(),
                        request.getCompanyName(),
                        request.getSecurityStationName(),
                        userModel.getNoSuperviseScopePermissions(),
                        userModel.isSuperviseDepartRole(),
                        Collections.EMPTY_LIST,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(),
                        request.getIdCardNo(),
                        request.getProjectIds(),
                        pageIndex,
                        request.getPageSize(),
                        Today);
            }
            case RecentWeek -> {
                criteria = AttendanceCriteria.create(request.getName(),
                        request.getCompanyName(),
                        request.getSecurityStationName(),
                        userModel.getNoSuperviseScopePermissions(),
                        userModel.isSuperviseDepartRole(),
                        Collections.EMPTY_LIST,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(),
                        request.getIdCardNo(),
                        request.getProjectIds(),
                        pageIndex,
                        request.getPageSize(),
                        RecentWeek);
            }
            case RecentMonth -> {
                criteria = AttendanceCriteria.create(
                        request.getName(),
                        request.getCompanyName(),
                        request.getSecurityStationName(),
                        userModel.getNoSuperviseScopePermissions(),
                        userModel.isSuperviseDepartRole(),
                        Collections.EMPTY_LIST,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(),
                        request.getIdCardNo(),
                        request.getProjectIds(),
                        pageIndex,
                        request.getPageSize(),
                        RecentMonth);
            }
        }

        criteria.setForExport(request.isForExport());
        EntityCollection<TenantEmployeeAttendanceEntity> result =
                attendanceRepository.search(criteria);

        Collection<AttendanceEmpReaderResponse> data =
                result.getData().stream().map(ix -> {
                    AttendanceEmpReaderResponse
                            response = AttendanceEmpReaderResponse.create(
                            request.getSearchModel(),
                            DateUtils.format(ix.getCreatedTime(), "yyyy-MM-dd HH:mm:ss"),
                            ix.getRecordStatus(),
                            ix.getAddress(),
                            ix.getName(),
                            ix.getIdCardNo(),
                            ix.getCompanyName(),
                            ix.getSecurityStationName(),
                            ix.getSecurityStationAddress(),
                            ix.getSuperviseDepartName(),
                            ix.getNote(),
                            ix.getAddress(),
                            ix.getProjectName(),
                            this.jsonProvider.toObject(Map.class,ix.getExtendJsonInfo())
                    );
                    return response;
                }).collect(Collectors.toList());


        return EntityCollection.create(data, result.getPageSize(), result.getTotalCount());
    }

    @Override
    public EntityCollection<AttendanceGroupSearchResponse> search(AttendanceGroupSearchRequest request) {
        int pageIndex = request.getPageIndex();
        if (pageIndex < 1) {
            pageIndex = 1;
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        Collection<String> superviseDepartIds =
                userModel.getMatchedSuperviseDepartIds(request.getSuperviseDepartIds());
        Collection<String> otherSpecialPermissionScopeIds = new ArrayList<>();
        SearchSpecialPermissionScopeDTO specialPermissionScope =
                this.specialPermissionScopeComponent.get(userModel);
        if (specialPermissionScope != null && specialPermissionScope.isApplySpecialPermissionScopeRule()) {
            otherSpecialPermissionScopeIds = specialPermissionScope.getScopeIds();
        }

        Collection<String> relativeOrganizationIds = new HashSet<>();
        if(!userModel.isSuperviseDepartRole() && userModel.getNoSuperviseScopePermissions()!=null) {
            relativeOrganizationIds = Arrays.stream(userModel.getNoSuperviseScopePermissions()).toList();
        }

        Collection<String> managedProjectIds = new HashSet<>();
        if(userModel.getNoSuperviseScopePermissions()!=null) {
            managedProjectIds =
                    Arrays.stream(userModel.getNoSuperviseScopePermissions())
                            .filter(ii -> BusinessConstants.isRelativeProjectIdStuffId(ii)).toList();

            if(!CollectionUtils.isEmpty(managedProjectIds)) {
                /**
                 * 针对赛演临保项目这种复杂的业务规则: 查询本公司 + 参与相关项目的数据 + 其中受限(通过: otherSpecialPermissionScopeIds控制)
                 */
                relativeOrganizationIds = Stream.of(userModel.getOrganizationId()).collect(Collectors.toSet());
            }
        }

        AttendanceAdvanceCriteria criteria =
                AttendanceAdvanceCriteria.create(
                        pageIndex, request.getPageSize(),
                        request.getDeskTypes(),
                        request.getName(),
                        request.getCompanyName(),
                        request.getProjectName(),
                        request.getProjectCompanyName(),
                        request.getSecurityStationName(),
                        superviseDepartIds,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(), request.getIdCardNo(),
                        relativeOrganizationIds,
                        request.getHasSecurityCertificateNo(),
                        request.getRecordStatuses(),
                        otherSpecialPermissionScopeIds,
                        ResourceType.getCalculatedResourceValueByDeskTypes(request.getDeskTypes()),
                        managedProjectIds
                );
        EntityCollection<AttendanceCompanyGroupDTO> result = this.attendanceRepository.searchByAdvanceGroup(criteria);

        Collection<AttendanceGroupSearchResponse> items =
                result.getData().stream().map(ii ->
                        AttendanceGroupSearchResponse.create(
                                ii.getEmployeeId(), ii.getName(), ii.getIdCard(), ii.getSecurityCertificateNo(),
                                ii.getOrganizationId(),
                                ii.getCompanyName(),
                                ii.getRecordStatusContactValue(), ii.getCountOfRecords(),
                                ii.getProjectName(),
                                ii.getProjectCompanyName()
                        )
                ).collect(Collectors.toList());

        return EntityCollection.create(items, criteria.getPageSize(), result.getTotalCount());
    }

    @Override
    public EntityCollection<AttendanceGroupDetailSearchResponse> search(AttendanceGroupDetailSearchRequest request) {
        int pageIndex = request.getPageIndex();
        if (pageIndex < 1) {
            pageIndex = 1;
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        Collection<String> superviseDepartIds =
                userModel.getMatchedSuperviseDepartIds(request.getSuperviseDepartIds());

        Collection<String> otherSpecialPermissionScopeIds = new ArrayList<>();
        SearchSpecialPermissionScopeDTO specialPermissionScope =
                this.specialPermissionScopeComponent.get(userModel);
        if (specialPermissionScope != null && specialPermissionScope.isApplySpecialPermissionScopeRule()) {
            otherSpecialPermissionScopeIds = specialPermissionScope.getScopeIds();
        }

        Collection<String> relativeOrganizationIds = new HashSet<>();
        if (!userModel.isSuperviseDepartRole() && userModel.getNoSuperviseScopePermissions() != null) {
            relativeOrganizationIds = Arrays.stream(userModel.getNoSuperviseScopePermissions()).toList();
        }

        Collection<String> managedProjectIds = new HashSet<>();
        if (userModel.getNoSuperviseScopePermissions() != null) {
            managedProjectIds =
                    Arrays.stream(userModel.getNoSuperviseScopePermissions())
                            .filter(ii -> BusinessConstants.isRelativeProjectIdStuffId(ii)).toList();

            if (!CollectionUtils.isEmpty(managedProjectIds)) {
                /**
                 * 针对赛演临保项目这种复杂的业务规则: 查询本公司 + 参与相关项目的数据 + 其中受限(通过: otherSpecialPermissionScopeIds控制)
                 */
                relativeOrganizationIds = Stream.of(userModel.getOrganizationId()).collect(Collectors.toSet());
            }
        }

        AttendanceAdvanceDetailCriteria criteria =
                AttendanceAdvanceDetailCriteria.create(
                        pageIndex,
                        request.getPageSize(),
                        request.getDeskTypes(),
                        request.getProjectName(),
                        request.getProjectCompanyName(),
                        ResourceType.getCalculatedResourceValueByDeskTypes(request.getDeskTypes()),
                        request.getEmployeeId(),
                        request.getName(),
                        request.getCompanyName(), request.getSecurityStationName(),
                        superviseDepartIds,
                        request.getCreatedStarDate(),
                        request.getCreatedEndDate(), request.getIdCardNo(),
                        relativeOrganizationIds,
                        request.getHasSecurityCertificateNo(),
                        request.getRecordStatuses(),
                        otherSpecialPermissionScopeIds,
                        managedProjectIds
                );

        EntityCollection<AttendanceCompanyGroupDetailDTO> result = this.attendanceRepository.searchByAdvanceGroupDetail(criteria);

        return EntityCollection.create(
                result.getData().stream().map(ii -> AttendanceGroupDetailSearchResponse.create(
                        ii.getName(),
                        ii.getIdCard(),
                        ii.getSecurityCertificateNo(),
                        ii.getOrganizationId(),
                        ii.getCompanyName(),
                        ii.getStatus(),
                        ii.getCreatedTime(),
                        ii.getAddress(),
                        ii.getSecurityStationId(),
                        ii.getSecurityDepartId(),
                        ii.getSecurityStationName(),
                        ii.getSecurityDepartName(),
                        ii.getSecurityStationAddress(),
                        ii.getProjectId(),
                        ii.getProjectName(),
                        this.jsonProvider.toObject(Map.class, ii.getExtendJsonInfo()),
                        ii.getProjectCompanyName()
                        )
                ).collect(Collectors.toList()),
                request.getPageSize(), result.getTotalCount());
    }

    @Override
    public void batchCommit(Collection<String> data) {
        this.attendanceService.dispatch(BatchAttendanceCommitCommand.create(data));
    }

    @Override
    public int cleanRecentlyRecords() {
        return this.attendanceService.dispatch(CleanRecentlyAttendanceCommitCommand.create());
    }

    @Override
    public Collection<MyAttendanceRecordResponse> searchMyRecords(MyAttendanceSearchRequest request) {

        Collection<MyAttendanceRecordDTO> records = attendanceRepository.search(MyAttendanceCriteria.create(
                request.getSearchModel(), request.getEmployeeId(),
                request.getPageIndex(),request.getPageSize()));

        return records.stream().map(ii -> MyAttendanceRecordResponse.create(
                ii.getOperatorTime(),
                ii.getOperatorType().getTypeName(),
                ii.getName(),
                ii.getOperatorAddress()
        )).collect(Collectors.toList());
    }

    @Override
    public Collection<AttendanceResponse> search(AttendanceSearchByTypeRequest request) {
        Collection<TenantEmployeeAttendanceEntity> data =
                this.attendanceRepository.getByTypeAndNumber(
                        request.getReferenceType(),
                        request.getReferenceNumber(),
                        request.getPageIndex(), request.getPageSize());

        return data.stream().map(ii -> {
            Collection<String> attachments = new ArrayList<>();
            if (!StringUtils.isBlank(ii.getAttachments())) {
                attachments = this.jsonProvider.toObjects(String.class, ii.getAttachments());
            }

            AttendanceResponse attendanceResponse = AttendanceResponse.create(
                    String.valueOf(ii.getId()),
                    ii.getName(),
                    ii.getCreatedTime(),
                    ii.getRecordStatus(),
                    ii.getReferenceType(),
                    ii.getSecurityStationName(),
                    ii.getNote(),
                    ii.getAddress(),
                    attachments
            );
            return attendanceResponse;
        }).collect(Collectors.toList());
    }

    @Override
    public boolean checkIfEnableFaceValidation() {
        return this.attendanceService.checkIfEnableFaceValidation();
    }
}
