package com.bcxin.tenant.open.dubbo.writer.providers.eventListeners;

import com.bcxin.tenant.open.domains.entities.RollCallEmployeeEntity;
import com.bcxin.tenant.open.domains.events.BatchDispatchRollCallEmployeeEvent;
import com.bcxin.tenant.open.dubbo.writer.providers.components.GeTuiHttpClient;
import com.bcxin.tenant.open.dubbo.writer.providers.components.dtos.GeTuiSharedContentDTO;
import com.bcxin.tenant.open.dubbo.writer.providers.components.dtos.GeTuiSharedContentResponse;
import com.bcxin.tenant.open.dubbo.writer.providers.components.dtos.RollCallRequestDTO;
import com.bcxin.tenant.open.domains.exceptions.GeTuiBadTenantException;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.RollCallType;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

@Component
public class BatchDispatchRollCallEmployeeEventListener implements ApplicationListener<BatchDispatchRollCallEmployeeEvent> {
    private static final Logger logger = LoggerFactory.getLogger(BatchDispatchRollCallEmployeeEventListener.class);
    private final GeTuiHttpClient geTuiHttpClient;
    private final KafkaTemplate kafkaTemplate;
    private final JsonProvider jsonProvider;

    public BatchDispatchRollCallEmployeeEventListener(GeTuiHttpClient geTuiHttpClient,
                                                      KafkaTemplate kafkaTemplate, JsonProvider jsonProvider) {
        this.geTuiHttpClient = geTuiHttpClient;
        this.kafkaTemplate = kafkaTemplate;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public void onApplicationEvent(BatchDispatchRollCallEmployeeEvent event) {
        Collection<RollCallEmployeeEntity> rollCallEmployeeEntities = event.getEmployees();
        if (CollectionUtils.isEmpty(rollCallEmployeeEntities)) {
            return;
        }

        Collection<Long> rollCallIds = rollCallEmployeeEntities.stream()
                .map(ii -> ii.getRollCallId())
                .collect(Collectors.toSet());

        Collection<GeTuiBadTenantException.GeTuiBadDataItem> dataItems = new ArrayList<>();

        for (Long rollCallId : rollCallIds) {
            Collection<String> actionOrganizationIds =
                    rollCallEmployeeEntities.stream().filter(ii -> ii.getRollCallId().equals(rollCallId))
                            .map(ii -> ii.getActionOrganizationId()).collect(Collectors.toSet());
            for (String orgId : actionOrganizationIds) {
                Map<String, Object> paramMap = new HashMap<>();
                paramMap.put("rollCallId", rollCallId);
                paramMap.put("orgId", orgId);
                Collection<String> cIds = new ArrayList<>();
                try {
                    Collection<RollCallEmployeeEntity> selectedRollCallEmployeeEntities =
                            rollCallEmployeeEntities.stream()
                                    .filter(ii -> ii.getRollCallId().equals(rollCallId) && ii.getActionOrganizationId().equalsIgnoreCase(orgId))
                                    .collect(Collectors.toSet());

                    RollCallEmployeeEntity selectedRollCallEmployee =
                            selectedRollCallEmployeeEntities.stream().findFirst().orElse(null);

                    if (selectedRollCallEmployee != null) {
                        String title = BusinessUtil.getRollCallGeTuiTitle(selectedRollCallEmployee.getActionOrganizationName());
                        String content = BusinessUtil.getRollCallGeTuiContent(selectedRollCallEmployee.getActionEmployeeName());

                        Collection<String> selectedTenantUserIds =
                                selectedRollCallEmployeeEntities.stream().map(ii -> ii.getCalledTenantUserId())
                                        .collect(Collectors.toSet());

                        cIds = selectedRollCallEmployeeEntities.stream()
                                .filter(ii -> StringUtils.hasLength(ii.getCalledTenantUserCid()))
                                .map(ii -> ii.getCalledTenantUserCid()).distinct()
                                .collect(Collectors.toSet());
                        if (!CollectionUtils.isEmpty(cIds)) {
                            GeTuiSharedContentDTO sharedContentDTO = null;
                            try {
                                RollCallRequestDTO requestDTO = RollCallRequestDTO.create(rollCallId,
                                        RollCallType.RollCall,
                                        selectedRollCallEmployee.getActionOrganizationName(),
                                        selectedRollCallEmployee.getActionEmployeeName()
                                );
                                sharedContentDTO = GeTuiSharedContentDTO.create(
                                        title, content, cIds, requestDTO);
                                GeTuiSharedContentResponse response = this.geTuiHttpClient.doPost(sharedContentDTO);
                                paramMap.put("req", sharedContentDTO);
                                paramMap.put("resp", response);
                                paramMap.put("error", false);

                                if ("-1".equalsIgnoreCase(response.getRetType())) {
                                    dataItems.add(GeTuiBadTenantException.GeTuiBadDataItem.create(
                                            String.valueOf(rollCallId),
                                            sharedContentDTO.getCids(),
                                            response.getMsg()
                                    ));
                                }
                            } catch (Exception ex) {
                                String contentJson = this.jsonProvider.getJson(sharedContentDTO);
                                this.kafkaTemplate.send(KafkaConstants.TOPIC_DISPATCH_ROLL_CALL_GE_TUI_REQUEST,
                                        String.valueOf(rollCallId),
                                        contentJson
                                );

                                throw ex;
                            }
                        } else {
                            paramMap.put("cids", cIds);
                            paramMap.put("tenantUserIds", selectedTenantUserIds);
                            paramMap.put("msg", "无对应的Cids信息; 因此, 无法进行消息推送");
                            paramMap.put("error", true);
                        }
                    }
                } catch (Exception ex) {
                    paramMap.put("msg", ex);
                    logger.error("个推推送发生异常：rollCallId={},orgId={}", rollCallId, orgId, ex);

                    dataItems.add(GeTuiBadTenantException.GeTuiBadDataItem.create(
                            String.valueOf(rollCallId),
                            cIds,
                            ex.getMessage()
                    ));
                } finally {
                    String valueJson = null;
                    try {
                        valueJson = this.jsonProvider.getJson(paramMap);
                        this.kafkaTemplate.send(KafkaConstants.TOPIC_DISPATCH_ROLL_CALL_GE_TUI_LOG,
                                String.valueOf(rollCallId),
                                this.jsonProvider.getJson(paramMap)
                        );
                    } catch (Exception ex) {
                        logger.error("记录消息发生异常(rollCallId={}):{}", rollCallId, valueJson, ex);
                    }
                }
            }
        }

        if (!CollectionUtils.isEmpty(dataItems)) {
            throw new GeTuiBadTenantException(dataItems);
        }
    }
}
