package com.bcxin.tenant.open.dubbo.writer.providers.configs;

import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.extension.plugins.MybatisPlusInterceptor;
import com.baomidou.mybatisplus.extension.plugins.inner.PaginationInnerInterceptor;
import com.baomidou.mybatisplus.extension.spring.MybatisSqlSessionFactoryBean;
import com.bcxin.tenant.open.domains.BillPaymentRuleConfig;
import com.bcxin.tenant.open.domains.mappers.RdEmployeeMapper;
import com.bcxin.tenant.open.domains.plugins.MybatisExecuteBatisPlugin;
import com.bcxin.tenant.open.infrastructure.tx.components.UnitWorkTxImpl;
import com.bcxin.tenant.open.infrastructures.UnitWork;
import com.bcxin.tenant.open.infrastructures.components.RetryProvider;
import com.bcxin.tenant.open.infrastructures.configs.TenantSystemConfig;
import com.bcxin.tenant.open.infrastructures.exceptions.IgnoreTenantException;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.annotation.MapperScan;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.retry.RetryPolicy;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.ExceptionClassifierRetryPolicy;
import org.springframework.retry.policy.NeverRetryPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;

@MapperScan(basePackageClasses = RdEmployeeMapper.class)
@Configuration
public class MybatisPlusConfig {

    @Bean
    public MybatisPlusInterceptor mybatisPlusInterceptor() {

        MybatisPlusInterceptor interceptor = new MybatisPlusInterceptor();
        // 添加分页插件
        PaginationInnerInterceptor pageInterceptor = new PaginationInnerInterceptor();
        // 设置请求的页面大于最大页后操作，true调回到首页，false继续请求。默认false
        pageInterceptor.setOverflow(false);
        // 单页分页条数限制，默认无限制
        pageInterceptor.setMaxLimit(5000L);
        // 设置数据库类型
        pageInterceptor.setDbType(DbType.MYSQL);

        interceptor.addInnerInterceptor(pageInterceptor);
        return interceptor;
    }


    /**
     * 在Spring 6.0+中必须添加如下;否则会报异常:  Invalid bound statement (not found)
     *
     * @param datasource
     * @return
     * @throws Exception
     */
    @Bean(name = "sqlSessionFactory")
    public SqlSessionFactory sqlSessionFactory(DataSource datasource,
                                               MybatisPlusInterceptor interceptor,
                                               MybatisExecuteBatisPlugin mybatisExecuteBatisPlugin)
            throws Exception {
        MybatisSqlSessionFactoryBean bean = new MybatisSqlSessionFactoryBean();
        bean.setDataSource(datasource);
        bean.setPlugins(interceptor, mybatisExecuteBatisPlugin);
        bean.setMapperLocations(
                new PathMatchingResourcePatternResolver()
                        .getResources("classpath*:mapper/*.xml"));
        return bean.getObject();
    }

    @Bean
    public MybatisExecuteBatisPlugin mybatisExecuteBatisPlugin(RetryProvider retryProvider) {
        return new MybatisExecuteBatisPlugin(retryProvider);
    }

    @Bean
    public RetryTemplate retryTemplate() {
        RetryTemplate retryTemplate = new RetryTemplate();

        FixedBackOffPolicy fixedBackOffPolicy = new FixedBackOffPolicy();
        fixedBackOffPolicy.setBackOffPeriod(200l);
        retryTemplate.setBackOffPolicy(fixedBackOffPolicy);
        ExceptionClassifierRetryPolicy retryPolicy = new ExceptionClassifierRetryPolicy();
        Map<Class<? extends Throwable>, RetryPolicy> policyMap
                = new HashMap<>();
        policyMap.put(IgnoreTenantException.class, new NeverRetryPolicy());
        policyMap.put(Exception.class, new SimpleRetryPolicy(3));

        retryPolicy.setPolicyMap(policyMap);

        retryTemplate.setRetryPolicy(retryPolicy);

        return retryTemplate;
    }

    @Bean
    public RetryProvider retryProvider(RetryTemplate retryTemplate) {
        return new RetryProvider() {
            @Override
            public <T> T doExecute(Supplier<T> supplier) {
                return retryTemplate.execute(vr -> {
                    return supplier.get();
                });
            }
        };
    }

    @Bean
    public UnitWork unitWork(PlatformTransactionManager platformTransactionManager) {
        return new UnitWorkTxImpl(platformTransactionManager);
    }

    /**
     * 使用@Value的方式不支持 - 的形式
     * @param paidByCallerRegionExpressions
     * @return
     */
    @Bean
    public BillPaymentRuleConfig billPaymentRuleConfig(@Value("#{'${tenant.rules.payment.sponsor.regions:}'.split(',')}")
                                                           Collection<String> paidByCallerRegionExpressions,
                                                       @Value("${tenant.rules.ignored-called-points:true}")boolean ignoredCalledPoints) {
        return new BillPaymentRuleConfig(paidByCallerRegionExpressions, ignoredCalledPoints);
    }

    /**
     * 站点所在的区域
     * @param regionCode
     * @return
     */
    @Bean
    public TenantSystemConfig tenantSystemConfig( @Value("${tenant.system.region-code:}")String regionCode) {
        return new TenantSystemConfig(regionCode);
    }
}
