package com.bcxin.tenant.open.dubbo.writer.providers.configs;

import com.bcxin.tenant.open.domains.entities.TenantEmployeeAttendanceEntity;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DutySignInType;
import com.bcxin.tenant.open.infrastructures.enums.RecordStatus;
import com.bcxin.tenant.open.jdks.AttendanceRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.UpdateEmployeeLocationRequest;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class KafkaComponent_Dispatch {
    private static Logger logger = LoggerFactory.getLogger(KafkaComponent_Dispatch.class);
    private final AttendanceRpcProvider attendanceRpcProvider;
    private final EmployeeWriterRpcProvider employeeWriterRpcProvider;
    private final JsonProvider jsonProvider;

    public KafkaComponent_Dispatch(AttendanceRpcProvider attendanceRpcProvider,
                                   EmployeeWriterRpcProvider employeeWriterRpcProvider, JsonProvider jsonProvider) {
        this.attendanceRpcProvider = attendanceRpcProvider;
        this.employeeWriterRpcProvider = employeeWriterRpcProvider;
        this.jsonProvider = jsonProvider;
    }

    /**
     * 确保这边的数据与backend的KafkaComponent_Dispatch的数据一致；否则会导致重复消费的问题
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-attendance-v5",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ATTENDANCE
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-attendance-v5",concurrency = "6")
    public void ackSyncTenantAttendance(List<ConsumerRecord<String, String>> records,
                                        Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效的签到数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<String> data =
                    records.stream().map(ii -> ii.value()).collect(Collectors.toList());
            this.attendanceRpcProvider.batchCommit(data);
        } catch (Exception ex) {
            logger.error("无效的签到数据({})发生异常", records.stream().map(ii -> ii.value()).collect(Collectors.joining(",")), ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    /**
     * 确保这边的数据与backend的KafkaComponent_Dispatch的数据一致；否则会导致重复消费的问题
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-dispatch-employee-attendance-v2",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_ATTENDANCE
            }, groupId = "${spring.kafka.consumer.group-id}-dispatch-employee-attendance-v2",concurrency = "6")
    public void ackSyncUpdateEmployeeLocation(List<ConsumerRecord<String, String>> records,
                                        Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("无效的签到数据:{}", records);
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<TenantEmployeeAttendanceEntity> attendances =
                    records.stream().map(ii -> this.jsonProvider.toObject(TenantEmployeeAttendanceEntity.class, ii.value()))
                            .collect(Collectors.toList());

            Collection<UpdateEmployeeLocationRequest.EmployeeLocation>
                    employeeLocations =
                    attendances.stream().map(ii -> UpdateEmployeeLocationRequest.EmployeeLocation.create(
                            ii.getTenantEmployeeId(),
                            ii.getRecordStatus() == RecordStatus.SignIn ? DutySignInType.SignIn : DutySignInType.SignOut,
                            ii.getLonLat().getLon(),
                            ii.getLonLat().getLat()
                    )).collect(Collectors.toList());

            employeeWriterRpcProvider.batchUpdateLocations(UpdateEmployeeLocationRequest.create(
                    employeeLocations
            ));

            logger.error("成功更新ids=人员({})的签到状态及经纬度", employeeLocations.stream().map(ii -> ii.getId()).collect(Collectors.joining(",")));
        } catch (Exception ex) {
            logger.error("同步签到到人员({})发生异常:{}", records.stream().map(ii -> ii.value()).collect(Collectors.joining(",")), ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }
}
