package com.bcxin.flink.data.webhook.tasks.jobs;

import com.bcxin.event.core.FlinkJobAbstract;
import com.bcxin.event.core.definitions.KafkaTopicConnectorDefinition;
import com.bcxin.flink.data.webhook.tasks.StreamingJobContext;
import com.bcxin.flink.data.webhook.tasks.components.MultiThreadConsumerSink;
import com.bcxin.flink.data.webhook.tasks.datas.Kafka2WebHookDefinition;
import com.bcxin.flink.streaming.cores.StreamingCoreEnvironments;
import com.bcxin.flink.streaming.cores.dtos.DebeziumJsonNodeDto;
import com.bcxin.flink.streaming.cores.utils.DebeziumJsonNodeDtoUtils;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.flink.api.common.eventtime.WatermarkStrategy;
import org.apache.flink.connector.kafka.source.KafkaSource;
import org.apache.flink.connector.kafka.source.enumerator.initializer.OffsetsInitializer;
import org.apache.flink.connector.kafka.source.reader.deserializer.KafkaRecordDeserializationSchema;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.flink.streaming.api.datastream.DataStreamSource;
import org.apache.flink.streaming.api.datastream.KeyedStream;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.kafka.connect.json.JsonDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.time.temporal.ChronoUnit;

public class WebHookSinkJob extends FlinkJobAbstract {
    private static final Logger logger = LoggerFactory.getLogger(WebHookSinkJob.class);
    private final Kafka2WebHookDefinition definition;

    public WebHookSinkJob(Kafka2WebHookDefinition definition) {
        this.definition = definition;
    }

    @Override
    protected void coreExecute() throws Exception {
        StreamingJobContext jobContext = StreamingJobContext.getInstance();
        StreamExecutionEnvironment env = StreamingCoreEnvironments.getStreamExecutionEnvironment(jobContext,true);
        logger.warn("begin to execute the core function:{}", jobContext.getName());

        KafkaTopicConnectorDefinition kafkaDefinition =
                definition.getTopicDefinition();

        definition.getWebHookConfigs().forEach(whc -> {
            doExecute(env, whc, kafkaDefinition);
        });

        logger.warn("流程订阅定义完毕");

        env.execute(String.format("WebHook-%s-%s", definition.getName(), definition.getJobId()));
    }

    private void doExecute(
            StreamExecutionEnvironment env,
            Kafka2WebHookDefinition.WebHookConfigWrapper whc,
            KafkaTopicConnectorDefinition kafkaDefinition) {
        logger.error("多线程MultiThreadConsumerSink-开始执行如下订阅主题={}; 推送到API:{};", kafkaDefinition.getTopic(), whc.getApiDefinition().getApi());

        KafkaRecordDeserializationSchema kafkaRecordDeserializationSchema =
                KafkaRecordDeserializationSchema.valueOnly(JsonDeserializer.class);
        String selectedTopic = whc.getTopic();
        KafkaSource<JsonNode> stringKafkaSource =
                KafkaSource.<ObjectNode>builder()
                        .setBootstrapServers(kafkaDefinition.getBootstrapServer())
                        .setTopics(selectedTopic)
                        /**
                         * 第一次直接从数据库加载
                         */
                        .setStartingOffsets(OffsetsInitializer.latest())
                        .setGroupId(kafkaDefinition.getConsumerGroupId())
                        .setDeserializer(kafkaRecordDeserializationSchema)
                        .build();

        DataStreamSource<JsonNode> dataStreamSource =
                env.fromSource(stringKafkaSource,
                        WatermarkStrategy.forBoundedOutOfOrderness(Duration.of(100, ChronoUnit.MILLIS)),
                        String.format("订阅主题数据:%s", selectedTopic)
                );

        KeyedStream<DebeziumJsonNodeDto, String> debeziumJsonNodeDtoKeyedStream =
                dataStreamSource.map(ix -> {
                    return DebeziumJsonNodeDtoUtils.translate(ix, kafkaDefinition.getPrimaryKeyName());
                }).keyBy(ii -> ii.getKey());
        debeziumJsonNodeDtoKeyedStream
                .keyBy(ii -> ii.getKey())
                .addSink(new MultiThreadConsumerSink(whc.getApiDefinition()))
                .name(String.format("webHookApi-%s", whc.getTitle()));
    }
}
