package com.bcxin.flink.event.job.web.apis.controllers;

import com.bcxin.event.core.EntityCollection;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.exceptions.NoFoundEventException;
import com.bcxin.event.core.exceptions.NotAllowedEventException;
import com.bcxin.event.job.domain.commands.CreateJobCommand;
import com.bcxin.event.job.domain.commands.DeleteJobCommand;
import com.bcxin.event.job.domain.commands.DeleteLogicJobCommand;
import com.bcxin.event.job.domain.commands.UpdateJobCommand;
import com.bcxin.event.job.domain.entities.JobEntity;
import com.bcxin.event.job.domain.enums.JobType;
import com.bcxin.event.job.domain.repositories.JobRepository;
import com.bcxin.event.job.domain.services.JobService;
import com.bcxin.flink.event.job.web.apis.configs.KafkaConfig;
import com.bcxin.flink.event.job.web.apis.controllers.requests.JobMetaSearchRequest;
import com.bcxin.flink.event.job.web.apis.controllers.requests.JobRequest;
import com.bcxin.flink.event.job.web.apis.controllers.requests.JobSimpleStreamRequest;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/job-metas")
public class JobMetaController extends ControllerBase {
    private final JobRepository jobRepository;
    private final JobService jobService;

    private final JsonProvider jsonProvider;
    private final KafkaConfig kafkaConfig;

    public JobMetaController(JobRepository jobRepository,
                             JobService jobService, JsonProvider jsonProvider, KafkaConfig kafkaConfig) {
        this.jobRepository = jobRepository;
        this.jobService = jobService;
        this.jsonProvider = jsonProvider;
        this.kafkaConfig = kafkaConfig;
    }

    @PostMapping
    public ResponseEntity post(@RequestBody JobRequest request) {
        CreateJobCommand command = request.getCreateCommand(jsonProvider);

        this.jobService.dispatch(command);

        return ok();
    }

    @PostMapping("/simple")
    public ResponseEntity postSimple(@RequestBody JobSimpleStreamRequest request) {
        JobRequest jobRequest = request.getJobRequest(kafkaConfig.getBootstrapServers());

        return this.post(jobRequest);
    }

    @PutMapping("/simple/{id}")
    public ResponseEntity putSimple(@PathVariable int id, @RequestBody JobSimpleStreamRequest request) {
        JobEntity job = this.jobRepository.getById(id);
        if (job == null) {
            throw new NoFoundEventException(String.format("找不到该配置(%s)", id));
        }

        if (job.getJobType() == JobType.CDC_BINLOG) {
            throw new NotAllowedEventException(String.format("该类型(%s)的配置不允许修改", job.getJobType()));
        }

        JobRequest jobRequest = request.getJobRequest(kafkaConfig.getBootstrapServers());

        return this.put(id, jobRequest);
    }

    @DeleteMapping("/simple/{id}")
    public ResponseEntity deleteSimple(@PathVariable int id) {
        JobEntity job = this.jobRepository.getById(id);
        if (job == null) {
            throw new NoFoundEventException(String.format("找不到该配置(%s)", id));
        }

        if (job.getJobType() == JobType.CDC_BINLOG) {
            throw new NotAllowedEventException(String.format("该类型(%s)的配置不允许修改", job.getJobType()));
        }

        this.jobService.dispatch(DeleteLogicJobCommand.create(id));
        return this.ok();
    }

    @PutMapping("/{id}")
    public ResponseEntity put(@PathVariable int id, @RequestBody JobRequest request) {
        UpdateJobCommand command = request.getUpdateCommand(jsonProvider, id);

        this.jobService.dispatch(command);

        return ok();
    }

    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable int id) {
        this.jobService.dispatch(DeleteJobCommand.create(id));

        return ok();
    }

    @PostMapping("/search")
    public ResponseEntity search(@RequestBody JobMetaSearchRequest request) {
        EntityCollection<JobEntity> result =
                this.jobRepository.search(request.getCriteria());

        EntityCollection cc = EntityCollection.create(result.getData().stream().map(ix -> {
            Map x = new HashMap();
            x.put("id", ix.getId());
            x.put("title", ix.getTitle());
            x.put("name", ix.getName());
            x.put("jobType", ix.getJobType());
            x.put("description", ix.getDescription());

            return x;
        }).collect(Collectors.toList()), result.getPageSize(), result.getTotalCount());

        return ok(cc);
    }

    @GetMapping("/{id}")
    public ResponseEntity getById(@PathVariable int id) {
        JobEntity job =
                this.jobRepository.getById(id);
        if (job == null) {
            return noFound();
        }

        return ok(job);
    }
}
