/// <reference types="node" />
import { QRCode, Options } from 'jsqr';
export type CodeResult = QRCode | null;
export declare class QrcodeDecoder {
    timerCapture: null | NodeJS.Timeout;
    canvasElem: null | HTMLCanvasElement;
    gCtx: null | CanvasRenderingContext2D;
    stream: null | MediaStream;
    videoElem: null | HTMLVideoElement;
    getUserMediaHandler: null;
    videoConstraints: MediaStreamConstraints;
    defaultOption: Options;
    constructor();
    /**
     * Verifies if canvas element is supported.
     */
    isCanvasSupported(): boolean;
    _createImageData(target: CanvasImageSource, width: number, height: number): ImageData;
    /**
     * Prepares the canvas element (which will
     * receive the image from the camera and provide
     * what the algorithm needs for checking for a
     * QRCode and then decoding it.)
     *
     *
     * @param  {DOMElement} canvasElem the canvas
     *                                 element
     * @param  {number} width      The width that
     *                             the canvas element
     *                             should have
     * @param  {number} height     The height that
     *                             the canvas element
     *                             should have
     * @return {DOMElement}            the canvas
     * after the resize if width and height
     * provided.
     */
    _prepareCanvas(width: number, height: number): void;
    /**
     * Based on the video dimensions and the canvas
     * that was previously generated captures the
     * video/image source and then paints into the
     * canvas so that the decoder is able to work as
     * it expects.
     * @param  {DOMElement} videoElem <video> dom element
     * @param  {Object} options     options (optional) - Additional options.
     *  inversionAttempts - (attemptBoth (default), dontInvert, onlyInvert, or invertFirst)
     *  refer to jsqr options: https://github.com/cozmo/jsQR
     */
    _captureToCanvas(videoElem: HTMLVideoElement, options: Options): Promise<CodeResult>;
    /**
     * Prepares the video element for receiving
     * camera's input. Releases a stream if there
     * was any (resets).
     *
     * @param  {DOMElement} videoElem <video> dom element
     * @param  {Object} options     options (optional) - Additional options.
     *  inversionAttempts - (attemptBoth (default), dontInvert, onlyInvert, or invertFirst)
     *  refer to jsqr options: https://github.com/cozmo/jsQR
     */
    decodeFromCamera(videoElem: HTMLVideoElement, options?: any): Promise<CodeResult>;
    /**
     * Prepares the video element for video file.
     *
     * @param  {DOMElement} videoElem <video> dom element
     * @param  {Object} options     options (optional) - Additional options.
     *  inversionAttempts - (attemptBoth (default), dontInvert, onlyInvert, or invertFirst)
     *  refer to jsqr options: https://github.com/cozmo/jsQR
     */
    decodeFromVideo(videoElem: HTMLVideoElement, options?: {}): Promise<CodeResult>;
    /**
     * Decodes an image from its src.
     * @param  {DOMElement} imageElem
     * @param  {Object} options     options (optional) - Additional options.
     *  inversionAttempts - (attemptBoth (default), dontInvert, onlyInvert, or invertFirst)
     *  refer to jsqr options: https://github.com/cozmo/jsQR
     */
    decodeFromImage(img: HTMLImageElement | string, options?: {
        crossOrigin?: string;
    }): Promise<CodeResult>;
    _decodeFromImageElm(imgObj: HTMLImageElement, options?: {}): QRCode;
    /**
     * Releases a video stream that was being
     * captured by prepareToVideo
     */
    stop(): this;
    /**
     * Sets the sourceId for the camera to use.
     */
    setGroupId(groupId: string): this;
    /**
     * Gets a list of all available video sources on
     * the current device.
     */
    getVideoDevices(): Promise<MediaDeviceInfo[]>;
}
export default QrcodeDecoder;
