import { GeoJSON, GeometryCollection } from 'geojson';
import { Coord } from '@turf/helpers';

/**
 * Rotates any geojson Feature or Geometry of a specified angle, around its `centroid` or a given `pivot` point.
 *
 * @function
 * @param {GeoJSON} geojson object to be rotated
 * @param {number} angle of rotation in decimal degrees, positive clockwise
 * @param {Object} [options={}] Optional parameters
 * @param {Coord} [options.pivot='centroid'] point around which the rotation will be performed
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} the rotated GeoJSON feature
 * @example
 * const poly = turf.polygon([[[0,29],[3.5,29],[2.5,32],[0,29]]]);
 * const options = {pivot: [0, 25]};
 * const rotatedPoly = turf.transformRotate(poly, 10, options);
 *
 * //addToMap
 * const addToMap = [poly, rotatedPoly];
 * rotatedPoly.properties = {stroke: '#F00', 'stroke-width': 4};
 */
declare function transformRotate<T extends GeoJSON | GeometryCollection>(geojson: T, angle: number, options?: {
    pivot?: Coord;
    mutate?: boolean;
}): T;

export { transformRotate as default, transformRotate };
