import { Polygon, Feature, FeatureCollection, LineString } from 'geojson';
import { Coord, Units } from '@turf/helpers';

/**
 * Returns the shortest {@link LineString|path} from {@link Point|start} to {@link Point|end} without colliding with
 * any {@link Feature} in obstacles {@link FeatureCollection}<{@link Polygon}>
 *
 * @function
 * @param {Coord} start point
 * @param {Coord} end point
 * @param {Object} [options={}] optional parameters
 * @param {Polygon|Feature<Polygon>|FeatureCollection<Polygon>} [options.obstacles] areas which path cannot travel
 * @param {Units} [options.units='kilometers'] unit in which resolution & minimum distance will be expressed in; Supports all valid Turf {@link https://turfjs.org/docs/api/types/Units Units}.
 * @param {number} [options.resolution=100] distance between matrix points on which the path will be calculated
 * @returns {Feature<LineString>} shortest path between start and end
 * @example
 * var start = [-5, -6];
 * var end = [9, -6];
 * var options = {
 *   obstacles: turf.polygon([[[0, -7], [5, -7], [5, -3], [0, -3], [0, -7]]]).geometry
 * };
 *
 * var path = turf.shortestPath(start, end, options);
 *
 * //addToMap
 * var addToMap = [start, end, options.obstacles, path];
 */
declare function shortestPath(start: Coord, end: Coord, options?: {
    obstacles?: Polygon | Feature<Polygon> | FeatureCollection<Polygon>;
    units?: Units;
    resolution?: number;
}): Feature<LineString>;

export { shortestPath as default, shortestPath };
