"use strict";Object.defineProperty(exports, "__esModule", {value: true});// index.js
var _geojsonrbush = require('@turf/geojson-rbush');
var _truncate = require('@turf/truncate');
var _linesegment = require('@turf/line-segment');
var _lineintersect = require('@turf/line-intersect');
var _nearestpointonline = require('@turf/nearest-point-on-line');
var _invariant = require('@turf/invariant');
var _meta = require('@turf/meta');
var _helpers = require('@turf/helpers');
function lineSplit(line, splitter) {
  if (!line) throw new Error("line is required");
  if (!splitter) throw new Error("splitter is required");
  var lineType = _invariant.getType.call(void 0, line);
  var splitterType = _invariant.getType.call(void 0, splitter);
  if (lineType !== "LineString") throw new Error("line must be LineString");
  if (splitterType === "FeatureCollection")
    throw new Error("splitter cannot be a FeatureCollection");
  if (splitterType === "GeometryCollection")
    throw new Error("splitter cannot be a GeometryCollection");
  var truncatedSplitter = _truncate.truncate.call(void 0, splitter, { precision: 7 });
  switch (splitterType) {
    case "Point":
      return splitLineWithPoint(line, truncatedSplitter);
    case "MultiPoint":
      return splitLineWithPoints(line, truncatedSplitter);
    case "LineString":
    case "MultiLineString":
    case "Polygon":
    case "MultiPolygon":
      return splitLineWithPoints(
        line,
        _lineintersect.lineIntersect.call(void 0, line, truncatedSplitter, {
          ignoreSelfIntersections: true
        })
      );
  }
}
function splitLineWithPoints(line, splitter) {
  var results = [];
  var tree = _geojsonrbush.geojsonRbush.call(void 0, );
  _meta.flattenEach.call(void 0, splitter, function(point) {
    results.forEach(function(feature, index) {
      feature.id = index;
    });
    if (!results.length) {
      results = splitLineWithPoint(line, point).features;
      tree.load(_helpers.featureCollection.call(void 0, results));
    } else {
      var search = tree.search(point);
      if (search.features.length) {
        var closestLine = findClosestFeature(point, search);
        results = results.filter(function(feature) {
          return feature.id !== closestLine.id;
        });
        tree.remove(closestLine);
        _meta.featureEach.call(void 0, splitLineWithPoint(closestLine, point), function(line2) {
          results.push(line2);
          tree.insert(line2);
        });
      }
    }
  });
  return _helpers.featureCollection.call(void 0, results);
}
function splitLineWithPoint(line, splitter) {
  var results = [];
  var startPoint = _invariant.getCoords.call(void 0, line)[0];
  var endPoint = _invariant.getCoords.call(void 0, line)[line.geometry.coordinates.length - 1];
  if (pointsEquals(startPoint, _invariant.getCoord.call(void 0, splitter)) || pointsEquals(endPoint, _invariant.getCoord.call(void 0, splitter)))
    return _helpers.featureCollection.call(void 0, [line]);
  var tree = _geojsonrbush.geojsonRbush.call(void 0, );
  var segments = _linesegment.lineSegment.call(void 0, line);
  tree.load(segments);
  var search = tree.search(splitter);
  if (!search.features.length) return _helpers.featureCollection.call(void 0, [line]);
  var closestSegment = findClosestFeature(splitter, search);
  var initialValue = [startPoint];
  var lastCoords = _meta.featureReduce.call(void 0, 
    segments,
    function(previous, current, index) {
      var currentCoords = _invariant.getCoords.call(void 0, current)[1];
      var splitterCoords = _invariant.getCoord.call(void 0, splitter);
      if (index === closestSegment.id) {
        previous.push(splitterCoords);
        results.push(_helpers.lineString.call(void 0, previous));
        if (pointsEquals(splitterCoords, currentCoords))
          return [splitterCoords];
        return [splitterCoords, currentCoords];
      } else {
        previous.push(currentCoords);
        return previous;
      }
    },
    initialValue
  );
  if (lastCoords.length > 1) {
    results.push(_helpers.lineString.call(void 0, lastCoords));
  }
  return _helpers.featureCollection.call(void 0, results);
}
function findClosestFeature(point, lines) {
  if (!lines.features.length) throw new Error("lines must contain features");
  if (lines.features.length === 1) return lines.features[0];
  var closestFeature;
  var closestDistance = Infinity;
  _meta.featureEach.call(void 0, lines, function(segment) {
    var pt = _nearestpointonline.nearestPointOnLine.call(void 0, segment, point);
    var dist = pt.properties.dist;
    if (dist < closestDistance) {
      closestFeature = segment;
      closestDistance = dist;
    }
  });
  return closestFeature;
}
function pointsEquals(pt1, pt2) {
  return pt1[0] === pt2[0] && pt1[1] === pt2[1];
}
var index_default = lineSplit;



exports.default = index_default; exports.lineSplit = lineSplit;
//# sourceMappingURL=index.cjs.map