package com.bcxin.tenant.domain.services.commands.organizations;

import com.bcxin.Infrastructures.commands.CommandAbstract;
import com.bcxin.Infrastructures.enums.*;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.api.interfaces.tenants.requests.organizations.CompanyRegistrationRequest;
import lombok.Getter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * description: 企业注册信息对象
 * author: linchunpeng
 * date:  2023-05-09 13:56
 */
@Getter
public class RegisterCompanyCommand extends CommandAbstract {


    //行业类型
    private final String industryCode;

    //机构类型
    private final String institutionalCode;

    //保安服务公司类型（机构类型 = 保安服务公司 必填）
    private final IndustryDetailType industryDetailType;


    //基本信息
    //公司名称
    private final String name;

    //总/分/子公司
    private final ParentSubsidiary type;

    //联系电话
    private final String telephone;

    //经济类型
    private final EconomicType economicType;

    //公司类型
    private final CompanyNature nature;

    //注册地区
    private final LocationCommandItem placeOfRegister;

    //经营地区
    private final LocationCommandItem placeOfBusiness;


    //总公司信息，保安行业必填
    //总公司名称（机构类型 = 保安服务公司 && (总/分/子公司 = 分 || 子) 必填）
    private final String parentName;

    //总公司证件类型（机构类型 = 保安服务公司 && (总/分/子公司 = 分 || 子) 必填）
    private final CompanyCertificateType parentCertificateType;

    //总公司证件号码（机构类型 = 保安服务公司 && (总/分/子公司 = 分 || 子) 必填）
    private final String parentCertificateNumber;

    //总公司法人姓名（机构类型 = 保安服务公司 && (总/分/子公司 = 分 || 子) 必填）
    private final String parentLegalPersonName;

    //总公司联系电话（机构类型 = 保安服务公司 && (总/分/子公司 = 分 || 子) 必填）
    private final String parentLegalPersonTelephone;


    //法人或负责人信息
    private final String legalPersonName;

    //法人手机号码
    private final String legalPersonTelephone;

    //法人证件类型
    private final CredentialType legalPersonCredentialType;

    //法人证件号码
    private final String legalPersonCredentialNumber;

    //法人国籍
    private final String legalPersonNationality;

    //保安负责人信息
    //保安负责人姓名
    private final String securityPersonName;

    //保安负责人手机号码
    private final String securityPersonTelephone;

    //保安负责人证件类型
    private final CredentialType securityPersonCredentialType;

    //保安负责人证件号码
    private final String securityPersonCredentialNumber;


    //注册信息
    //注册资本（万元）
    private final String registeredCapital;

    //固定资本（万元，机构类型 = 保安服务公司 必填）
    private final String fixedCapital;

    //年营业额（万元，机构类型 = 保安服务公司 必填）
    private final String annualSalesVolume;

    //年利税（万元，机构类型 = 保安服务公司 必填）
    private final String annualProfitTax;

    //证件类型", notes = "com.bcxin.Infrastructures.enums.CompanyCertificateType
    private final CompanyCertificateType certificateType;

    //统一社会信用代码
    private final String unifySocialCreditCode;

    //营业执照附件
    private final String unifySocialCreditCodeFile;


    //资质信息
    //保安服务许可/保安培训备案证号（机构类型 = 保安服务公司 || 保安培训单位 必填）
    private final String servicePermitNumber;

    //发证机关名称（机构类型 = 保安服务公司 || 保安培训单位 必填）
    private final String nameOfIssuingAuthority;

    //保安服务许可证首次发证日期或者培训单位的发证日期（机构类型 = 保安服务公司 || 保安培训单位 必填）
    private final Date firstIssueServicePermit;

    //保安服务许可证扫描件或者保安培训许可（备案）证扫描件（机构类型 = 保安服务公司 || 保安培训单位 必填）
    private final String servicePermitAttachment;


    //批准文号（机构类型 = 保安服务公司 必填）
    private final String securityApprovalNumber;

    //服务范围（机构类型 = 保安服务公司 必填）
    private final List<String> serviceScopeList;

    //人力防范评定等级（机构类型 = 保安服务公司 必填）", notes = "com.bcxin.Infrastructures.enums.SecurityPreparednessRating
    private final SecurityPreparednessRating securityPreparednessRating;

    //武装守护押运评定等级（机构类型 = 保安服务公司 必填）", notes = "com.bcxin.Infrastructures.enums.SecurityArmedRating
    private final SecurityArmedRating securityArmedRating;

    //安全技术防范评定等级（机构类型 = 保安服务公司 必填）", notes = "com.bcxin.Infrastructures.enums.SecuritySafetyDefenseRating
    private final SecuritySafetyDefenseRating securitySafetyDefenseRating;

    //是否有风险评估资质（机构类型 = 保安服务公司 必填）
    private final Boolean isRiskAssessment;


    //培训单位的培训内容（机构类型 = 保安培训单位 必填）
    private final String trainContent;


    //自行招用保安员单位
    //拟自招保安员数（机构类型 = 自行招用保安员单位 必填）
    private final Integer numberOfSecurityGuards;

    //是否治安保卫重点单位（机构类型 = 自行招用保安员单位 必填）
    private final Boolean isPublicSecuritySecurity;

    //是否物业企业（机构类型 = 自行招用保安员单位 必填）
    private final Boolean isPropertyEnterprise;

    //自招保安单位性质（机构类型 = 自行招用保安员单位 必填）
    private final String natureOfSelfRecruitedUnits;


    //监管归属
    //监管地区
    private final LocationCommandItem placeOfSupervise;

    //监管机构
    private final String superviseDepartId;

    //监管机构名称
    private final String superviseDepartName;


    //系统管理员
    private final ContactCommandItem administrator;

    //动态扩展配置（默认监管归属绑定）
    private final Map<String, String> dynamic;

    //动态扩展字段 json
    private final Map<String, Object> dynamicFields;


    //分享单位id
    private final String shareOrganizationId;

    //分享单位名称(冗余)
    private final String shareOrganizationName;

    //分享职员id
    private final String shareEmployeeId;

    //分享职员姓名(冗余)
    private final String shareEmployeeName;


    @Override
    public void validate() {
        if (this.getInstitutionalCode().equals("0102")
                || this.getInstitutionalCode().equals("0103")
                || this.getInstitutionalCode().equals("0104")) {
            if (this.getType() == null) {
                throw new BadTenantException("总/分/子公司不能为空！");
            }
            if (StringUtils.isBlank(this.getTelephone())) {
                throw new BadTenantException("联系电话不能为空！");
            }
            if (this.getEconomicType() == null) {
                throw new BadTenantException("经济类型不能为空！");
            }
            if (this.getNature() == null) {
                throw new BadTenantException("公司类型不能为空！");
            }
            if (StringUtils.isBlank(this.getLegalPersonNationality())) {
                throw new BadTenantException("法人国籍不能为空！");
            }
            if (StringUtils.isBlank(this.getRegisteredCapital())) {
                throw new BadTenantException("注册资本不能为空！");
            }
            this.validateLocation(this.getPlaceOfSupervise(), "监管地区", false);
            if (StringUtils.isBlank(this.getSuperviseDepartId())) {
                throw new BadTenantException("监管机构不能为空！");
            }
            if (StringUtils.isBlank(this.getSuperviseDepartName())) {
                throw new BadTenantException("监管机构名称不能为空！");
            }
        }
        switch (this.getInstitutionalCode()) {
            case "0102":
                //验证机构类型为保安服务公司时必填
                if (this.getIndustryDetailType() == null) {
                    throw new BadTenantException("保安服务公司类型不能为空！");
                }
                if (StringUtils.isBlank(this.getFixedCapital())) {
                    throw new BadTenantException("固定资本不能为空！");
                }
                if (StringUtils.isBlank(this.getAnnualSalesVolume())) {
                    throw new BadTenantException("年营业额不能为空！");
                }
                if (StringUtils.isBlank(this.getAnnualProfitTax())) {
                    throw new BadTenantException("年利税不能为空！");
                }
                if (StringUtils.isBlank(this.getSecurityApprovalNumber())) {
                    throw new BadTenantException("批准文号不能为空！");
                }
                if (CollectionUtils.isEmpty(this.getServiceScopeList())) {
                    throw new BadTenantException("服务范围不能为空！");
                }
                if (this.getSecurityPreparednessRating() == null) {
                    throw new BadTenantException("人力防范评定等级不能为空！");
                }
                if (this.getSecurityArmedRating() == null) {
                    throw new BadTenantException("武装守护押运评定等级不能为空！");
                }
                if (this.getSecuritySafetyDefenseRating() == null) {
                    throw new BadTenantException("安全技术防范评定等级不能为空！");
                }
                if (this.getIsRiskAssessment() == null) {
                    throw new BadTenantException("是否有风险评估资质不能为空！");
                }
                for (String serviceScope : this.getServiceScopeList()) {
                    if (serviceScope.equals(ServiceScope.SecurityRiskAssessment.getTypeValue()) && !this.getIsRiskAssessment()) {
                        //安全风险评估，是否有风险评估资质必须为：是
                        throw new BadTenantException("服务范围包含：安全风险评估，是否有风险评估资质必须选择：是");
                    }
                }

                if (StringUtils.isBlank(this.getServicePermitNumber())) {
                    throw new BadTenantException("保安服务许可证号不能为空！");
                }
                if (StringUtils.isBlank(this.getNameOfIssuingAuthority())) {
                    throw new BadTenantException("发证机关名称不能为空！");
                }
                if (this.getFirstIssueServicePermit() == null) {
                    throw new BadTenantException("发证日期不能为空！");
                }
                if (StringUtils.isBlank(this.getServicePermitAttachment())) {
                    throw new BadTenantException("保安服务许可证扫描件不能为空！");
                }

                if (this.getType().ordinal() == ParentSubsidiary.Branch.ordinal()
                        || this.getType().ordinal() == ParentSubsidiary.Subsidiary.ordinal()) {
                    //验证总/分/子公司 = 分 || 子 必填项
                    if (StringUtils.isBlank(this.getParentName())) {
                        throw new BadTenantException("总公司名称不能为空！");
                    }
                    if (this.getParentCertificateType() == null) {
                        throw new BadTenantException("总公司证件类型不能为空！");
                    }
                    if (StringUtils.isBlank(this.getParentCertificateNumber())) {
                        throw new BadTenantException("总公司证件号码不能为空！");
                    }
                    if (StringUtils.isBlank(this.getParentLegalPersonName())) {
                        throw new BadTenantException("总公司法人姓名不能为空！");
                    }
                    if (StringUtils.isBlank(this.getParentLegalPersonTelephone())) {
                        throw new BadTenantException("总公司联系电话不能为空！");
                    }
                }
                break;
            case "0103":
                //验证机构类型为保安培训单位时必填
                if (StringUtils.isBlank(this.getTrainContent())) {
                    throw new BadTenantException("培训单位的培训内容不能为空！");
                }
                if (StringUtils.isBlank(this.getServicePermitNumber())) {
                    throw new BadTenantException("保安培训许可（备案）证号不能为空！");
                }
                if (StringUtils.isBlank(this.getNameOfIssuingAuthority())) {
                    throw new BadTenantException("发证机关名称不能为空！");
                }
                if (this.getFirstIssueServicePermit() == null) {
                    throw new BadTenantException("发证日期不能为空！");
                }
                if (StringUtils.isBlank(this.getServicePermitAttachment())) {
                    throw new BadTenantException("保安培训许可（备案）证扫描件不能为空！");
                }
                break;
            case "0104":
                //验证机构类型为自行招用保安员单位时必填
                if (StringUtils.isBlank(this.getSecurityPersonName())) {
                    throw new BadTenantException("保安负责人姓名不能为空");
                }
                if (StringUtils.isBlank(this.getSecurityPersonTelephone())) {
                    throw new BadTenantException("保安负责人手机号码不能为空");
                }
                if (this.getSecurityPersonCredentialType() == null) {
                    throw new BadTenantException("保安负责人证件类型不能为空");
                }
                if (StringUtils.isBlank(this.getSecurityPersonCredentialNumber())) {
                    throw new BadTenantException("保安负责人证件编号不能为空");
                }
                if (this.getNumberOfSecurityGuards() == null) {
                    throw new BadTenantException("拟自招保安员数不能为空！");
                }
                if (this.getIsPublicSecuritySecurity() == null) {
                    throw new BadTenantException("是否治安保卫重点单位不能为空！");
                }
                if (this.getIsPropertyEnterprise() == null) {
                    throw new BadTenantException("是否物业企业不能为空！");
                }
                if (StringUtils.isBlank(this.getNatureOfSelfRecruitedUnits())) {
                    throw new BadTenantException("自招保安单位性质不能为空！");
                }
                break;
        }

        //只有保安行业才走下面这个的校验逻辑,其他行业不走
        if ("01".contains(this.getIndustryCode())) {
            //不是大活行业、群防群治行业、行业主管部门
            if (!this.getInstitutionalCode().equals("0406")) {
                //不是党政机关
                this.validateLocation(this.getPlaceOfRegister(), "注册地区", true);
                if (this.getCertificateType() == null) {
                    throw new BadTenantException("证件类型不能为空！");
                }
                if (StringUtils.isBlank(this.getUnifySocialCreditCode())) {
                    throw new BadTenantException("统一社会信用代码不能为空！");
                }
                if (StringUtils.isBlank(this.getUnifySocialCreditCodeFile())) {
                    throw new BadTenantException("营业执照扫描件不能为空！");
                }
            }
            this.validateLocation(this.getPlaceOfBusiness(), "经营地区", true);
            if (StringUtils.isBlank(this.getLegalPersonName())) {
                throw new BadTenantException("法人姓名不能为空！");
            }
            if (StringUtils.isBlank(this.getLegalPersonTelephone())) {
                throw new BadTenantException("法人手机号码不能为空！");
            }
            if (this.getLegalPersonCredentialType() == null) {
                throw new BadTenantException("法人证件类型不能为空！");
            }
            if (StringUtils.isBlank(this.getLegalPersonCredentialNumber())) {
                throw new BadTenantException("法人证件号码不能为空！");
            }
        }

        if ("02,07".equals(this.getIndustryCode())) {
            //大活行业、行业主管部门
            this.validateLocation(this.getPlaceOfBusiness(), "办公地区", true);
        }

        if ("04,06".contains(this.getIndustryCode())) {
            //群防群治行业、内保行业
            this.validateLocation(this.getPlaceOfSupervise(), "监管地区", false);
            if (StringUtils.isBlank(this.getSuperviseDepartId())) {
                throw new BadTenantException("监管机构不能为空！");
            }
            if (StringUtils.isBlank(this.getSuperviseDepartName())) {
                throw new BadTenantException("监管机构名称不能为空！");
            }
        }

        this.validateContact(this.getAdministrator(), "系统管理员", false);
    }

    private void validateContact(ContactCommandItem commandItem, String category, boolean isValidNationality) {
        if (commandItem == null) {
            throw new BadTenantException(String.format("%s不能为空!", category));
        }

        if (StringUtils.isBlank(commandItem.getName())) {
            throw new BadTenantException(String.format("%s的姓名不能为空!", category));
        }

        if (StringUtils.isBlank(commandItem.getTelephone())) {
            throw new BadTenantException(String.format("%s的手机号码不能为空!", category));
        }

        if (commandItem.getCredentialType() == null) {
            throw new BadTenantException(String.format("%s的证件类型不能为空!", category));
        }

        if (StringUtils.isBlank(commandItem.getCredentialNumber())) {
            throw new BadTenantException(String.format("%s的证件号码不能为空!", category));
        }

        if (isValidNationality && StringUtils.isBlank(commandItem.getNationality())) {
            throw new BadTenantException(String.format("%s的国籍不能为空!", category));
        }
    }

    private void validateLocation(LocationCommandItem commandItem, String category, boolean isValidAddress) {
        if (commandItem == null) {
            throw new BadTenantException(String.format("%s不能为空!", category));
        }

        if (commandItem.getProvince() == null || StringUtils.isBlank(commandItem.getProvince().getCode())) {
            throw new BadTenantException(String.format("%s的省份不能为空!", category));
        }

//        if (commandItem.getCity() == null || StringUtils.isBlank(commandItem.getCity().getCode())) {
//            throw new BadTenantException(String.format("%s的城市不能为空!", category));
//        }
//
//        if (commandItem.getDistrict() == null || StringUtils.isBlank(commandItem.getDistrict().getCode())) {
//            throw new BadTenantException(String.format("%s的区域不能为空!", category));
//        }

        if (isValidAddress && StringUtils.isBlank(commandItem.getAddress())) {
            throw new BadTenantException(String.format("%s的地址不能为空!", category));
        }
    }

    public static RegisterCompanyCommand create(CompanyRegistrationRequest registrationRequest) {
        return new RegisterCompanyCommand(registrationRequest.getIndustryCode(), registrationRequest.getInstitutionalCode(), registrationRequest.getIndustryDetailType(),
                registrationRequest.getName(), registrationRequest.getType(), registrationRequest.getTelephone(), registrationRequest.getEconomicType(),
                registrationRequest.getNature(), LocationCommandItem.create(registrationRequest.getPlaceOfRegister()), LocationCommandItem.create(registrationRequest.getPlaceOfBusiness()),
                registrationRequest.getParentName(), registrationRequest.getParentCertificateType(), registrationRequest.getParentCertificateNumber(),
                registrationRequest.getParentLegalPersonName(), registrationRequest.getParentLegalPersonTelephone(),
                registrationRequest.getLegalPersonName(), registrationRequest.getLegalPersonTelephone(), registrationRequest.getLegalPersonCredentialType(),
                registrationRequest.getLegalPersonCredentialNumber(), registrationRequest.getLegalPersonNationality(), registrationRequest.getSecurityPersonName(),
                registrationRequest.getSecurityPersonTelephone(), registrationRequest.getSecurityPersonCredentialType(), registrationRequest.getSecurityPersonCredentialNumber(),
                registrationRequest.getRegisteredCapital(), registrationRequest.getFixedCapital(),
                registrationRequest.getAnnualSalesVolume(), registrationRequest.getAnnualProfitTax(), registrationRequest.getCertificateType(),
                registrationRequest.getUnifySocialCreditCode(), registrationRequest.getUnifySocialCreditCodeFile(), registrationRequest.getServicePermitNumber(),
                registrationRequest.getNameOfIssuingAuthority(), registrationRequest.getFirstIssueServicePermit(), registrationRequest.getServicePermitAttachment(),
                registrationRequest.getSecurityApprovalNumber(), registrationRequest.getServiceScopeList(), registrationRequest.getSecurityPreparednessRating(),
                registrationRequest.getSecurityArmedRating(), registrationRequest.getSecuritySafetyDefenseRating(),
                registrationRequest.getIsRiskAssessment(), registrationRequest.getTrainContent(), registrationRequest.getNumberOfSecurityGuards(),
                registrationRequest.getIsPublicSecuritySecurity(), registrationRequest.getIsPropertyEnterprise(), registrationRequest.getNatureOfSelfRecruitedUnits(),
                LocationCommandItem.create(registrationRequest.getPlaceOfSupervise()), registrationRequest.getSuperviseDepartId(), registrationRequest.getSuperviseDepartName(),
                RegisterCompanyCommand.ContactCommandItem.create(registrationRequest.getAdministrator()), registrationRequest.getDynamic(),
                registrationRequest.getDynamicFields(),
                registrationRequest.getShareOrganizationId(), registrationRequest.getShareOrganizationName(), registrationRequest.getShareEmployeeId(), registrationRequest.getShareEmployeeName());
    }

    public RegisterCompanyCommand(String industryCode, String institutionalCode, IndustryDetailType industryDetailType,
                                  String name, ParentSubsidiary type, String telephone, EconomicType economicType,
                                  CompanyNature nature, LocationCommandItem placeOfRegister, LocationCommandItem placeOfBusiness,
                                  String parentName, CompanyCertificateType parentCertificateType, String parentCertificateNumber,
                                  String parentLegalPersonName, String parentLegalPersonTelephone,
                                  String legalPersonName, String legalPersonTelephone, CredentialType legalPersonCredentialType,
                                  String legalPersonCredentialNumber, String legalPersonNationality, String securityPersonName, String securityPersonTelephone,
                                  CredentialType securityPersonCredentialType, String securityPersonCredentialNumber,
                                  String registeredCapital, String fixedCapital,
                                  String annualSalesVolume, String annualProfitTax, CompanyCertificateType certificateType,
                                  String unifySocialCreditCode, String unifySocialCreditCodeFile, String servicePermitNumber,
                                  String nameOfIssuingAuthority, Date firstIssueServicePermit, String servicePermitAttachment,
                                  String securityApprovalNumber, List<String> serviceScopeList, SecurityPreparednessRating securityPreparednessRating,
                                  SecurityArmedRating securityArmedRating, SecuritySafetyDefenseRating securitySafetyDefenseRating,
                                  Boolean isRiskAssessment, String trainContent, Integer numberOfSecurityGuards,
                                  Boolean isPublicSecuritySecurity, Boolean isPropertyEnterprise, String natureOfSelfRecruitedUnits,
                                  LocationCommandItem placeOfSupervise, String superviseDepartId, String superviseDepartName,
                                  ContactCommandItem administrator, Map<String, String> dynamic,
                                  Map<String, Object> dynamicFields,
                                  String shareOrganizationId, String shareOrganizationName, String shareEmployeeId, String shareEmployeeName) {
        this.industryCode = industryCode;
        this.institutionalCode = institutionalCode;
        this.industryDetailType = industryDetailType;
        this.name = name;
        this.type = type;
        this.telephone = telephone;
        this.economicType = economicType;
        this.nature = nature;
        this.placeOfRegister = placeOfRegister;
        this.placeOfBusiness = placeOfBusiness;
        this.parentName = parentName;
        this.parentCertificateType = parentCertificateType;
        this.parentCertificateNumber = parentCertificateNumber;
        this.parentLegalPersonName = parentLegalPersonName;
        this.parentLegalPersonTelephone = parentLegalPersonTelephone;
        this.legalPersonName = legalPersonName;
        this.legalPersonTelephone = legalPersonTelephone;
        this.legalPersonCredentialType = legalPersonCredentialType;
        this.legalPersonCredentialNumber = legalPersonCredentialNumber;
        this.legalPersonNationality = legalPersonNationality;
        this.securityPersonName = securityPersonName;
        this.securityPersonTelephone = securityPersonTelephone;
        this.securityPersonCredentialType = securityPersonCredentialType;
        this.securityPersonCredentialNumber = securityPersonCredentialNumber;
        this.registeredCapital = registeredCapital;
        this.fixedCapital = fixedCapital;
        this.annualSalesVolume = annualSalesVolume;
        this.annualProfitTax = annualProfitTax;
        this.certificateType = certificateType;
        this.unifySocialCreditCode = unifySocialCreditCode;
        this.unifySocialCreditCodeFile = unifySocialCreditCodeFile;
        this.servicePermitNumber = servicePermitNumber;
        this.nameOfIssuingAuthority = nameOfIssuingAuthority;
        this.firstIssueServicePermit = firstIssueServicePermit;
        this.servicePermitAttachment = servicePermitAttachment;
        this.securityApprovalNumber = securityApprovalNumber;
        this.serviceScopeList = serviceScopeList;
        this.securityPreparednessRating = securityPreparednessRating;
        this.securityArmedRating = securityArmedRating;
        this.securitySafetyDefenseRating = securitySafetyDefenseRating;
        this.isRiskAssessment = isRiskAssessment;
        this.trainContent = trainContent;
        this.numberOfSecurityGuards = numberOfSecurityGuards;
        this.isPublicSecuritySecurity = isPublicSecuritySecurity;
        this.isPropertyEnterprise = isPropertyEnterprise;
        this.natureOfSelfRecruitedUnits = natureOfSelfRecruitedUnits;
        this.placeOfSupervise = placeOfSupervise;
        this.superviseDepartId = superviseDepartId;
        this.superviseDepartName = superviseDepartName;
        this.administrator = administrator;
        this.dynamic = dynamic;
        this.dynamicFields = dynamicFields;

        if (this.industryCode.equals("04")) {
            this.shareOrganizationId = shareOrganizationId;
            this.shareOrganizationName = shareOrganizationName;
            this.shareEmployeeId = shareEmployeeId;
            this.shareEmployeeName = shareEmployeeName;
        } else {
            this.shareOrganizationId = null;
            this.shareOrganizationName = null;
            this.shareEmployeeId = null;
            this.shareEmployeeName = null;
        }
    }


    @Getter
    public static class ContactCommandItem {
        private final String name;
        private final String telephone;
        private final CredentialType credentialType;
        private final String credentialNumber;
        private final String nationality;

        public ContactCommandItem(String name, String telephone, CredentialType credentialType,
                                  String credentialNumber, String nationality) {
            this.name = name;
            this.telephone = telephone;
            this.credentialType = credentialType;
            this.credentialNumber = credentialNumber;
            this.nationality = nationality;
        }

        public static RegisterCompanyCommand.ContactCommandItem create(String name, String telephone, CredentialType credentialType,
                                                                       String credentialNumber) {
            return new ContactCommandItem(name, telephone, credentialType, credentialNumber, null);
        }


        public static RegisterCompanyCommand.ContactCommandItem create(CompanyRegistrationRequest.ContactRequest contactRequest) {
            if (contactRequest != null) {
                return new ContactCommandItem(contactRequest.getName(), contactRequest.getTelephone(), contactRequest.getCredentialType(), contactRequest.getCredentialNumber(), contactRequest.getNationality());
            }
            return null;
        }
    }

}
