package com.bcxin.tenant.domain.entities;

import com.bcxin.Infrastructures.entities.EntityAbstract;
import com.bcxin.Infrastructures.entities.EventEntityAbstract;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.utils.DateUtil;
import com.bcxin.Infrastructures.utils.UUIDUtil;
import com.bcxin.tenant.domain.enums.EventAction;
import com.bcxin.tenant.domain.enums.EventProcessedStatus;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.springframework.util.StringUtils;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;

@Getter
@Setter(AccessLevel.PROTECTED)
@Table(name = "tenant_events")
@Entity
public class TenantEventEntity extends EventEntityAbstract implements IAggregate,Comparable<TenantEventEntity> {
    @Id
    private String id;

    @Column(length = 5000)
    private String content;

    @Enumerated(EnumType.ORDINAL)
    @Column(nullable = false, name = "event_action")
    private EventAction eventAction;

    @Enumerated(EnumType.ORDINAL)
    @Column(nullable = false, name = "status")
    private EventProcessedStatus status;

    @Column(length = 5000)
    private String result;

    @Column(name = "keyword", nullable = false, length = 200)
    private String keyword;

    @Column(name = "last_processed_time")
    private Timestamp lastProcessedTime;

    protected TenantEventEntity() {
        super();
    }

    public static TenantEventEntity create(
            EventAction eventAction, String keyword, String content) {
        TenantEventEntity tenantEvent = new TenantEventEntity();
        tenantEvent.setId(String.format("%s-%s", DateUtil.format2ShortDateTime(tenantEvent.getCreatedTime()), UUIDUtil.getShortUuid()));
        tenantEvent.setStatus(EventProcessedStatus.Init);
        tenantEvent.setEventAction(eventAction);
        tenantEvent.setContent(content);
        tenantEvent.setKeyword(keyword);

        tenantEvent.setMapKey(eventAction.name());

        return tenantEvent;
    }

    public void done(String result) {
        this.setStatus(EventProcessedStatus.Processed);
        this.setResult(String.format("%s-%s", this.getResult(), result));
        if(this.getResult().length()>2000) {
            this.setResult(result);
        }

        this.setLastProcessedTime(Timestamp.from(Instant.now()));
    }

    public void error(String result) {
        if (this.getEventAction()==EventAction.TenantUserAfterCreatedEventForSms &&
                StringUtils.hasLength(result) && result.contains("不正确")) {
            this.setStatus(EventProcessedStatus.FailedAndIgnored);
        } else {
            this.setStatus(EventProcessedStatus.Failed);
        }

        this.setResult(result);
        this.setLastProcessedTime(Timestamp.from(Instant.now()));
    }

    @Override
    public int compareTo(TenantEventEntity o) {
        if (o == null) {
            return 1;
        }

        return (int)(this.getVersion() - o.getVersion());
    }
}
