package com.bcxin.tenant.domain.entities;

import com.bcxin.Infrastructures.entities.EntityAbstract;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.entities.OperatorValueType;
import com.bcxin.Infrastructures.enums.EmploymentStatus;
import com.bcxin.Infrastructures.enums.OccupationType;
import com.bcxin.Infrastructures.enums.TrueFalseStatus;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Date;

@Getter
@Setter(AccessLevel.PROTECTED)
@Table(name = "tenant_employee_records")
@Entity
public class EmployeeRecordEntity extends EntityAbstract implements IAggregate {

    @Id
    /**
     * 对于自增, 没有加Idenytity的话, 会报错: A different object with the same identifier value was already associated with the session
     */
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private long id;

    @Column(name = "employee_id", nullable = false)
    private String employeeId;

    @Column(name = "tenant_user_id", nullable = false)
    private String tenantUserId;

    @Column(name = "organization_id", nullable = false)
    private String organizationId;

    /**
     * 职业: 保安员, 其他
     */
    @Enumerated(EnumType.ORDINAL)
    @Column(name = "occupation_type", nullable = false)
    private OccupationType occupationType;

    /**
     * 待入职，入职, 离职申请中，离职
     */
    @Enumerated(EnumType.ORDINAL)
    @Column(name = "status", nullable = false)
    private EmploymentStatus status;

    /**
     * 操作人
     */
    @Embedded
    @AttributeOverrides({
            @AttributeOverride(name = "id",
                    column = @Column(name = "operator_id", nullable = true, length = 150)),
            @AttributeOverride(name = "name",
                    column = @Column(name = "operator_name", nullable = true, length = 200)),
            @AttributeOverride(name = "createdTime",
                    column = @Column(name = "operator_created_time", nullable = true))
    })
    private OperatorValueType operator;

    /**
     * 入职或者离职时间
     */
    @Column(name = "action_time", nullable = true)
    private Date actionTime;

    /**
     * 入职或者离职备注
     */
    @Column(name = "action_note", length = 500)
    private String actionNote;

    /**
     * 职位
     */
    @Column(name = "position", length = 200)
    private String position;

    /**
     * 是否组织管理员
     */
    @Column(name = "is_domain_admin", nullable = false)
    private TrueFalseStatus domainAdmin;

    /**
     * 面谈记录
     */
    @Column(name = "interview", nullable = true, length = 500)
    private String interview;

    @Column(name = "department_id", nullable = true, length = 200)
    private String departmentId;

    @Column(name = "department_name", nullable = true, length = 200)
    private String departmentName;

    @Column(name = "created_time", nullable = false)
    private Timestamp createdTime;

    public EmployeeRecordEntity() {
        this.setCreatedTime(Timestamp.from(Instant.now()));
    }

    public static EmployeeRecordEntity create(EmployeeEntity employee) {
        EmployeeRecordEntity employeeRecord = new EmployeeRecordEntity();
        employeeRecord.setEmployeeId(employee.getId());
        employeeRecord.setTenantUserId(employee.getTenantUser().getId());
        employeeRecord.setOrganizationId(employee.getOrganization().getId());

        employeeRecord.setOccupationType(employee.getOccupationType());
        employeeRecord.setStatus(employee.getStatus());
        employeeRecord.setPosition(employee.getPosition());

        if (employee.getStatus() == EmploymentStatus.OnJob) {
            employeeRecord.setActionTime(employee.getHiredDate());
            /**
             * 如果有离职过, 那么LeaveOperator将不为空
             */
            if (employee.getLeaveOperator() != null) {
                employeeRecord.setActionNote("复职");
            } else {
                employeeRecord.setActionNote("入职");
            }

            employeeRecord.setOperator(employee.getHiredOperator());
        } else {
            employeeRecord.setActionTime(employee.getLeaveTime());
            employeeRecord.setActionNote(employee.getLeaveNote());
            employeeRecord.setOperator(employee.getLeaveOperator());
        }

        if (employee.getDefaultDepartment() != null) {
            employeeRecord.setDepartmentId(employee.getDefaultDepartment().getId());
            employeeRecord.setDepartmentName(employee.getDefaultDepartment().getName());
        }

        employeeRecord.setDomainAdmin(employee.getDomainAdmin());
        employeeRecord.setInterview(employee.getInterview());

        return employeeRecord;
    }
}
