package com.bcxin.tenant.domain.repository.customize;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.CacheProvider;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.components.models.DataCacheItem;
import com.bcxin.Infrastructures.enums.ProcessedStatus;
import com.bcxin.tenant.domain.configs.EnvConfig;
import com.bcxin.tenant.domain.entities.*;
import com.bcxin.tenant.domain.enums.ImportedDataCategory;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.repositories.*;
import com.bcxin.tenant.domain.repositories.composites.BatchImportDepart;
import com.bcxin.tenant.domain.repositories.composites.BatchImportDepartResult;
import com.bcxin.tenant.domain.repositories.composites.EmployeeImportWrapper;
import com.bcxin.tenant.domain.services.commands.BatchImportEmployeeCommand;
import com.bcxin.tenant.domain.services.commands.BatchLeaveEmployeeExcelCommand;
import com.bcxin.tenant.domain.services.commands.contracts.BatchImportContractCommand;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.persistence.EntityManager;

@Slf4j
@Repository
public class CompositeDataRepositoryImpl implements CompositeDataRepository {
    private final EntityManager entityManager;
    private final DepartmentRepository departmentRepository;
    private final UnitWork unitWork;
    private final UniqueDataConstraintRepository uniqueDataConstraintRepository;
    private final EmployeeRepository employeeRepository;
    private final JsonProvider jsonProvider;
    private final ImportDataEntityRepository importDataEntityRepository;
    private final RetryProvider retryProvider;
    private final ThreadPoolTaskExecutor taskExecutor;
    private final CacheProvider cacheProvider;
    private final TenantDbReader tenantDbReader;
    private final EnvConfig envConfig;
    public CompositeDataRepositoryImpl(EntityManager entityManager,
                                       DepartmentRepository departmentRepository,
                                       UnitWork unitWork,
                                       UniqueDataConstraintRepository uniqueDataConstraintRepository,
                                       EmployeeRepository employeeRepository,
                                       JsonProvider jsonProvider,
                                       ImportDataEntityRepository importDataEntityRepository,
                                       RetryProvider retryProvider, ThreadPoolTaskExecutor taskExecutor,
                                       CacheProvider cacheProvider,
                                       TenantDbReader tenantDbReader,
                                       EnvConfig envConfig) {
        this.entityManager = entityManager;
        this.departmentRepository = departmentRepository;
        this.unitWork = unitWork;
        this.uniqueDataConstraintRepository = uniqueDataConstraintRepository;
        this.employeeRepository = employeeRepository;
        this.jsonProvider = jsonProvider;
        this.importDataEntityRepository = importDataEntityRepository;
        this.retryProvider = retryProvider;
        this.taskExecutor = taskExecutor;
        this.cacheProvider = cacheProvider;
        this.tenantDbReader = tenantDbReader;
        this.envConfig = envConfig;
    }

    @Override
    public BatchImportDepartResult execute(BatchImportDepart data) {
        return null;
    }

    @Override
    public String execute(OrganizationEntity organization, BatchImportEmployeeCommand command) {
        ImportDataEntity importDataEntity = ImportDataEntity.create(
                organization.getId(),
                command.getPath(),
                ImportedDataCategory.Employee);
        for (BatchImportEmployeeCommand.EmployeeCommandItem commandItem : command.getCommandItems()) {
            String originalData = jsonProvider.getJson(commandItem.getDataItem());
            if (commandItem.IsValid()) {
                importDataEntity.addDataItem(originalData);
            } else {
                importDataEntity.addProcessedDataItem(
                        originalData,
                        ProcessedStatus.Error,
                        commandItem.getErrorResult());
            }
        }

        //存储所有的数据a
        retryProvider.execute(() -> {
            this.unitWork.executeTran(() -> {
                this.importDataEntityRepository.save(importDataEntity);
            });
        }, 10);


        //return importDataEntity;
        this.cacheProvider.put(importDataEntity.getId(),
                DataCacheItem.create(EmployeeImportWrapper.create(organization, importDataEntity)));

        return importDataEntity.getId();
    }

    @Override
    public String execute(OrganizationEntity organization, BatchImportContractCommand command) {
        ImportDataEntity importDataEntity = ImportDataEntity.create(
                organization.getId(),
                command.getPath(),
                ImportedDataCategory.Contract);

        for (BatchImportContractCommand.ContractCommandItem commandItem : command.getItems()) {
            String originalData = jsonProvider.getJson(commandItem.getData());
            if (commandItem.IsValid()) {
                importDataEntity.addDataItem(originalData);
            } else {
                importDataEntity.addProcessedDataItem(
                        originalData,
                        ProcessedStatus.Error,
                        commandItem.getErrorResult());
            }
        }

        retryProvider.execute(() -> {
            this.unitWork.executeTran(() -> {
                this.importDataEntityRepository.save(importDataEntity);
            });
        }, 10);

        return importDataEntity.getId();
    }

    @Override
    public String execute(OrganizationEntity organization, BatchLeaveEmployeeExcelCommand command) {
        ImportDataEntity importDataEntity = ImportDataEntity.create(
                organization.getId(),
                command.getPath(),
                ImportedDataCategory.Contract);

        for (BatchLeaveEmployeeExcelCommand.EmployeeCommandItem commandItem : command.getCommandItems()) {
            String originalData = jsonProvider.getJson(commandItem.getDataItem());
            if (StringUtils.hasLength(commandItem.getDataItem().getErrMsg())){
                importDataEntity.addProcessedDataItem(
                        originalData,
                        ProcessedStatus.Error,
                        commandItem.getDataItem().getErrMsg());
            }else if (commandItem.IsValid()) {
                importDataEntity.addDataItem(originalData);
            } else {
                importDataEntity.addProcessedDataItem(
                        originalData,
                        ProcessedStatus.Error,
                        commandItem.getErrorResult());
            }
        }

        retryProvider.execute(() -> {
            this.unitWork.executeTran(() -> {
                this.importDataEntityRepository.save(importDataEntity);
            });
        }, 10);

        return importDataEntity.getId();
    }
}
