package com.bcxin.tenant.backend.configs;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.domain.entities.TenantUserEntity;
import com.bcxin.tenant.domain.services.TenantUserService;
import com.bcxin.tenant.domain.services.commands.BatchCreateTenantUserCommand;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.kafka.support.Acknowledgment;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

/**
 * ExamEnrollmentUserKafkaComponent 单元测试
 */
@ExtendWith(MockitoExtension.class)
@DisplayName("ExamEnrollmentUserKafkaComponent 单元测试")
class ExamEnrollmentUserKafkaComponentTest {

    @Mock
    private TenantUserService userService;

    @Mock
    private JsonProvider jsonProvider;

    @Mock
    private Acknowledgment acknowledgment;

    @InjectMocks
    private ExamEnrollmentUserKafkaComponent component;

    private ConsumerRecord<String, String> createRecord;
    private ConsumerRecord<String, String> updateRecord;
    private ConsumerRecord<String, String> deleteRecord;
    private String validKafkaMessage;
    private String validSysDynamicJson;

    @BeforeEach
    void setUp() {
        // 准备有效的sys_dynamic_json
        validSysDynamicJson = "{\"items\":{\"name\":\"测试用户\",\"telephone\":\"13800138000\",\"subjectNumber\":\"2222\",\"credential_type\":\"IdCard\"},\"pageId\":\"1185616470190931968\",\"version\":3,\"businessType\":\"ExamEnrollmentCriteria\"}";

        // 准备新增操作的Kafka消息（before为null，after不为null）
        validKafkaMessage = "{\"before\":null,\"after\":{\"id\":\"1186310137083936770\",\"page_id\":\"1185616470190931968\",\"credential_number\":\"222221112\",\"name\":\"测试用户\",\"telephone\":\"13800138000\",\"sex\":\"1\",\"birthdate\":null,\"subject_number\":\"2222\",\"attachment\":null,\"domain_id\":\"428564963670360064\",\"sys_dynamic_json\":\"" + validSysDynamicJson.replace("\"", "\\\"") + "\",\"last_updater_id\":\"#1\",\"creator_id\":\"003z7uXv1\",\"creator_name\":\"平台运营超管\",\"last_updated_time\":1763041587000,\"created_time\":1763033643000},\"source\":{\"version\":\"1.9.7.Final\",\"connector\":\"mysql\",\"name\":\"mysql_binlog_source\",\"ts_ms\":1763013411000,\"snapshot\":\"false\",\"db\":\"tms\",\"sequence\":null,\"table\":\"bus_exam_enrollment_users\",\"server_id\":1101932253,\"gtid\":\"4251265f-9279-11ee-8a88-fa163e45afbd:123341575\",\"file\":\"mysql-bin.246923\",\"pos\":289280,\"row\":0,\"thread\":74080137,\"query\":null},\"op\":\"c\",\"ts_ms\":1763013204321,\"transaction\":null}";

        createRecord = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 2L, "1186310137083936770", validKafkaMessage);

        // 准备更新操作的Kafka消息（before不为null，after不为null）
        String updateMessage = "{\"before\":{\"id\":\"1186310137083936770\",\"name\":\"旧名称\"},\"after\":{\"id\":\"1186310137083936770\",\"name\":\"新名称\",\"telephone\":\"13800138000\",\"credential_number\":\"222221112\"},\"op\":\"u\"}";
        updateRecord = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 3L, "1186310137083936770", updateMessage);

        // 准备删除操作的Kafka消息（before不为null，after为null）
        String deleteMessage = "{\"before\":{\"id\":\"1186310137083936770\",\"name\":\"测试用户\"},\"after\":null,\"op\":\"d\"}";
        deleteRecord = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 4L, "1186310137083936770", deleteMessage);
    }

    @Test
    @DisplayName("测试处理新增操作 - 成功创建用户")
    void testAckExamEnrollmentUsersListener_CreateOperation_Success() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(createRecord);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("id", "1186310137083936770");
        afterObject.put("name", "测试用户");
        afterObject.put("telephone", "13800138000");
        afterObject.put("credential_number", "222221112");
        afterObject.put("sys_dynamic_json", validSysDynamicJson);
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        JSONObject dynamicJsonObject = new JSONObject();
        JSONObject itemsObject = new JSONObject();
        itemsObject.put("credential_type", "IdCard");
        dynamicJsonObject.put("items", itemsObject);

        when(jsonProvider.toObject(JSONObject.class, validKafkaMessage)).thenReturn(dataObject);
        when(jsonProvider.toObject(JSONObject.class, validSysDynamicJson)).thenReturn(dynamicJsonObject);

        Collection<TenantUserEntity> createdUsers = new ArrayList<>();
        // 使用mock对象，因为TenantUserEntity没有公共构造函数
        TenantUserEntity user = mock(TenantUserEntity.class);
        createdUsers.add(user);
        when(userService.create(any(BatchCreateTenantUserCommand.class))).thenReturn(createdUsers);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        // jsonProvider.toObject 会被调用2次：一次在过滤记录时（第66行），一次在convertToCommandItem时（第133行）
        verify(jsonProvider, atLeast(1)).toObject(JSONObject.class, validKafkaMessage);
        verify(jsonProvider, times(1)).toObject(JSONObject.class, validSysDynamicJson);

        ArgumentCaptor<BatchCreateTenantUserCommand> commandCaptor = ArgumentCaptor.forClass(BatchCreateTenantUserCommand.class);
        verify(userService, times(1)).create(commandCaptor.capture());

        BatchCreateTenantUserCommand capturedCommand = commandCaptor.getValue();
        assertNotNull(capturedCommand);
        assertEquals(1, capturedCommand.getItems().size());

        BatchCreateTenantUserCommand.TenantUserCommandItem item = capturedCommand.getItems().iterator().next();
        assertEquals("测试用户", item.getName());
        assertEquals("13800138000", item.getTelephone());
        assertEquals("222221112", item.getCredentialNumber());
        assertEquals(CredentialType.IdCard, item.getCredentialType());

        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试处理更新操作 - 应跳过")
    void testAckExamEnrollmentUsersListener_UpdateOperation_Skip() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(updateRecord);

        JSONObject dataObject = new JSONObject();
        JSONObject beforeObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        beforeObject.put("id", "1186310137083936770");
        afterObject.put("id", "1186310137083936770");
        dataObject.put("before", beforeObject);
        dataObject.put("after", afterObject);

        when(jsonProvider.toObject(JSONObject.class, updateRecord.value())).thenReturn(dataObject);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试处理删除操作 - 应跳过")
    void testAckExamEnrollmentUsersListener_DeleteOperation_Skip() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(deleteRecord);

        JSONObject dataObject = new JSONObject();
        JSONObject beforeObject = new JSONObject();
        beforeObject.put("id", "1186310137083936770");
        dataObject.put("before", beforeObject);
        dataObject.put("after", null);

        when(jsonProvider.toObject(JSONObject.class, deleteRecord.value())).thenReturn(dataObject);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试处理空记录列表")
    void testAckExamEnrollmentUsersListener_EmptyRecords() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, never()).acknowledge();
    }

    @Test
    @DisplayName("测试处理null记录列表")
    void testAckExamEnrollmentUsersListener_NullRecords() {
        // Given
        List<ConsumerRecord<String, String>> records = null;

        // When
        // CollectionUtils.isEmpty(null) 返回 true，不会抛出异常
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        // 验证不会调用userService，也不会acknowledge
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, never()).acknowledge();
    }

    @Test
    @DisplayName("测试必填字段验证 - 姓名为空")
    void testAckExamEnrollmentUsersListener_MissingName() {
        // Given
        String messageWithoutName = "{\"before\":null,\"after\":{\"id\":\"1186310137083936770\",\"telephone\":\"13800138000\",\"credential_number\":\"222221112\"},\"op\":\"c\"}";
        ConsumerRecord<String, String> record = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 5L, "1186310137083936770", messageWithoutName);
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(record);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("telephone", "13800138000");
        afterObject.put("credential_number", "222221112");
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        when(jsonProvider.toObject(JSONObject.class, messageWithoutName)).thenReturn(dataObject);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试必填字段验证 - 手机号为空")
    void testAckExamEnrollmentUsersListener_MissingTelephone() {
        // Given
        String messageWithoutTelephone = "{\"before\":null,\"after\":{\"id\":\"1186310137083936770\",\"name\":\"测试用户\",\"credential_number\":\"222221112\"},\"op\":\"c\"}";
        ConsumerRecord<String, String> record = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 6L, "1186310137083936770", messageWithoutTelephone);
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(record);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("name", "测试用户");
        afterObject.put("credential_number", "222221112");
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        when(jsonProvider.toObject(JSONObject.class, messageWithoutTelephone)).thenReturn(dataObject);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试必填字段验证 - 证件号为空")
    void testAckExamEnrollmentUsersListener_MissingCredentialNumber() {
        // Given
        String messageWithoutCredentialNumber = "{\"before\":null,\"after\":{\"id\":\"1186310137083936770\",\"name\":\"测试用户\",\"telephone\":\"13800138000\"},\"op\":\"c\"}";
        ConsumerRecord<String, String> record = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 7L, "1186310137083936770", messageWithoutCredentialNumber);
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(record);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("name", "测试用户");
        afterObject.put("telephone", "13800138000");
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        when(jsonProvider.toObject(JSONObject.class, messageWithoutCredentialNumber)).thenReturn(dataObject);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        verify(userService, never()).create(any(BatchCreateTenantUserCommand.class));
        verify(acknowledgment, times(1)).acknowledge();
    }

    @Test
    @DisplayName("测试从sys_dynamic_json提取credential_type - 成功")
    void testExtractCredentialType_Success() {
        // Given
        JSONObject dynamicJsonObject = new JSONObject();
        JSONObject itemsObject = new JSONObject();
        itemsObject.put("credential_type", "IdCard");
        dynamicJsonObject.put("items", itemsObject);

        when(jsonProvider.toObject(JSONObject.class, validSysDynamicJson)).thenReturn(dynamicJsonObject);

        // When
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(createRecord);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("name", "测试用户");
        afterObject.put("telephone", "13800138000");
        afterObject.put("credential_number", "222221112");
        afterObject.put("sys_dynamic_json", validSysDynamicJson);
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        when(jsonProvider.toObject(JSONObject.class, validKafkaMessage)).thenReturn(dataObject);

        Collection<TenantUserEntity> createdUsers = new ArrayList<>();
        // 使用mock对象，因为TenantUserEntity没有公共构造函数
        TenantUserEntity user = mock(TenantUserEntity.class);
        createdUsers.add(user);
        when(userService.create(any(BatchCreateTenantUserCommand.class))).thenReturn(createdUsers);

        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        ArgumentCaptor<BatchCreateTenantUserCommand> commandCaptor = ArgumentCaptor.forClass(BatchCreateTenantUserCommand.class);
        verify(userService, times(1)).create(commandCaptor.capture());

        BatchCreateTenantUserCommand capturedCommand = commandCaptor.getValue();
        BatchCreateTenantUserCommand.TenantUserCommandItem item = capturedCommand.getItems().iterator().next();
        assertEquals(CredentialType.IdCard, item.getCredentialType());
    }

    @Test
    @DisplayName("测试从sys_dynamic_json提取credential_type - 使用默认值")
    void testExtractCredentialType_UseDefault() {
        // Given
        String messageWithoutCredentialType = "{\"before\":null,\"after\":{\"id\":\"1186310137083936770\",\"name\":\"测试用户\",\"telephone\":\"13800138000\",\"credential_number\":\"222221112\",\"sys_dynamic_json\":\"{}\"},\"op\":\"c\"}";
        ConsumerRecord<String, String> record = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 8L, "1186310137083936770", messageWithoutCredentialType);
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(record);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("name", "测试用户");
        afterObject.put("telephone", "13800138000");
        afterObject.put("credential_number", "222221112");
        afterObject.put("sys_dynamic_json", "{}");
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        JSONObject emptyDynamicJson = new JSONObject();

        when(jsonProvider.toObject(JSONObject.class, messageWithoutCredentialType)).thenReturn(dataObject);
        when(jsonProvider.toObject(JSONObject.class, "{}")).thenReturn(emptyDynamicJson);

        Collection<TenantUserEntity> createdUsers = new ArrayList<>();
        // 使用mock对象，因为TenantUserEntity没有公共构造函数
        TenantUserEntity user = mock(TenantUserEntity.class);
        createdUsers.add(user);
        when(userService.create(any(BatchCreateTenantUserCommand.class))).thenReturn(createdUsers);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        ArgumentCaptor<BatchCreateTenantUserCommand> commandCaptor = ArgumentCaptor.forClass(BatchCreateTenantUserCommand.class);
        verify(userService, times(1)).create(commandCaptor.capture());

        BatchCreateTenantUserCommand capturedCommand = commandCaptor.getValue();
        BatchCreateTenantUserCommand.TenantUserCommandItem item = capturedCommand.getItems().iterator().next();
        // 应该使用默认值IdCard
        assertEquals(CredentialType.IdCard, item.getCredentialType());
    }

    @Test
    @DisplayName("测试处理异常 - userService抛出异常")
    void testAckExamEnrollmentUsersListener_ServiceException() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(createRecord);

        JSONObject dataObject = new JSONObject();
        JSONObject afterObject = new JSONObject();
        afterObject.put("name", "测试用户");
        afterObject.put("telephone", "13800138000");
        afterObject.put("credential_number", "222221112");
        afterObject.put("sys_dynamic_json", validSysDynamicJson);
        dataObject.put("before", null);
        dataObject.put("after", afterObject);

        JSONObject dynamicJsonObject = new JSONObject();
        JSONObject itemsObject = new JSONObject();
        itemsObject.put("credential_type", "IdCard");
        dynamicJsonObject.put("items", itemsObject);

        when(jsonProvider.toObject(JSONObject.class, validKafkaMessage)).thenReturn(dataObject);
        when(jsonProvider.toObject(JSONObject.class, validSysDynamicJson)).thenReturn(dynamicJsonObject);
        when(userService.create(any(BatchCreateTenantUserCommand.class))).thenThrow(new RuntimeException("服务异常"));

        // When & Then
        assertThrows(BadTenantException.class, () -> {
            component.ackExamEnrollmentUsersListener(records, acknowledgment);
        });

        verify(acknowledgment, never()).acknowledge();
    }

    @Test
    @DisplayName("测试批量处理多条新增记录")
    void testAckExamEnrollmentUsersListener_MultipleCreateRecords() {
        // Given
        List<ConsumerRecord<String, String>> records = new ArrayList<>();
        records.add(createRecord);

        // 创建第二条记录
        String message2 = "{\"before\":null,\"after\":{\"id\":\"1186310137083936771\",\"name\":\"测试用户2\",\"telephone\":\"13800138001\",\"credential_number\":\"222221113\",\"sys_dynamic_json\":\"" + validSysDynamicJson.replace("\"", "\\\"") + "\"},\"op\":\"c\"}";
        ConsumerRecord<String, String> record2 = new ConsumerRecord<>("tms.binlog-cdc.topic.v2.bus_exam_enrollment_users", 0, 9L, "1186310137083936771", message2);
        records.add(record2);

        JSONObject dataObject1 = new JSONObject();
        JSONObject afterObject1 = new JSONObject();
        afterObject1.put("name", "测试用户");
        afterObject1.put("telephone", "13800138000");
        afterObject1.put("credential_number", "222221112");
        afterObject1.put("sys_dynamic_json", validSysDynamicJson);
        dataObject1.put("before", null);
        dataObject1.put("after", afterObject1);

        JSONObject dataObject2 = new JSONObject();
        JSONObject afterObject2 = new JSONObject();
        afterObject2.put("name", "测试用户2");
        afterObject2.put("telephone", "13800138001");
        afterObject2.put("credential_number", "222221113");
        afterObject2.put("sys_dynamic_json", validSysDynamicJson);
        dataObject2.put("before", null);
        dataObject2.put("after", afterObject2);

        JSONObject dynamicJsonObject = new JSONObject();
        JSONObject itemsObject = new JSONObject();
        itemsObject.put("credential_type", "IdCard");
        dynamicJsonObject.put("items", itemsObject);

        when(jsonProvider.toObject(JSONObject.class, validKafkaMessage)).thenReturn(dataObject1);
        when(jsonProvider.toObject(JSONObject.class, message2)).thenReturn(dataObject2);
        when(jsonProvider.toObject(JSONObject.class, validSysDynamicJson)).thenReturn(dynamicJsonObject);

        Collection<TenantUserEntity> createdUsers = new ArrayList<>();
        TenantUserEntity user1 = new TenantUserEntity();
        TenantUserEntity user2 = new TenantUserEntity();
        createdUsers.add(user1);
        createdUsers.add(user2);
        when(userService.create(any(BatchCreateTenantUserCommand.class))).thenReturn(createdUsers);

        // When
        component.ackExamEnrollmentUsersListener(records, acknowledgment);

        // Then
        ArgumentCaptor<BatchCreateTenantUserCommand> commandCaptor = ArgumentCaptor.forClass(BatchCreateTenantUserCommand.class);
        verify(userService, times(1)).create(commandCaptor.capture());

        BatchCreateTenantUserCommand capturedCommand = commandCaptor.getValue();
        assertEquals(2, capturedCommand.getItems().size());

        verify(acknowledgment, times(1)).acknowledge();
    }
}

