package com.bcxin.identity.domains.entities;

import com.bcxin.Infrastructures.IdWorker;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.entities.EntityAbstract;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.exceptions.ForbidTenantException;
import com.bcxin.Infrastructures.exceptions.NotAllowedTenantException;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.identity.domains.components.PasswordEncoder;
import com.bcxin.identity.domains.enums.AlgorithmType;
import com.bcxin.identity.domains.enums.EventAction;
import com.bcxin.identity.domains.events.IdentityUser_WechatBindEvent;
import com.bcxin.identity.domains.exceptions.IdentityNotFoundException;
import com.bcxin.identity.domains.snapshots.WeChatSnapShoot;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@Table(name = "identity_user")
@Entity
@Getter
@Setter(AccessLevel.PROTECTED)
public class IdentityUserEntity extends EntityAbstract implements IAggregate {
    @Id
    @Column(length = 100)
    private String id;

    @Column(name = "name", length = 30, nullable = false)
    private String name;

    @Column(name = "telephone", length = 50, nullable = false)
    private String telephone;

    @Column(name = "idnum", length = 50)
    private String idNum;

    @Column(name = "created_time", nullable = false)
    private Timestamp createdTime;

    @OneToMany(mappedBy = "identityUser", cascade = CascadeType.ALL,orphanRemoval = true, fetch = FetchType.LAZY)
    private Collection<PrincipalAbstract> principals;

    @OneToMany(mappedBy = "identityUser",cascade = CascadeType.ALL, fetch = FetchType.LAZY)
    private Collection<IdentityUserEventEntity> identityUserEvents;

    @Column(name = "note",length = 2000)
    private String note;

    @Column(name = "we_chat_json")
    private String weChatJson;
    /**
     * 数据库唯一约束
     * 用于关联租户用户信息
     */
    @Column(name = "tenant_user_id",length = 50)
    private String tenantUserId;

    public void addUsernamePasswordPrincipal(String username, String plainPassword, PasswordEncoder passwordEncoder) {
        Collection<PrincipalAbstract> principalAbstracts = this.getPrincipals();

        long principalId =
                TenantContext.getInstance()
                        .resolve(IdWorker.class)
                        .nextId();

        AlgorithmType algorithmType = AlgorithmType.SM2;
        String password = passwordEncoder.encode(algorithmType, plainPassword);
        principalAbstracts.add(
                UserNamePasswordPrincipalEntity.create(this, String.valueOf(principalId), username, algorithmType, password)
        );

        this.setPrincipals(principalAbstracts);
    }

    public void addWechatCgiBin(String openId,String unionId) {
        Collection<PrincipalAbstract> principalAbstracts = this.getPrincipals();
        long principalId =
                TenantContext.getInstance()
                        .resolve(IdWorker.class)
                        .nextId();
        principalAbstracts.add(
                WechatPrincipalEntity.create(this, String.valueOf(principalId), openId, this.getName(), unionId)
        );

        this.setPrincipals(principalAbstracts);
    }

    protected IdentityUserEntity() {

    }

    public static IdentityUserEntity create(String tenantUserId, String name, String telephone,String idNum) {
        IdentityUserEntity identityUser = new IdentityUserEntity();
        long id = TenantContext.getInstance()
                .resolve(IdWorker.class)
                .nextId();

        identityUser.setId(String.valueOf(id));
        identityUser.setName(name);
        identityUser.setTelephone(telephone);
        identityUser.setIdNum(idNum);
        identityUser.setTenantUserId(tenantUserId);

        identityUser.setCreatedTime(Timestamp.from(Instant.now()));
        identityUser.setPrincipals(new ArrayList<>());

        return identityUser;
    }


    public void changePassword(
            String userName,
            PasswordEncoder passwordEncoder,
                               String oldPassword, String newPassword,
                               String confirmPassword) {
        UserNamePasswordPrincipalEntity userNamePasswordPrincipal = this.getByUserName(userName);
        if (userNamePasswordPrincipal == null) {
            throw new IdentityNotFoundException();
        }

        userNamePasswordPrincipal.changePassword(passwordEncoder, oldPassword, newPassword, confirmPassword);
    }

    public void resetPassword(
            String userName,
            PasswordEncoder passwordEncoder,
            String newPassword,
            String confirmPassword,
            boolean isAdminReset) {
        Collection<PrincipalAbstract> principals = this.getPrincipals();
        if (principals == null) {
            throw new IdentityNotFoundException();
        }

        UserNamePasswordPrincipalEntity userNamePasswordPrincipal = this.getByUserName(userName);
        if (userNamePasswordPrincipal == null) {
            throw new IdentityNotFoundException();
        }

        userNamePasswordPrincipal.resetPassword(passwordEncoder, newPassword, confirmPassword, isAdminReset);
    }

    public void bindWechat(JsonProvider jsonProvider,String openId, String nicky, String unionId) {
        Collection<PrincipalAbstract> principals = this.getPrincipals().stream().collect(Collectors.toList());
        if (principals == null) {
            principals = new ArrayList<>();
        }
        WechatPrincipalEntity selectedWechatPrincipal;
        Optional<PrincipalAbstract> principalOptional =
                principals.stream().filter(ii -> ii instanceof WechatPrincipalEntity).findFirst();
        if (principalOptional.isPresent()) {
            selectedWechatPrincipal = (WechatPrincipalEntity) principalOptional.get();
            if (!selectedWechatPrincipal.getOpenId().equals(openId)) {
                //如果已经绑定过微信，需要判断，是否同一个微信绑定
                throw new NotAllowedTenantException("该账号已经绑定微信信息, 清解绑后重试");
            }
            selectedWechatPrincipal.setOpenId(openId);
            selectedWechatPrincipal.setNicky(nicky);
            selectedWechatPrincipal.setUnionId(unionId);
        } else {
            selectedWechatPrincipal = WechatPrincipalEntity.create(this, openId, nicky, unionId);
            principals.add(selectedWechatPrincipal);
        }
        this.setPrincipals(principals);
        WeChatSnapShoot weChatSnapShoot = WeChatSnapShoot.create(unionId, nicky);
        this.setWeChatJson(jsonProvider.getJson(weChatSnapShoot));
        this.recordUserEvent(EventAction.BIND_WECHAT, openId, selectedWechatPrincipal);
    }

    public void unbindWechat() {
        Collection<PrincipalAbstract> principals = this.getPrincipals().stream().collect(Collectors.toList());
        if (principals == null || principals.size()==0) {
            throw new NotAllowedTenantException("该账号未绑定微信, 无须解绑操作");
        }

        Optional<PrincipalAbstract> principalOptional =
                principals.stream().filter(ii -> ii.getClass().isAssignableFrom(WechatPrincipalEntity.class)).findFirst();
        if (!principalOptional.isPresent()) {
            throw new NotAllowedTenantException("该账号未绑定微信, 无须解绑操作");
        }

        WechatPrincipalEntity wechatPrincipalEntity = (WechatPrincipalEntity) principalOptional.get();
        principals.remove(wechatPrincipalEntity);

        this.setPrincipals(principals);
        /**
         * 添加解绑事件k
         */
        this.recordUserEvent(EventAction.UNBIND_WECHAT, wechatPrincipalEntity.getOpenId(), null);
        this.setWeChatJson(null);
        this.recordEvent(IdentityUser_WechatBindEvent.create(this.getTenantUserId(), null, null));
    }

    public void change(String name ,String telephone,String idNum,boolean requiredPhoneAsLoginName) {
        this.setName(name);
        this.setTelephone(telephone);
        this.setIdNum(idNum);
        Collection<PrincipalAbstract> principals = this.getPrincipals();
        if (principals == null) {
            principals = new ArrayList<>();
        }
        Optional<UserNamePasswordPrincipalEntity> userNamePasswordOptional =
                principals.stream().filter(ii -> ii instanceof UserNamePasswordPrincipalEntity)
                        .map(ii -> (UserNamePasswordPrincipalEntity) ii)
                        .findFirst();
        if (userNamePasswordOptional.isPresent()) {
            userNamePasswordOptional.get().changeUserName(requiredPhoneAsLoginName ? telephone:idNum);
        }
    }

    public <T> void recordUserEvent(EventAction eventAction, T data, PrincipalAbstract principal) {
        /*
        String eventData =
                TenantContext.getInstance().resolve(JsonProvider.class)
                        .getJson(data);

        IdentityUserEventEntity userEventEntity = IdentityUserEventEntity
                .create(eventAction, eventData, this, principal);

        Collection<IdentityUserEventEntity> identityUserEventEntities = this.getIdentityUserEvents();
        if (identityUserEventEntities == null) {
            identityUserEventEntities = new ArrayList<>();
        }

        identityUserEventEntities.add(userEventEntity);

        this.setIdentityUserEvents(identityUserEventEntities);

         */
    }

    public void changeNote(String note) {
        this.setNote(note);
    }

    private UserNamePasswordPrincipalEntity getByUserName(String userName) {
        Collection<PrincipalAbstract> principals = this.getPrincipals();
        if (principals == null) {
            throw new IdentityNotFoundException();
        }

        Optional<UserNamePasswordPrincipalEntity> userNamePasswordPrincipalOptional =
                principals.stream().filter(ii -> ii instanceof UserNamePasswordPrincipalEntity)
                        .map(ii -> (UserNamePasswordPrincipalEntity) ii)
                        .filter(ii -> StringUtils.equalsIgnoreCase(ii.getUserName(), userName))
                        .findFirst();

        if (userNamePasswordPrincipalOptional.isPresent()) {
            return userNamePasswordPrincipalOptional.get();
        }

        return null;
    }

    public void changeUserName(String userName,PasswordEncoder passwordEncoder,String currentPassword) {
        Optional<UserNamePasswordPrincipalEntity> userNamePasswordPrincipalOptional =
                principals.stream().filter(ii -> ii instanceof UserNamePasswordPrincipalEntity)
                        .map(ii -> (UserNamePasswordPrincipalEntity) ii)
                        .findFirst();
        if (!userNamePasswordPrincipalOptional.isPresent()) {
            throw new NotFoundTenantException();
        }

        UserNamePasswordPrincipalEntity userNamePasswordPrincipal = userNamePasswordPrincipalOptional.get();
        if (!passwordEncoder.isMatched(userNamePasswordPrincipal.getPassword(),currentPassword)) {
            throw new ForbidTenantException("原密码不正确");
        }

        userNamePasswordPrincipal.changeUserName(userName);
    }
}
