package com.bcxin.autodownloadupload.common.utils;

import cn.hutool.core.io.FileUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * description: 文件工具类
 * author: linchunpeng
 * date:  2023-04-18 15:15
 */
@Slf4j
public class FileUtils {


    /**
     * description: 根据url下载文件
     * author: linchunpeng
     * date:  2023-04-18 15:15
     */
    public static String downloadByUrl(String urlStr, String savePath) throws Exception {
        InputStream inputStream = null;
        FileOutputStream fos = null;
        try {
            URL url = new URL(urlStr);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            //设置超时间为5秒
            conn.setConnectTimeout(5 * 1000);
            //防止屏蔽程序抓取而返回403错误
            conn.setRequestProperty("User-Agent", "Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt)");
            //设置部分请求头信息，根据自己的实际需要来书写，不需要的也可以删掉
            conn.setRequestProperty("api_token", "Bearer_");
            conn.setRequestProperty("Cookie", "XXL_JOB_LOGIN_IDENTITY=");
            //得到输入流
            inputStream = conn.getInputStream();
            //获取自己数组
            byte[] getData = readInputStream(inputStream);
            //文件保存位置
            if (!FileUtil.exist(savePath)) {
                log.info("目录：{}，不存在，创建", savePath);
                FileUtil.mkdir(savePath);
                if (FileUtil.exist(savePath)) {
                    log.info("创建成功");
                } else {
                    log.info("创建失败");
                }
            }

            //获取文件名及后缀
            String headerField = conn.getHeaderField("Content-Disposition");
            String[] split = headerField.split("\"");
            String fileName = URLDecoder.decode(split[1], "utf-8");
            String fileSavePath = savePath + File.separator + fileName;
            //开始写入
            File file = new File(fileSavePath);
            fos = new FileOutputStream(file);
            fos.write(getData);
            log.info("文件: {}，下载成功", urlStr);
            return fileSavePath;
        } catch (Exception e) {
            log.error("文件：{}，下载异常", urlStr, e);
            throw e;
        } finally {
            try {
                if (inputStream != null) inputStream.close();
                if (fos != null) fos.close();
            } catch (Exception e) {
                log.error("文件：{}，关闭流异常", urlStr, e);
            }
        }
    }

    /**
     * 根据url地址下载文件
     * @param fileUrl 文件地址
     * @param localFilePath 保存本地路径
     */
    public static boolean downloadFile(String fileUrl, String localFilePath, String serverUrl, String obsUrl) {
        // 提取日期并判断是否在两年内
        LocalDate extractedDate = extractDateFromUrl(fileUrl);
        if (extractedDate == null) {
            log.info("文件：{}，无法提取日期，继续下载", fileUrl);
            // 无法提取日期时，继续下载
        } else if (!isWithinTwoYears(extractedDate)) {
            log.info("文件：{}，日期：{}，不在最近两年内，跳过下载", fileUrl, extractedDate);
            return true; // 跳过下载返回true，以便记录日志
        }
        
        URL urlfile = null;
        HttpResponse executeAsync = null;
        try {
            if (fileUrl.startsWith("/upload")) {
                if (serverUrl.contains("v5qy.baibaodun.cn")) {
                    //互联网
                    if (fileUrl.startsWith("/uploads")) {
                        fileUrl = fileUrl.substring(8);
                    } else {
                        fileUrl = fileUrl.substring(7);
                    }
                    fileUrl = obsUrl.concat(fileUrl);
                } else {
                    fileUrl = serverUrl.concat(fileUrl);
                }
            }
            if (fileUrl.startsWith("https:/") && !fileUrl.startsWith("https://")) {
                fileUrl = fileUrl.replace("https:/", "https://");
            }
            if (fileUrl.startsWith("http:/") && !fileUrl.startsWith("http://")) {
                fileUrl = fileUrl.replace("http:/", "http://");
            }
            urlfile = new URL(fileUrl);
            String filePath = null;
            String path = urlfile.getPath();
            if (path.equals("/getResource.do")) {  //判断地址是http://bj.baibaodun.cn:8801/ars-web/getResource.do?path=upload/2020-08-28/1598607857060.zip 就截取 =后面的路径
                filePath = fileUrl.substring(fileUrl.indexOf("=") + 1);
            } else if (fileUrl.contains("=")) { //截取双路径如 http://v5qy.baibaodun.cn/obpm/v2/sync/file/download?f=https%3A%2F%2Fbcxin-v5-prod.obs.cn-north-1.myhuaweicloud.com%2Fuploads%2F2022%2F07-12%2F658979513459277824--__AOkTvsah9VLicJhQ1FO%2F1657606996828%2F%E4%BF%9D%E5%AE%89%E6%9C%8D%E5%8A%A1%E5%85%AC%E5%8F%B8%E8%AE%BE%E7%AB%8B%E5%88%86%E5%85%AC%E5%8F%B8%E5%A4%87%E6%A1%88.zip
                if (fileUrl.substring(fileUrl.indexOf("=") + 1).contains("https")) {
                    filePath = fileUrl.substring(fileUrl.indexOf("=") + 1).substring(fileUrl.substring(fileUrl.indexOf("=") + 1).indexOf("com") + 3);
                } else {
                    if (fileUrl.substring(fileUrl.indexOf("=") + 1).indexOf("/") != 0) {
                        filePath = "/" + fileUrl.substring(fileUrl.indexOf("=") + 1);
                    } else {
                        filePath = fileUrl.substring(fileUrl.indexOf("=") + 1);
                    }
                }
            } else {
                filePath = path;
            }
            filePath = filePath.replaceAll("/upload/", "/uploads/");
            if (filePath.contains("/uploads/")) {
                filePath = filePath.substring(filePath.indexOf("/uploads/"));
            } else {
                filePath = "/uploads/" + filePath;
            }
            filePath = filePath.replaceAll("/+", "/");

            File f = new File(localFilePath + filePath);
            if (!f.getParentFile().exists() && !f.getParentFile().isDirectory()) {
                FileUtil.mkdir(f.getParentFile());
                //f.getParentFile().mkdirs();
            }

            HttpRequest get = HttpUtil.createGet(fileUrl, true);
            get.header("accessToken", "eyJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJhdXRoMCIsImV4cCI6MTY2Njg2MDAwMSwidXNlcm5hbWUiOiI3TkVMR1RYaSJ9.Fvx16149p8Grp-a_QwFRGro_sWD1xqL2QV5beL9bRcc");
            executeAsync = get.executeAsync();
            if (executeAsync.getStatus() != 200) {
                return false;
            }
            executeAsync.writeBody(new File(localFilePath+filePath),null);
        } catch (Exception e) {
            log.error("文件：{}，下载异常：{}", fileUrl, e.getMessage());
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (executeAsync != null) executeAsync.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return true;
    }


    /**
     * description: 判断文件是否能访问
     * author: linchunpeng
     * date:  2023-04-23 16:48
     */
    public static boolean checkFileHutool(String fileUrl) {
        HttpResponse executeAsync = null;
        try {
            if (fileUrl.startsWith("https:/")) {
                if (!fileUrl.substring(0, 8).equals("https://")) {
                    fileUrl = fileUrl.replace("https:/", "https://");
                }
            }
            HttpRequest get = HttpUtil.createGet(fileUrl, true);
            get.header("accessToken", "eyJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJhdXRoMCIsImV4cCI6MTY2Njg2MDAwMSwidXNlcm5hbWUiOiI3TkVMR1RYaSJ9.Fvx16149p8Grp-a_QwFRGro_sWD1xqL2QV5beL9bRcc");
            executeAsync = get.executeAsync();
            int status = executeAsync.getStatus();
            if (status != 200) {
                log.info("文件访问失败，访问结果：{}，文件地址：{}", status, fileUrl);
            }
            return status == 200;
        } catch (Exception e) {
            log.error("文件：{}，判断是否能访问异常：{}", fileUrl, e.getMessage());
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (executeAsync != null) executeAsync.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * description: 判断文件是否能访问
     * author: linchunpeng
     * date:  2023-04-23 16:48
     */
    public static boolean checkFile(String fileUrl) {
        try {
            URL url = new URL(fileUrl);
            HttpURLConnection urlcon = (HttpURLConnection) url.openConnection();
            urlcon.setRequestMethod("GET");
            urlcon.setRequestProperty("Content-type", "application/x-www-form-urlencoded");
            int responseCode = urlcon.getResponseCode();
            if (responseCode != 200) {
                log.info("第二次文件访问失败，访问结果：{}，文件地址：{}", responseCode, fileUrl);
            }
            return responseCode == HttpURLConnection.HTTP_OK;
        } catch (Exception e) {
            log.error("文件：{}，第二次访问异常：{}", fileUrl, e.getMessage());
            e.printStackTrace();
            return false;
        }
    }


    /**
     * description: 获取url的连接状态
     * author: linchunpeng
     * date:  2023-04-28 10:17
     */
    public static int getUrlStatus(String fileUrl) {
        HttpResponse executeAsync = null;
        try {
            HttpRequest get = HttpUtil.createGet(fileUrl, true);
            executeAsync = get.executeAsync();
            return executeAsync.getStatus();
        } catch (Exception e) {
            e.printStackTrace();
            return 404;
        } finally {
            try {
                if (executeAsync != null) executeAsync.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * description: 从输入流中获取字节数组
     * author: linchunpeng
     * date:  2023-04-18 15:33
     */
    private static byte[] readInputStream(InputStream inputStream) throws IOException {
        byte[] buffer = new byte[4 * 1024];
        int len = 0;
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        while ((len = inputStream.read(buffer)) != -1) {
            bos.write(buffer, 0, len);
        }
        bos.close();
        return bos.toByteArray();
    }

    /**
     * description: 从URL中提取日期
     * 支持两种格式：
     * 1. /upload/2020-08-28/ (yyyy-MM-dd)
     * 2. /uploads/2022/07-12/ (yyyy/MM-dd)
     * @param fileUrl 文件URL
     * @return 提取的日期，如果无法提取则返回null
     */
    private static LocalDate extractDateFromUrl(String fileUrl) {
        if (fileUrl == null || fileUrl.isEmpty()) {
            return null;
        }
        
        // 格式1: /upload/2020-08-28/ 或 /uploads/2020-08-28/
        Pattern pattern1 = Pattern.compile("/(?:upload|uploads)/(\\d{4}-\\d{2}-\\d{2})/");
        Matcher matcher1 = pattern1.matcher(fileUrl);
        if (matcher1.find()) {
            String dateStr = matcher1.group(1);
            try {
                return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            } catch (DateTimeParseException e) {
                log.warn("文件：{}，日期格式解析失败：{}", fileUrl, dateStr);
            }
        }
        
        // 格式2: /uploads/2022/07-12/ 或 /upload/2022/07-12/
        Pattern pattern2 = Pattern.compile("/(?:upload|uploads)/(\\d{4})/(\\d{2}-\\d{2})/");
        Matcher matcher2 = pattern2.matcher(fileUrl);
        if (matcher2.find()) {
            String year = matcher2.group(1);
            String monthDay = matcher2.group(2);
            String dateStr = year + "-" + monthDay;
            try {
                return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            } catch (DateTimeParseException e) {
                log.warn("文件：{}，日期格式解析失败：{}", fileUrl, dateStr);
            }
        }
        
        // 格式3: /2025/11-27/ (直接以年份开头的路径，不在upload/uploads下)
        Pattern pattern3 = Pattern.compile("/(\\d{4})/(\\d{2}-\\d{2})/");
        Matcher matcher3 = pattern3.matcher(fileUrl);
        if (matcher3.find()) {
            String year = matcher3.group(1);
            String monthDay = matcher3.group(2);
            String dateStr = year + "-" + monthDay;
            try {
                return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            } catch (DateTimeParseException e) {
                log.warn("文件：{}，日期格式解析失败：{}", fileUrl, dateStr);
            }
        }
        
        // 格式4: IDCARD_2022-07-01 或类似的前缀格式
        Pattern pattern4 = Pattern.compile("/([A-Z_]+)?(\\d{4}-\\d{2}-\\d{2})/");
        Matcher matcher4 = pattern4.matcher(fileUrl);
        if (matcher4.find()) {
            String dateStr = matcher4.group(2);
            try {
                return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            } catch (DateTimeParseException e) {
                log.warn("文件：{}，日期格式解析失败：{}", fileUrl, dateStr);
            }
        }
        
        return null;
    }

    /**
     * description: 判断日期是否在最近两年内（从当前日期往前推730天）
     * @param date 要判断的日期
     * @return 如果在两年内返回true，否则返回false
     */
    private static boolean isWithinTwoYears(LocalDate date) {
        if (date == null) {
            return false;
        }
        LocalDate twoYearsAgo = LocalDate.now().minusDays(730);
        return !date.isBefore(twoYearsAgo);
    }


    public static void responseWithFile(String realPath, HttpServletResponse response) throws IOException {
        File selectedFile = new File(realPath);
        response.setContentType("application/x-download; charset=UTF-8");
        response.setHeader("Content-Disposition", "attachment;filename=\"" + java.net.URLEncoder.encode(selectedFile.getName(), "UTF-8") + "\"");

        try (ServletOutputStream outputStream = response.getOutputStream()) {
            try (BufferedInputStream reader = new BufferedInputStream(new FileInputStream(selectedFile))) {
                byte[] buffer = new byte[4096];
                int i = -1;
                while ((i = reader.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, i);
                }
            }
            outputStream.flush();
        }
    }

    /**
     * 测试方法：测试日期提取和判断功能
     */
    public static void main(String[] args) {
        String[] testUrls = {
            "https://02obs.baibaodun.cn/2025/11-27/fafcddbb-6589-4fc5-b9bb-fb6487902193/1764194126362.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2025/11-27/8d8c4102-90f6-4727-a3a9-5d15f470cc09/upload301764194123355yasuo.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2025/11-27/df39afd5-da5e-4c05-9e26-77289551a4cd/upload671764194141717yasuo.png",
            "https://bcxin-saas-prod.obs.cn-north-1.myhuaweicloud.com/upload/2025-11-27/1764194155844.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/9904c578-b938-434e-b487-51087727fce7/1653016876717.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/5bfdc38b-5e03-4de3-b18e-85039aed089e/1653016740081.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/c9810ecf-ea51-4b2f-bd5a-efbc058f8dc9/1653016741355.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/55f0b714-209d-40e6-bde4-6d223eefcbf4/1653016755504.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/08-08/1dc8b9bc-8b61-4bcd-80d1-fb181bb1267f/upload81691501523830yasuo.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/08-08/74b3d8f3-0fec-41a6-afe1-a0e4bdc5e8ba/upload491691501541120yasuo.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/08-08/d0a1d75b-a8b4-4f32-9983-c2f7ba5f6669/1691501527682.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/08-08/28c3159e-92d7-4a88-a57f-4c315d2c5b0c/1691501590231.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/02-16/f0c1eaeb-4769-47c8-94be-d656162c3a9b/upload851676537065373yasuo.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/02-16/9bbfbc87-39fd-4aef-bb94-cada70b5164a/upload351676537043561yasuo.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/02-16/4fb9c3f7-0f96-41bd-94ad-7520a964e2cf/1676537137753.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/2023/02-16/658689e5-e7dc-4bb2-84ff-541b51208093/1676537046599.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/IDCARD_2022-07-01/aed2ca82-c01d-40d0-b06d-2e2e7b76e464/1656655848931.png",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/IDCARD_2022-07-01/0bc8b507-e8a5-46db-b3d5-2af002f21d19/1656655716164.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/IDCARD_2022-07-01/616f9563-4b3b-4904-9a01-a569ee16e9e1/1656655697660.jpg",
            "https://02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com/IDCARD_2022-07-01/488a8473-113e-4bf6-aa97-5a792ebf1444/1656655699309.jpg",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/1da42eaf-5c94-4c4a-a5d1-9b54c4c70301.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/c8850aad-f0c9-4ade-92ed-2ae0f204cd6d.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/ab093b89-68f2-4872-93ec-b11c9e08c4b8.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/792ed15a-43f8-407a-93ef-547e4dfe9acb.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/b093b15f-1028-4f54-8290-ab76b9edf709.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/589e5534-5f72-4aa7-b010-b255b65aa07a.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/8bd9ef1b-e69b-4cd3-9429-66e732219e87.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/f89101cc-1ef7-48f6-810b-5748428ad162.pdf",
            "https://bcxinsaasprod.baibaodun.cn:443/upload/common/2025-11-27/b08622de-afe3-4af5-a138-740319736ad7.pdf"
        };

        System.out.println("==========================================");
        System.out.println("日期提取和判断功能测试");
        System.out.println("==========================================");
        System.out.println("当前日期: " + LocalDate.now());
        System.out.println("两年前的日期: " + LocalDate.now().minusDays(730));
        System.out.println("==========================================\n");

        int totalCount = 0;
        int extractedCount = 0;
        int withinTwoYearsCount = 0;
        int outsideTwoYearsCount = 0;
        int noDateCount = 0;
        
        java.util.List<String> withinTwoYearsUrls = new java.util.ArrayList<>();
        java.util.List<String> outsideTwoYearsUrls = new java.util.ArrayList<>();
        java.util.List<String> noDateUrls = new java.util.ArrayList<>();

        for (String url : testUrls) {
            totalCount++;
            LocalDate extractedDate = extractDateFromUrl(url);
            
            System.out.println("URL: " + url);
            if (extractedDate != null) {
                extractedCount++;
                boolean withinTwoYears = isWithinTwoYears(extractedDate);
                System.out.println("  提取日期: " + extractedDate);
                System.out.println("  是否在两年内: " + withinTwoYears);
                
                if (withinTwoYears) {
                    withinTwoYearsCount++;
                    withinTwoYearsUrls.add(url);
                    System.out.println("  结果: ✓ 可以下载");
                } else {
                    outsideTwoYearsCount++;
                    outsideTwoYearsUrls.add(url);
                    System.out.println("  结果: ✗ 跳过下载（不在两年内）");
                }
            } else {
                noDateCount++;
                noDateUrls.add(url);
                System.out.println("  提取日期: null");
                System.out.println("  结果: ✓ 可以下载（无法提取日期，继续下载）");
            }
            System.out.println();
        }

        System.out.println("==========================================");
        System.out.println("统计结果:");
        System.out.println("  总URL数: " + totalCount);
        System.out.println("  成功提取日期: " + extractedCount);
        System.out.println("  在两年内: " + withinTwoYearsCount);
        System.out.println("  不在两年内: " + outsideTwoYearsCount);
        System.out.println("  无法提取日期: " + noDateCount);
        System.out.println("  会进行下载: " + (withinTwoYearsCount + noDateCount));
        System.out.println("  跳过下载: " + outsideTwoYearsCount);
        System.out.println("==========================================");
        
        int downloadCount = withinTwoYearsCount + noDateCount;
        System.out.println("\n==========================================");
        System.out.println("【会进行下载的URL列表】（" + downloadCount + "个）:");
        if (downloadCount == 0) {
            System.out.println("  无");
        } else {
            int index = 1;
            // 在两年内的URL
            for (String url : withinTwoYearsUrls) {
                System.out.println("  " + (index++) + ". " + url);
            }
            // 无法提取日期的URL（也会下载）
            for (String url : noDateUrls) {
                System.out.println("  " + (index++) + ". " + url);
            }
        }
        System.out.println("==========================================");
        
        System.out.println("\n【不在两年内的URL列表】（" + outsideTwoYearsCount + "个，跳过下载）:");
        if (outsideTwoYearsUrls.isEmpty()) {
            System.out.println("  无");
        } else {
            for (int i = 0; i < outsideTwoYearsUrls.size(); i++) {
                System.out.println("  " + (i + 1) + ". " + outsideTwoYearsUrls.get(i));
            }
        }
        
        System.out.println("\n==========================================");
    }

}
