package com.bcxin.runtime.domain.repositories.syncs.components;

import com.bcxin.runtime.domain.repositories.syncs.dtos.DataSetRowGroupDto;
import com.bcxin.runtime.domain.snapshoots.JdbcConnectionSnapshot;
import com.bcxin.runtime.domain.snapshoots.JdbcMapSnapshot;
import com.bcxin.runtime.domain.snapshoots.TableMapSnapshot;
import com.bcxin.runtime.domain.syncs.components.DbSyncExecutor;
import com.bcxin.runtime.domain.syncs.dtos.DataSetDto;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasNoSupportException;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

@Component
public class DbSyncExecutorImpl implements DbSyncExecutor {
    private static Map<String, DataSource> dataSourceMap = new ConcurrentHashMap<String, DataSource>();
    private final JsonProvider jsonProvider;
    private final BeanFactory beanFactory;

    public DbSyncExecutorImpl(JsonProvider jsonProvider, BeanFactory beanFactory) {
        this.jsonProvider = jsonProvider;
        this.beanFactory = beanFactory;
    }

    public void execute(JdbcMapSnapshot snapshot, Collection<DataSetDto.Row> rows,Collection<String> deletedIds) {
        assert snapshot != null;
        assert snapshot.getJdbcConnection() != null;
        DataSource dataSource = getDataSource(snapshot.getJdbcConnection());
        int pageIndex = 0;
        int pageSize = 30;
        Collection<DataSetDto.Row> pagedRows = null;
        String sqlParseBeanName = snapshot.getJdbcConnection().getSqlParserBeanName();
        if(!StringUtils.hasLength(sqlParseBeanName)) {
            throw new SaasNoSupportException(String.format("sqlParseBeanName(%s)无效!", snapshot.getJdbcConnection().getUrl()));
        }

        SqlParser sqlParser = (SqlParser) this.beanFactory.getBean(sqlParseBeanName);
        do {
            pagedRows = rows.stream().skip(pageIndex * pageSize).limit(pageSize)
                    .collect(Collectors.toList());
            pageIndex++;
            try (Connection connection = dataSource.getConnection()) {
                boolean defaultAutoCommit = connection.getAutoCommit();
                connection.setAutoCommit(false);
                try (Statement statement = connection.createStatement()) {
                    Collection<DataSetRowGroupDto> dataSetRowGroups = getDataSetRowGroups(sqlParser, statement,snapshot.getTableMap(),pagedRows);
                    //todo 根据版本进行脏数据的避免，比如: 通过LastModifyTime来进行比较
                    for (DataSetRowGroupDto rowGroup : dataSetRowGroups) {
                        Collection<String> rowSqls = sqlParser.generate(snapshot.getTableMap(), rowGroup);
                        rowSqls.forEach(sql -> {
                            try {
                                statement.addBatch(sql);
                            } catch (SQLException ext) {
                                ext.printStackTrace();

                                throw new SaasBadException(String.format("批量执行SQL发生异常->添加SQL:%s", sql), ext);
                            }
                        });
                    }
                    if (pageIndex == 1) {
                        Collection<String> deletedSqls = sqlParser.generateDeletedSql(snapshot.getTableMap(), deletedIds);

                        if (deletedSqls != null) {
                            deletedSqls.forEach(sql -> {
                                try {
                                    statement.addBatch(sql);
                                } catch (SQLException ext) {
                                    ext.printStackTrace();

                                    throw new SaasBadException(String.format("批量执行SQL发生异常->添加SQL:%s", sql), ext);
                                }
                            });
                        }
                    }

                    statement.executeBatch();
                }

                connection.commit();
                connection.setAutoCommit(defaultAutoCommit);
            } catch (Exception ex) {
                ex.printStackTrace();

                throw new SaasBadException("批量执行SQL发生异常", ex);
            }
        } while (pagedRows != null && pagedRows.size() > 0);
    }

    public DataSource getDataSource(JdbcConnectionSnapshot snapshot) {
        String dataSourceMapKey = String.format("url:%s;username:%s", snapshot.getUrl(), snapshot.getUsername());
        DataSource dataSource = dataSourceMap.get(dataSourceMapKey);
        if (dataSource == null) {
            synchronized (DbSyncExecutorImpl.class) {
                dataSource = dataSourceMap.get(dataSourceMapKey);
                if(dataSource==null) {
                    HikariDataSource hikariDataSource = new HikariDataSource();
                    hikariDataSource.setConnectionTimeout(60_0000);
                    hikariDataSource.setMinimumIdle(2);
                    hikariDataSource.setMaximumPoolSize(2000);
                    hikariDataSource.setMaxLifetime(60_0000);
                    hikariDataSource.setValidationTimeout(5_000);
                    hikariDataSource.setIdleTimeout(30_0000);

                    hikariDataSource.setDriverClassName(snapshot.getDriverClass());
                    hikariDataSource.setJdbcUrl(snapshot.getUrl());
                    hikariDataSource.setUsername(snapshot.getUsername());
                    hikariDataSource.setPassword(snapshot.getPassword());

                    dataSource = hikariDataSource;
                    dataSourceMap.put(dataSourceMapKey, dataSource);
                }
            }
        }

        return dataSource;
    }

    private static Collection<DataSetRowGroupDto> getDataSetRowGroups(SqlParser sqlParser, Statement statement,TableMapSnapshot tableMap,
                                                                      Collection<DataSetDto.Row> pagedRows) throws SQLException {
        if (pagedRows == null || pagedRows.size() == 0) {
            return Collections.emptyList();
        }

        String whereIds =
                pagedRows.stream().map(ii -> {
                    String rowId = sqlParser.getProcessedId(tableMap,ii);
                    //return String.format("'%s'", ii.getId());
                    return  String.format("'%s'", rowId);
                }).distinct().collect(Collectors.joining(","));


        String querySql = String.format("select id from %s where id in (%s)", tableMap.getTableName(), whereIds);
        ResultSet resultSet = statement.executeQuery(querySql);
        Collection<String> existsIds = new ArrayList<>();
        while (resultSet.next()) {
            existsIds.add(resultSet.getString(1));
        }
        return pagedRows.stream().map(ii -> {
            String processedId = sqlParser.getProcessedId(tableMap,ii);
            boolean isNew = !existsIds.stream().anyMatch(ix -> ix.equalsIgnoreCase(processedId));
            return DataSetRowGroupDto.create(ii, isNew);
        }).collect(Collectors.toList());
    }
}
