package com.bcxin.runtime.domain.syncs.entities;
import com.alibaba.fastjson.JSON;
import com.bcxin.runtime.domain.enums.BooleanStatus;
import com.bcxin.runtime.domain.syncs.enums.SyncProcessStatus;
import com.bcxin.saas.core.IAggregate;
import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.persistence.*;
import java.util.*;

@Getter
@Setter(AccessLevel.PROTECTED)
@Entity
@Table(name = "sync_frm_data_queues")
public class DataSyncQueueEntity implements IAggregate {
    private static final Logger logger = LoggerFactory.getLogger(DataSyncQueueEntity.class);
    private static final Collection<SyncProcessStatus> completedSyncProcessedStatuses = Arrays.asList(
            SyncProcessStatus.Done,
            SyncProcessStatus.DoneFromCallback,
            SyncProcessStatus.DoneViaFTP);
    private static final Collection<SyncProcessStatus> errorSyncProcessedStatuses = Arrays.asList(
            SyncProcessStatus.Error,
            SyncProcessStatus.ErrorFromCallback,
            SyncProcessStatus.ErrorViaFTP);

    public static final String CALLBACK_MSG_ID="msgId";
    public static final String CALLBACK_RESULT="result";
    public static final String CALLBACK_EXEC_STATUS="execStatus";

    @Id
    private String id;
    @Column(name = "name", length = 300)
    private String name;
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "created_time", nullable = false)
    private Date createdTime;
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "last_processed_time", nullable = true)
    private Date lastProcessedTime;
    @Column(nullable = false, length = 3000)
    private String data;

    @Column(name = "size", nullable = false)
    private int size;

    @Column(name = "need_sync_file", nullable = false)
    private BooleanStatus needSyncFile;

    @Enumerated(EnumType.ORDINAL)
    @Column(name = "file_sync_status", nullable = false)
    private SyncProcessStatus fileSyncStatus;

    @Column(name = "file_sync_result")
    private String fileSyncResult;

    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "last_file_processed_time")
    private Date lastFileProcessedTime;

    @Column(name = "form_sync_config", nullable = false,length = 5000)
    private String formSyncConfig;

    @Enumerated(EnumType.ORDINAL)
    @Column(nullable = false)
    private SyncProcessStatus status;

    @Column(name = "is_archived", nullable = false)
    private BooleanStatus isArchived;

    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "archived_time")
    public Date ArchivedTime;

    @Column(name = "retry_count", nullable = false)
    private int retryCount;
    @Column(length = 8000)
    private String result;

    @Column(name = "form_sync_id", nullable = false, length = 100)
    private String formSyncId;

    @Setter
    @Column(name = "short_result", length = 800)
    private String shortResult;

    @Column(name = "third_party_reference_id", length = 100)
    private String thirdPartyReferenceId;

    @Column(name = "third_party_callback",length = 2000)
    private String thirdPartyCallback;

    @Column(name = "third_party_tracking", length = 3000)
    private String thirdPartyTracking;

    public void makeArchived() {
        this.setIsArchived(BooleanStatus.TRUE);
        this.setArchivedTime(new Date());
    }

    public void changeDataSyncStatus(SyncProcessStatus status, String result) {
        this.setStatus(status);
        this.setResult(result);
        this.setLastProcessedTime(new Date());
    }

    public void changeFileSyncStatus(SyncProcessStatus status, String result) {
        this.setFileSyncStatus(status);
        this.setFileSyncResult(result);
        this.setLastFileProcessedTime(new Date());
    }

    public void refreshDate(String data) {
        this.setLastProcessedTime(new Date());
        this.setData(data);
    }

    protected DataSyncQueueEntity() {
    }

    protected DataSyncQueueEntity(String name, String data, int size, boolean needSyncFile,
                                  String formSyncId, String formSyncConfig) {
        this.setId(UUID.randomUUID().toString());
        this.setNeedSyncFile(needSyncFile ? BooleanStatus.TRUE : BooleanStatus.FALSE);
        this.setSize(size);

        this.setCreatedTime(new Date());
        this.setName(name);
        this.setData(data);
        this.setStatus(SyncProcessStatus.Initialize);
        this.setFileSyncStatus(SyncProcessStatus.Initialize);
        this.setFormSyncConfig(formSyncConfig);
        this.setFormSyncId(formSyncId);
        this.setIsArchived(BooleanStatus.FALSE);

        if (!needSyncFile) {
            this.changeFileSyncStatus(SyncProcessStatus.Done, "无需同步文件");
        }
    }

    public static DataSyncQueueEntity create(String name, String data, int size, boolean needSyncFile,
                                             String referenceId, String formSyncConfig) {
        return new DataSyncQueueEntity(name, data, size, needSyncFile, referenceId, formSyncConfig);
    }

    public void assignThirdParty(
            boolean dispatchFile,
            SyncProcessStatus status,
            String thirdPartyReferenceId,String result) {
        /*
        List<ThirdPartyHistory> thirdPartyHistories = null;
        String preMsg = "";
        try {
            thirdPartyHistories = JSON.parseArray(this.getThirdPartyTracking(), ThirdPartyHistory.class);
        } catch (Exception ex) {
            logger.error(String.format("设置(%s)assignThirdParty(%s)信息", this.getId(), thirdPartyReferenceId), ex);
            preMsg = String.format("解析getThirdPartyTracking出错:%s", ex.getMessage());
        }

        if (thirdPartyHistories == null) {
            thirdPartyHistories = new ArrayList<>();
        }

        try {
            //thirdPartyHistories.add(ThirdPartyHistory.create(thirdPartyReferenceId, status, String.format("preMsg=%s;result=%s;", preMsg, result)));
            thirdPartyHistories.add(ThirdPartyHistory.create(thirdPartyReferenceId, status, ""));
            this.setThirdPartyTracking(JSON.toJSONString(thirdPartyHistories));
        }
        catch (Exception ex) {
            this.setThirdPartyTracking(String.format("preMsg=%s;result=%s;异常=%s;", preMsg, result, ex.getMessage()));
        }
         */
        this.setThirdPartyTracking(String.format("%s;%s-%s", this.getThirdPartyTracking(), thirdPartyReferenceId, status));
        this.setThirdPartyReferenceId(thirdPartyReferenceId);

        if (dispatchFile) {
            this.changeFileSyncStatus(status, result);
        } else {
            this.changeDataSyncStatus(status, result);
        }
    }

    public void assignCallback(String execStatus,String callBack) {
        if ("1".equals(execStatus)) {
            this.setStatus(SyncProcessStatus.DoneFromCallback);
        } else {
            this.setStatus(SyncProcessStatus.ErrorFromCallback);
        }

        this.setThirdPartyCallback(callBack);
    }

    public void retry(String message) {
        if (errorSyncProcessedStatuses.contains(this.getStatus())) {
            this.changeDataSyncStatus(this.getStatus() == SyncProcessStatus.Error ?
                    SyncProcessStatus.Initialize : SyncProcessStatus.SyncViaFTP, message);
        }

        if (errorSyncProcessedStatuses.contains(this.getFileSyncStatus())) {
            this.changeFileSyncStatus(this.getFileSyncStatus() == SyncProcessStatus.Error ?
                    SyncProcessStatus.Initialize : SyncProcessStatus.SyncViaFTP, message);
        }

        this.setRetryCount(this.getRetryCount() + 1);

        this.setLastProcessedTime(new Date());
    }

    public void changeFileSyncStatusAndRetryCount(SyncProcessStatus status, String result) {
        this.setFileSyncStatus(status);
        this.setFileSyncResult(result);
        this.setLastFileProcessedTime(new Date());
        this.setRetryCount(this.getRetryCount() + 1);
    }



    @Data
    private static class ThirdPartyHistory {
        private String id;
        private SyncProcessStatus status;
        private Date executedTime;
        private String result;

        public static ThirdPartyHistory create(String id, SyncProcessStatus status, String result) {
            ThirdPartyHistory thirdPartyHistory = new ThirdPartyHistory();
            thirdPartyHistory.setId(id);
            thirdPartyHistory.setResult(result);
            thirdPartyHistory.setStatus(status);
            thirdPartyHistory.setExecutedTime(new Date());

            return thirdPartyHistory;
        }
    }
}
