package com.bcxin.runtime.domain.repositories.syncs.components.impls;

import com.bcxin.runtime.domain.repositories.syncs.components.SqlParser;
import com.bcxin.runtime.domain.repositories.syncs.dtos.DataSetRowGroupDto;
import com.bcxin.runtime.domain.snapshoots.TableMapSnapshot;
import com.bcxin.runtime.domain.syncs.dtos.DataSetDto;
import com.bcxin.saas.core.AppConfigProperty;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import org.springframework.util.StringUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

public abstract class SqlParserAbstract implements SqlParser {
    private final JsonProvider jsonProvider;
    private final AppConfigProperty appConfigProperty;

    public SqlParserAbstract(JsonProvider jsonProvider,
                             AppConfigProperty appConfigProperty) {
        this.jsonProvider = jsonProvider;
        this.appConfigProperty = appConfigProperty;
    }

    @Override
    public Collection<String> generate(TableMapSnapshot tableMap, DataSetRowGroupDto rowGroup) {
        DataSetDto.Row row = rowGroup.getRow();
        Collection<String> businessSqls = new ArrayList<>();

        if (rowGroup.isNew()) {
            if (isV5SpecialTable(tableMap)) {
                TableMapSnapshot tableMapSnapshot = getTDocumentMapSnapshot(tableMap, row);

                businessSqls.add(getMappedSql(tableMapSnapshot, rowGroup));
            }

        }

        businessSqls.add(getMappedSql(tableMap, rowGroup));
        return businessSqls;
    }

    @Override
    public Collection<String> generateDeletedSql(TableMapSnapshot tableMap, Collection<String> ids) {
        if (ids == null || ids.size() == 0) {
            return null;
        }

        String deletedIds = ids.stream().map(ii -> String.format("'%s'", ii))
                .collect(Collectors.joining(","));

        Collection<String> deletedSqls = new ArrayList<>();

        if (isV5SpecialTable(tableMap)) {
            deletedSqls.add(String.format("DELETE FROM %s where id in (%s)",
                    getTDocumentMapSnapshotTemplate().getTableName(), deletedIds));
        }

        deletedSqls.add(String.format("DELETE FROM %s where id in (%s)", tableMap.getTableName(), deletedIds));


        return deletedSqls;
    }

    /**
     * 获取合并后的T_Document的列的默认值和强制值
     *
     * @param tableMap
     * @param row
     * @return
     */
    protected TableMapSnapshot getTDocumentMapSnapshot(TableMapSnapshot tableMap, DataSetDto.Row row) {
        String json = this.jsonProvider.getJson(getTDocumentMapSnapshotTemplate());
        TableMapSnapshot cloneTable = this.jsonProvider.getData(json, TableMapSnapshot.class);
        Collection<TableMapSnapshot.Column> businessDataColumns = this.getMappedColumn(tableMap, row);

        cloneTable.getColumns().forEach(cl -> {
            if (cl.getDefaultValue() != null && String.valueOf(cl.getDefaultValue()).contains("#{")) {
                String expectedColumn = String.valueOf(cl.getDefaultValue()).replace("#{", "")
                        .replace("}#", "");
                Optional<TableMapSnapshot.Column> selectedColumnOptional =
                        businessDataColumns.stream().filter(ii -> ii.getFieldName().equalsIgnoreCase(expectedColumn))
                                .findFirst();
                if (selectedColumnOptional.isPresent()) {
                    cl.setFixedValue(selectedColumnOptional.get().getFixedValue());
                    cl.setDefaultValue(selectedColumnOptional.get().getDefaultValue());
                    cl.setOnlyForNew(selectedColumnOptional.get().isOnlyForNew());
                }
            }
        });

        cloneTable.getColumns().forEach(mp -> {
            Optional<TableMapSnapshot.Column> mpColumnOptional =
                    tableMap.getColumns().stream().filter(ix -> ix.getFieldName().equalsIgnoreCase(mp.getFieldName())).findFirst();
            if (mpColumnOptional.isPresent()) {
                mp.setFixedValue(mpColumnOptional.get().getFixedValue());
                mp.setDefaultValue(mpColumnOptional.get().getDefaultValue());
                mp.setOnlyForNew(mpColumnOptional.get().isOnlyForNew());
            }
        });

        return cloneTable;
    }

    private static TableMapSnapshot _tableMapSnapshotTemplate;
    private static SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddhhmmss");

    public static TableMapSnapshot getTDocumentMapSnapshotTemplate() {
        if (_tableMapSnapshotTemplate == null) {
            _tableMapSnapshotTemplate = new TableMapSnapshot();
            _tableMapSnapshotTemplate.setTableName("t_document");
            _tableMapSnapshotTemplate.addColumn("ID", null);
            _tableMapSnapshotTemplate.addColumn("LASTMODIFIED", null);
            _tableMapSnapshotTemplate.addColumn("FORMNAME", null);
            _tableMapSnapshotTemplate.addColumn("AUTHOR", "#{AUTHOR}");
            _tableMapSnapshotTemplate.addColumn("CREATED", null);
            _tableMapSnapshotTemplate.addColumn("FORMID", null);
            _tableMapSnapshotTemplate.addColumn("ISTMP", 0);
            _tableMapSnapshotTemplate.addColumnWithFixedValue("SORTID", "0000000000000000000000000000000816");
            _tableMapSnapshotTemplate.addColumn("VERSIONS", 2);
            _tableMapSnapshotTemplate.addColumn("APPLICATIONID", null);
            _tableMapSnapshotTemplate.addColumn("STATEINT", 0);
            _tableMapSnapshotTemplate.addColumn("LASTMODIFIER", "#{LASTMODIFIER}");
            _tableMapSnapshotTemplate.addColumn("DOMAINID", "#{DOMAINID}");
            _tableMapSnapshotTemplate.addColumn("AUDITORLIST", "{}");
            _tableMapSnapshotTemplate.addColumn("COAUDITORLIST", "{}");
            _tableMapSnapshotTemplate.addColumn("STATELABELINFO", "[{\"nodes\":[{\"nodeId\":\"\",\"stateLabel\":\"\",\"state\":\"\",\"auditors\":[],\"coAuditors\":[]}]}]");
            _tableMapSnapshotTemplate.addColumn("MAPPINGID", "#{ID}");
        }
        try {
            //每次设置一个新的
            Optional<TableMapSnapshot.Column> sortIdColumnOptional =
                    _tableMapSnapshotTemplate.getColumns().stream().filter(ii -> "SORTID".equalsIgnoreCase(ii.getFieldName())).findFirst();
            if (sortIdColumnOptional.isPresent()) {
                sortIdColumnOptional.get().setFixedValue(String.format("00000000000%s", simpleDateFormat.format(new Date())));
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        return _tableMapSnapshotTemplate;
    }

    public Collection<TableMapSnapshot.Column> getMappedColumn(TableMapSnapshot tableMap, DataSetDto.Row row) {
        Collection<TableMapSnapshot.Column> tableMappedColumns = tableMap.getColumns();
        if (tableMappedColumns == null || tableMappedColumns.size() == 0) {
            tableMappedColumns = row.getFieldValues().stream().map(ii -> {
                TableMapSnapshot.Column tc = new TableMapSnapshot.Column();
                tc.setFieldName(ii.getName());
                tc.setFixedValue(ii.getValue());

                return tc;
            }).collect(Collectors.toList());

            if (tableMap.getColumns() != null) {
                for (TableMapSnapshot.Column originalColumn : tableMap.getColumns()) {
                    Optional<TableMapSnapshot.Column> columnOptional =
                            tableMappedColumns.stream().filter(ii -> ii.getFieldName()
                                    .equalsIgnoreCase(originalColumn.getFieldName()))
                                    .findFirst();
                    if (columnOptional.isPresent()) {
                        columnOptional.get().setDefaultValue(originalColumn.getDefaultValue());
                        columnOptional.get().setFixedValue(originalColumn.getFixedValue());
                    } else {
                        tableMappedColumns.add(originalColumn);
                    }
                }
            }
        }

        return tableMappedColumns;
    }

    @Override
    public String getProcessedId(TableMapSnapshot tableMap, DataSetDto.Row row) {
        Optional<TableMapSnapshot.Column> idColumnOptional =
                tableMap.getColumns().stream().filter(ii ->
                        TABLE_PRIMARY_KEY_COLUMN_ID.equalsIgnoreCase(ii.getFieldName())).findFirst();
        //若id获取为其它别名字段同步
        Object expectedValue = null;
        String idValue = null;
        if (idColumnOptional.get().getFixedValue() != null && !String.valueOf(idColumnOptional.get().getFixedValue()).equalsIgnoreCase(TABLE_PRIMARY_KEY_COLUMN_ID)) {
            expectedValue = getFieldValue(idColumnOptional.get(), row, getMappedColumn(tableMap, row));
            idValue = String.valueOf(expectedValue);
        } else {
            expectedValue = row.getId();
            if (idColumnOptional.isPresent()) {
                expectedValue = formatIdValue(row.getId(), row, tableMap.getColumns());
            }
            idValue = String.valueOf(expectedValue);
        }

        //针对保安员同步至互联网智能人事ID值特殊处理（临时处理方案）
        if (TENANT_USERS.equalsIgnoreCase(tableMap.getTableName())) {
            idValue = idValue.split("___")[0];
        }
        return idValue;
    }

    private final String SQL_NULL_VALUE = "NULL";
    private final String TABLE_PRIMARY_KEY_COLUMN_ID = "id";
    private final String TABLE_FORM_ID = "FORMID";
    private final String TENANT_USERS = "TENANT_USERS";

    public TableMapSnapshot.Column getPrimaryKeyColumn(Collection<TableMapSnapshot.Column> tableMappedColumns, TableMapSnapshot tableMap, DataSetDto.Row row) {
        for (TableMapSnapshot.Column mappedColumn : tableMappedColumns) {
            if (TABLE_PRIMARY_KEY_COLUMN_ID.equalsIgnoreCase(mappedColumn.getFieldName())) {
                return mappedColumn;
            }
        }
        return null;
    }

    public void setPrimaryKeyFieldValue(Collection<TableMapSnapshot.Column> tableMappedColumns, TableMapSnapshot.Column primaryKeyColumn, TableMapSnapshot tableMap, DataSetDto.Row row) {
        getFieldValue(primaryKeyColumn, row, tableMappedColumns);
    }

    protected Object getFieldValue(TableMapSnapshot.Column ii, DataSetDto.Row row,
                                   Collection<TableMapSnapshot.Column> sourceColumns) {
        Object expectedValue = null;
        Optional<Object> expectedValueOptional =
                row.getFieldValues().stream().filter(rwf -> rwf.getName().equalsIgnoreCase(ii.getFieldName()) &&
                        rwf.getValue() != null)
                        .map(rwf -> {
                            Object fieldValue = rwf.getValue();
                            if (fieldValue != null && fieldValue.getClass().isAssignableFrom(Long.class)) {
                                return rwf.getOriginalValue();
                            }

                            return fieldValue;
                        }).findFirst();
        if (!expectedValueOptional.isPresent()) {
            expectedValue = ii.getFixedValue();
            if (expectedValue == null) {
                expectedValue = ii.getDefaultValue();
            }
        } else {
            expectedValue = expectedValueOptional.get();
            if (ii.getFixedValue() != null) {
                expectedValue = ii.getFixedValue();
            }

            if (expectedValue == null) {
                expectedValue = ii.getDefaultValue();
            }
        }

        /***
         * 当字段需要采用动态映射的方式的时候执行(#{其他字段})
         */
        if (expectedValue != null && String.valueOf(expectedValue).contains("#{") &&
                String.valueOf(expectedValue).contains("}")) {
            String replacedExpectedValue = String.valueOf(expectedValue);
            String[] replacedExpectedValueSplitedValues = replacedExpectedValue.split("}");
            String replacedName = replacedExpectedValueSplitedValues[0];
            String fixedValue = "";
            if (replacedExpectedValueSplitedValues.length > 1) {
                fixedValue = replacedExpectedValueSplitedValues[1];
            }

            String mappingName = replacedName.replace("#{", "")
                    .replace("}", "");
            /*
            Optional<DataSetDto.FieldValue> fieldValueOptional = row.getFieldValues().stream()
                    .filter(mrf -> mappingName.equalsIgnoreCase(mrf.getName()))
                    .findFirst();
            if (!fieldValueOptional.isPresent()) {
                throw new SaasBadException(String.format("找不到字段(%s)映射", mappingName));
            }
            expectedValue = fieldValueOptional.get().getValue();

             */
            expectedValue = getPlaceHolderValue(row, mappingName) + fixedValue;
        }

        if (expectedValue != null && TABLE_PRIMARY_KEY_COLUMN_ID.equalsIgnoreCase(ii.getFieldName())) {
            try {
                String formIdColumnName = TABLE_FORM_ID;
                String originalFormId = String.valueOf(getPlaceHolderValue(row, formIdColumnName));
                if (StringUtils.hasLength(originalFormId)) {
                    Optional<TableMapSnapshot.Column> formIdColumnOptional =
                            sourceColumns.stream().filter(so -> so.getFieldName().equalsIgnoreCase(formIdColumnName))
                                    .findFirst();

                    if (formIdColumnOptional.isPresent()) {
                        String newFormId = String.valueOf(this.getFieldValue(formIdColumnOptional.get(), row, sourceColumns));

                        if (SQL_NULL_VALUE.equalsIgnoreCase(newFormId) || !StringUtils.hasLength(newFormId)) {
                            newFormId = "";
                        }

                        expectedValue = String.valueOf(expectedValue).replace(originalFormId, newFormId);
                    }
                }

            } catch (Exception ex) {
                //当系统不是tlk的表单表的时候, 无需获取FORMID信息
                //ex.printStackTrace();
            }
        }

        if (expectedValue == null) {
            expectedValue = SQL_NULL_VALUE;
        }

        return expectedValue;
    }

    protected boolean isV5SpecialTable(TableMapSnapshot tableMap) {
        return this.appConfigProperty.isNeedTDocument() &&
                tableMap.getTableName().toLowerCase(Locale.ROOT).contains("tlk_");
    }

    protected String getMappedSql(TableMapSnapshot tableMap, DataSetRowGroupDto rowGroup) {
        if (tableMap == null || !StringUtils.hasLength(tableMap.getTableName())) {
            throw new SaasBadException("tableMap 无效");
        }
        DataSetDto.Row row = rowGroup.getRow();
        StringBuilder columnBuilder = new StringBuilder();
        StringBuilder valueBuilder = new StringBuilder();
        StringBuilder updateColumnBuilder = new StringBuilder();

        Collection<TableMapSnapshot.Column> tableMappedColumns = getMappedColumn(tableMap, row);
        Set<String> pushedColumns = new HashSet<>();
        Collection<String> duplicateColumnNames = new ArrayList<>();

        String lastUpdateTime = null;
        for (TableMapSnapshot.Column ii : tableMappedColumns) {
            {
                String fieldName = ii.getFieldName();
                /**
                 * 保证不重复加字段
                 */
                if (pushedColumns.add(ii.getFieldName())) {
                    try {
                        Object expectedValue = getFieldValue(ii, row, tableMappedColumns);

                        if (columnBuilder.length() > 0) {
                            columnBuilder.append(",");
                            valueBuilder.append(",");
                        }

                        String actualValue = null;
                        if (expectedValue.getClass().isAssignableFrom(String.class)) {
                            if (!String.valueOf(expectedValue).equalsIgnoreCase(SQL_NULL_VALUE)) {
                                actualValue = String.format("'%s'", String.valueOf(expectedValue).replaceAll("'", "''"));
                                //valueBuilder.append(String.format("'%s'", String.valueOf(expectedValue).replaceAll("'", "''")));
                            } else {
                                actualValue = "null";
                                //valueBuilder.append("null");
                            }
                        } else if (expectedValue.getClass().isAssignableFrom(Date.class)) {
                            actualValue = String.format("'%s'", expectedValue);
                            //valueBuilder.append(String.format("'%s'", expectedValue));
                        } else {
                            actualValue = String.valueOf(expectedValue);
                            // valueBuilder.append(expectedValue);
                        }
                        valueBuilder.append(actualValue);

                        columnBuilder.append(ii.getFieldName());

                        if (ii.getFieldName().equalsIgnoreCase("lastmodified") || ii.getFieldName().equalsIgnoreCase("last_sync_time")) {
                            lastUpdateTime = actualValue;
                        }
                        if (!ii.getFieldName().equalsIgnoreCase("id") && !ii.isOnlyForNew()) {
                            duplicateColumnNames.add(ii.getFieldName());
                            if (updateColumnBuilder.length() > 0) {
                                updateColumnBuilder.append(",");
                            }
                            updateColumnBuilder.append(String.format("%s=%s", ii.getFieldName(), actualValue));
                        }
                    } catch (Exception ex) {
                        throw new SaasBadException(String.format("解析字段(%s)发生异常", fieldName), ex);
                    }
                }
            }
        }

        if (rowGroup.isNew()) {
            return String.format("insert into %s (%s) values(%s) ", tableMap.getTableName(), columnBuilder, valueBuilder);
        } else {
            try {
                if (lastUpdateTime == null) {
                    Optional<DataSetDto.FieldValue> fieldValue = row.getFieldValues().stream().filter(ii ->
                            ii.getName().equalsIgnoreCase("LASTMODIFIED")).findFirst();
                    lastUpdateTime = (String) fieldValue.get().getValue();
                }
                if (lastUpdateTime != null) {
                    lastUpdateTime = lastUpdateTime.replaceAll("'", "");
                }
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
                Date date = simpleDateFormat.parse(lastUpdateTime);
                SimpleDateFormat simpleDateFormat1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                lastUpdateTime = simpleDateFormat1.format(date);
            } catch (Exception e) {
                e.printStackTrace();
            }
            //针对保安员同步至互联网智能人事ID值特殊处理（临时处理方案）
            if (TENANT_USERS.equalsIgnoreCase(tableMap.getTableName())) {
                return String.format("update %s set %s where id='%s' ", tableMap.getTableName(),
                        updateColumnBuilder, String.valueOf(this.formatIdValue(row.getId(), row, tableMappedColumns)).split("___")[0]);
            } else {
                return String.format("update %s set %s where id='%s' and lastmodified < '%s' ", tableMap.getTableName(),
                        updateColumnBuilder, this.formatIdValue(row.getId(), row, tableMappedColumns), lastUpdateTime);
            }
        }
    }

    protected abstract String buildFinalSql(String sql, Collection<String> duplicateColumnNames);

    /**
     * 针对表字段对应名字不一致时，值获取方式
     *
     * @param row
     * @param mappingName
     * @return
     */
    protected Object getPlaceHolderValue(DataSetDto.Row row, String mappingName) {
        Optional<DataSetDto.FieldValue> fieldValueOptional = row.getFieldValues().stream()
                .filter(mrf -> mappingName.equalsIgnoreCase(mrf.getName()))
                .findFirst();
        if (!fieldValueOptional.isPresent()) {
            throw new SaasBadException(String.format("找不到字段(%s)映射", mappingName));
        }
        Object fieldValue = fieldValueOptional.get().getValue();
        if (fieldValue != null && fieldValue.getClass().isAssignableFrom(Long.class)) {
            return fieldValueOptional.get().getOriginalValue();
        }
        return fieldValueOptional.get().getValue();
    }

    private Object formatIdValue(
            Object expectedValue, DataSetDto.Row row,
            Collection<TableMapSnapshot.Column> sourceColumns) {
        try {
            String formIdColumnName = TABLE_FORM_ID;
            String originalFormId = String.valueOf(getPlaceHolderValue(row, formIdColumnName));
            if (StringUtils.hasLength(originalFormId)) {
                Optional<TableMapSnapshot.Column> formIdColumnOptional =
                        sourceColumns.stream().filter(so -> so.getFieldName().equalsIgnoreCase(formIdColumnName))
                                .findFirst();

                if (formIdColumnOptional.isPresent()) {
                    String newFormId = String.valueOf(this.getFieldValue(formIdColumnOptional.get(), row, sourceColumns));

                    if (SQL_NULL_VALUE.equalsIgnoreCase(newFormId) || !StringUtils.hasLength(newFormId)) {
                        newFormId = "";
                    }

                    expectedValue = String.valueOf(expectedValue).replace(originalFormId, newFormId);
                }
            }
        } catch (Exception ex) {
            //当源表单不是tlk_生成的表单的时候, 无需进行formId数据的获取
            ex.printStackTrace();
        }

        return expectedValue;
    }
}
