package com.bcxin.identity.domains.components;

import com.bcxin.identity.domains.enums.AlgorithmType;
import com.bcxin.identity.domains.utils.PasswordUtils;
import com.bcxin.identity.domains.utils.SecurityUtil;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;

@Slf4j
@Component
public class DefaultPasswordEncoderImpl implements PasswordEncoder {
    private final Logger logger = LoggerFactory.getLogger(DefaultPasswordEncoderImpl.class);

    @Override
    public boolean isMatched(String expectedPassword, String actualPassword) {
        if (!StringUtils.hasLength(actualPassword) || !StringUtils.hasLength(expectedPassword)) {
            return false;
        }

        boolean isMatched = true;
        StopWatch stopWatch = new StopWatch();

        try {
            stopWatch.start("PasswordUtils.initVerify");
            if(PasswordUtils.initVerify(actualPassword, expectedPassword)) {
                return true;
            }
            stopWatch.stop();

            stopWatch.start("SecurityUtil.decryptPassword");
            if (actualPassword.equals(SecurityUtil.decryptPassword(expectedPassword))) {
                return true;
            }
            stopWatch.stop();

            stopWatch.start("SecurityUtil.decryptPassword");
            if (PasswordUtils.validatePassword(actualPassword, expectedPassword)) {
                return true;
            }
            stopWatch.stop();

            stopWatch.start("SecurityUtil.encryptOld");
            isMatched = expectedPassword.equals(SecurityUtil.encryptOld(actualPassword));

            return isMatched;
        } catch (Exception ex) {
            isMatched = false;
            log.error("密码比对异常", ex);
        } finally {
            stopWatch.stop();
            if (!isMatched) {
                logger.error("{}-expectedPassword={};actualPassword={};{};", isMatched, expectedPassword, actualPassword, stopWatch);
            }
        }

        return false;
    }

    @Override
    public String encode(AlgorithmType algorithmType, String password) {
        if (algorithmType == AlgorithmType.SM2) {
            return PasswordUtils.initSign(password);
        }

        return PasswordUtils.entryptPassword(password).get("hashPassword");
    }

    public static void main(String[] args) {
        System.out.println(PasswordUtils.initSign("123456"));
    }
}
