package com.bcxin.tenant.bcx.domains.repositories;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bcxin.tenant.bcx.domains.entities.MetaModuleEntity;
import com.bcxin.tenant.bcx.domains.mappers.MetaModuleMapper;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaModuleSearchCriteria;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Repository;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.List;

/**
 * 模块元数据仓库实现
 */
@Repository
public class MetaModuleRepositoryImpl extends RepositoryAbstract<MetaModuleEntity> implements MetaModuleRepository {
    private final MetaModuleMapper mapper;

    protected MetaModuleRepositoryImpl(MetaModuleMapper mapper) {
        super(mapper);
        this.mapper = mapper;
    }

    @Override
    public EntityCollection<MetaModuleEntity> search(MetaModuleSearchCriteria criteria) {
        // 创建 Page 对象
        Page<MetaModuleEntity> page = new Page<>(criteria.getPageIndex(), criteria.getPageSize());

        // 使用 LambdaQueryWrapper 构建查询条件
        LambdaQueryWrapper<MetaModuleEntity> queryWrapper = new LambdaQueryWrapper<>();
        
        // deleted = 0
        queryWrapper.eq(MetaModuleEntity::isDeleted, false);
        
        // id = #{criteria.id}
        if (!StringUtil.isEmpty(criteria.getId())) {
            queryWrapper.eq(MetaModuleEntity::getId, criteria.getId());
        }
        
        // id IN (...)
        List<String> ids = criteria.getIds();
        if (ids != null && !ids.isEmpty()) {
            queryWrapper.in(MetaModuleEntity::getId, ids);
        }
        
        // project_id = #{criteria.projectId}
        if (!StringUtil.isEmpty(criteria.getProjectId())) {
            queryWrapper.eq(MetaModuleEntity::getProjectId, criteria.getProjectId());
        }
        
        // name LIKE CONCAT('%',#{criteria.name},'%')
        if (!StringUtil.isEmpty(criteria.getName())) {
            queryWrapper.like(MetaModuleEntity::getName, criteria.getName());
        }
        
        // ORDER BY created_time DESC
        queryWrapper.orderByDesc(MetaModuleEntity::getCreatedTime);

        // 使用 mapper 查询数据
        IPage<MetaModuleEntity> result = mapper.selectPage(page, queryWrapper);

        // 转换为 EntityCollection
        return EntityCollection.create(
                result.getRecords(),
                criteria.getPageSize(),
                result.getTotal()
        );
    }

    @Override
    public List<MetaModuleEntity> getByIds(List<String> ids) {
        if (ids == null || ids.isEmpty()) {
            return Collections.emptyList();
        }
        return this.mapper.selectBatchIds(ids);
    }

    @Override
    public void batchMarkAsDeleted(List<String> ids, Timestamp updateTime) {
        if (ids == null || ids.isEmpty()) {
            return;
        }
        
        LambdaUpdateWrapper<MetaModuleEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(MetaModuleEntity::getId, ids)
                     .eq(MetaModuleEntity::isDeleted, false)
                     .set(MetaModuleEntity::isDeleted, true)
                     .set(MetaModuleEntity::getLastUpdatedTime, updateTime);
        
        this.mapper.update(null, updateWrapper);
    }
}
