package com.bcxin.tenant.bcx.domains.aspects;

import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.enums.MetaCreatorReferenceType;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;

/**
 * AOP切面：自动从TenantContext获取用户信息并赋值给command对象
 * 适用于MetaProjectServiceImpl的所有dispatch方法
 */
@Aspect
@Component
public class TenantUserContextAspect {
    
    private static final Logger logger = LoggerFactory.getLogger(TenantUserContextAspect.class);

    /**
     * 在dispatch方法执行前，自动填充用户上下文信息到command对象
     * 拦截所有ServiceImpl的dispatch方法
     */
    @Before("execution(* com.bcxin.tenant.bcx.domains.services.impls.*ServiceImpl.dispatch(..))")
    public void fillTenantUserContext(JoinPoint joinPoint) {
        try {
            // 获取方法参数（command对象）
            Object[] args = joinPoint.getArgs();
            if (args == null || args.length == 0) {
                return;
            }
            
            Object command = args[0];
            if (command == null) {
                return;
            }

            // 从TenantContext获取用户信息
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance()
                    .getUserContext()
                    .get();
            
            if (userModel == null) {
                logger.warn("TenantContext中未获取到用户信息，跳过自动填充");
                return;
            }

            // 获取用户信息
            MetaCreatorReferenceType referenceType = userModel.getUserType();
            String referenceNumber = userModel.getOrganizationId();
            String id = userModel.getId();

            // 通过反射设置字段值
            setFieldValue(command, "referenceType", referenceType);
            setFieldValue(command, "referenceNumber", referenceNumber);
            setFieldValue(command, "creatorId", id);
            setFieldValue(command, "lastUpdaterId", id);
            
            logger.debug("成功填充用户上下文信息到command对象: referenceType={}, referenceNumber={}, id={}",
                    referenceType, referenceNumber, id);
            
        } catch (Exception ex) {
            // 记录错误但不中断业务流程
            logger.error("填充用户上下文信息时发生异常", ex);
        }
    }

    /**
     * 通过反射设置对象的私有final字段值
     * 
     * @param target 目标对象
     * @param fieldName 字段名称
     * @param value 要设置的值
     */
    private void setFieldValue(Object target, String fieldName, Object value) {
        try {
            Field field = findField(target.getClass(), fieldName);
            if (field == null) {
                logger.debug("字段 {} 在 {} 中不存在，跳过", fieldName, target.getClass().getSimpleName());
                return;
            }
            
            field.setAccessible(true);
            field.set(target, value);
            
        } catch (IllegalAccessException e) {
            logger.warn("无法访问字段 {} 在 {}: {}", fieldName, target.getClass().getSimpleName(), e.getMessage());
        } catch (Exception e) {
            logger.error("设置字段 {} 时发生异常", fieldName, e);
        }
    }

    /**
     * 在类及其父类中查找字段
     * 
     * @param clazz 目标类
     * @param fieldName 字段名称
     * @return 找到的字段，如果不存在则返回null
     */
    private Field findField(Class<?> clazz, String fieldName) {
        Class<?> currentClass = clazz;
        while (currentClass != null) {
            try {
                return currentClass.getDeclaredField(fieldName);
            } catch (NoSuchFieldException e) {
                // 在父类中继续查找
                currentClass = currentClass.getSuperclass();
            }
        }
        return null;
    }
}

