package com.bcxin.tenant.open.rest.apis.utils;

import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.UserDetailResponse;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.*;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.ProjectAdvanceSearchRequest;
import com.bcxin.tenant.open.jdks.requests.ProjectSearchRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import com.bcxin.tenant.open.jdks.responses.EmployeeDetailResponse;
import com.bcxin.tenant.open.jdks.responses.ProjectOverviewResponse;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import com.bcxin.tenant.open.rest.apis.caches.TenantSecurityEmployeeCache;
import com.bcxin.tenant.open.rest.apis.caches.UserDetailCache;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.responses.EmployeeDefaultStationResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class UserUtil {
    private static final Logger logger = LoggerFactory.getLogger(UserUtil.class);

    public static UserDetailResponse get(HotCacheProvider hotCacheProvider,
                                         IdentityRpcProvider identityRpcProvider,
                                         String employeeId, DispatchAccountType accountType
    ) {
        UserDetailResponse response = null;
        UserDetailCache userDetailCache =
                hotCacheProvider.get(CacheKeyManage.getEmpDetailKey(employeeId), UserDetailCache.class);
        if (userDetailCache == null) {
            response = identityRpcProvider.getByEmployeeId(employeeId, accountType);
            if (response == null) {
                return null;
            }

            userDetailCache = UserDetailCache.create(
                    response.getId(),
                    response.getEmployeeId(),
                    response.getName(),
                    response.getOrganizationId(),
                    response.getOccupationType(),
                    response.getImIdentity(),
                    response.getInstitutional(),
                    response.getSuperviseRegionCode(),
                    response.getOrgName(),
                    response.isDomainAdmin(),
                    response.getResponsibleOfStationIds(),
                    response.getSelectedStationIds(),
                    response.getCLevel(),
                    response.getRts(),
                    response.getOrgLatitude(),
                    response.getOrgLongitude(),
                    response.getSupervise(),
                    response.getPaymentOrgId()
            );

            hotCacheProvider.put(CacheKeyManage.getEmpDetailKey(employeeId), 24 * 60 * 60, userDetailCache, false);
        } else {
            response = UserDetailResponse.create(
                    userDetailCache.getId(),
                    userDetailCache.getEmployeeId(),
                    userDetailCache.getName(),
                    userDetailCache.getOrganizationId(),
                    userDetailCache.getOccupationType(),
                    userDetailCache.getImIdentity(),
                    accountType,
                    userDetailCache.getInstitutional(),
                    userDetailCache.getSuperviseRegionCode(),
                    userDetailCache.getOrgName(),
                    userDetailCache.isDomainAdmin(),
                    userDetailCache.getResponsibleOfStationIds(),
                    userDetailCache.getSelectedStationIds(),
                    userDetailCache.getCLevel(),
                    userDetailCache.getRts(),
                    userDetailCache.getPaymentOrgId()
            );

            response.assignIsSupervise(userDetailCache.getSupervise() != null ? userDetailCache.getSupervise().booleanValue() : false);
            response.assignOrgLatLon(userDetailCache.getOrgLatitude(), userDetailCache.getOrgLongitude());
        }

        return response;
    }

    public static TenantEmployeeContext.PermissionModel getPermission(HotCacheProvider hotCacheProvider,
                                                                      IdentityRpcProvider identityRpcProvider,
                                                                      EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                                                      DispatchDataScopeRpcProvider dataScopeRpcProvider,
                                                                      Collection<String> subScopeCompanyIds,
                                                                      UserDetailResponse userDetailResponse,
                                                                      ProjectReaderRpcProvider projectReaderRpcProvider,
                                                                      boolean advancePermission) {
        if (BusinessUtil.checkIfDesk(userDetailResponse.getAccountType()) || advancePermission) {
            /**
             * 调度台账号的话，实时刷新数据;
             * 他量也少; 缓存1小时
             */
            TenantEmployeeContext.PermissionModel permissionModel =
                    hotCacheProvider.get(CacheKeyManage.getEmpDeskPermissionKey(userDetailResponse.getEmployeeId(), userDetailResponse.getAccountType()), () -> {
                        UserDetailResponse detailResponse =
                                identityRpcProvider.getByEmployeeId(userDetailResponse.getEmployeeId(), DispatchAccountType.Desk);

                        StringBuilder sb = new StringBuilder();
                        // EmployeeDetailResponse refreshEmpInfo = employeeReaderRpcProvider.get(userDetailResponse.getEmployeeId(), DeskType.Normal);
                        if (detailResponse == null) {
                            //return null;
                            throw new NoFoundTenantException("找不到当前用户");
                        }

                        List<String> selectedStationIds = new ArrayList<>();
                        if (!CollectionUtils.isEmpty(userDetailResponse.getSelectedStationIds())) {
                            selectedStationIds.addAll(userDetailResponse.getSelectedStationIds());
                            sb.append(String.format("u.selectedStationIds=%s;",
                                    CollectionUtils.isEmpty(userDetailResponse.getSelectedStationIds()) ? "EMPTY" :
                                            userDetailResponse.getSelectedStationIds().stream().collect(Collectors.joining(","))
                            ));
                        }

                        sb.append(String.format("调度台类型=%s;当前的用户为:id=%s, name=%s, institutional=%s, isEventOrganizer=%s;",
                                userDetailResponse.getAccountType(),
                                userDetailResponse.getId(),
                                userDetailResponse.getName(),
                                userDetailResponse.getInstitutional(),
                                BusinessUtil.isEventOrganizer(userDetailResponse.getInstitutional())
                        ));
                        /**
                         * 针对承办方来说; 管理人员的数据权限, 通过如下来获取
                         */
                        if (BusinessUtil.isEventOrganizer(userDetailResponse.getInstitutional())) {
                            Collection<String> permissionIds =
                                    dataScopeRpcProvider.getDataScopeById(
                                            userDetailResponse.getEmployeeId(),
                                            DispatchDataScopeType.EventOrganizerManager
                                    );
                            sb.append(String.format("获取到的权限信息为=%s",
                                    CollectionUtils.isEmpty(permissionIds) ? "EMPTY" : permissionIds.stream().collect(Collectors.joining(","))
                            ));

                            if (!CollectionUtils.isEmpty(permissionIds)) {
                                Collection<String> matchedStationIds =
                                        permissionIds.stream().filter(ii -> ii.endsWith("--__W052b1XNoLmPckF6ch1"))
                                                .collect(Collectors.toSet());

                                selectedStationIds.addAll(matchedStationIds);
                                Collection<String> matchedNoStationIds =
                                        permissionIds.stream().filter(ii -> !ii.endsWith("--__W052b1XNoLmPckF6ch1"))
                                                .collect(Collectors.toSet());
                                subScopeCompanyIds.addAll(matchedNoStationIds);
                            }


                            /**
                             * 针对组织管理员; 有用本企业所有项目的权限;
                             * 添加这个的目的是为了解决:
                             * 本公司发起的项目, 但是驻勤点由其他企业创建, 人员是其他企业的人员
                             */
                            if (userDetailResponse.isDomainAdmin()) {
                                Collection<String> companyManagedProjectIds =
                                        projectReaderRpcProvider.getIdsByOrganIds(
                                                Stream.of(userDetailResponse.getOrganizationId()).collect(Collectors.toSet()));

                                sb.append(String.format("根据getIdsByOrganIds获取到的companyManagedProjectIds=%s;",
                                        (CollectionUtils.isEmpty(companyManagedProjectIds) ? "EMPTY" : companyManagedProjectIds.stream().collect(Collectors.joining(",")))));
                                if (!CollectionUtils.isEmpty(companyManagedProjectIds)) {
                                    subScopeCompanyIds.addAll(companyManagedProjectIds);
                                }
                            }


                            /**
                             * 对管理单位限制的岗点信息
                             */
                            permissionIds =
                                    dataScopeRpcProvider.getDataScopeById(
                                            userDetailResponse.getOrganizationId(),
                                            DispatchDataScopeType.EventOrganizerLimitedResource
                                    );
                            if (!CollectionUtils.isEmpty(permissionIds)) {
                                selectedStationIds.addAll(permissionIds);
                            }

                            sb.append(String.format("获取到的stationId=%s;companyIds=%s;EventOrganizerLimitedResource.permissionIds=%s;",
                                    CollectionUtils.isEmpty(selectedStationIds) ? "EMPTY" : selectedStationIds.stream().collect(Collectors.joining(",")),
                                    CollectionUtils.isEmpty(subScopeCompanyIds) ? "EMPTY" : subScopeCompanyIds.stream().collect(Collectors.joining(",")),
                                    CollectionUtils.isEmpty(permissionIds) ? "EMPTY" : permissionIds.stream().collect(Collectors.joining(","))
                            ));
                        }

                        Set<String> limitedStationIds = new HashSet<>();
                        if (!CollectionUtils.isEmpty(detailResponse.getResponsibleOfStationIds())) {
                            limitedStationIds.addAll(detailResponse.getResponsibleOfStationIds());
                            sb.append(String.format("获取到的getResponsibleOfStationIds=%s;",
                                    CollectionUtils.isEmpty(detailResponse.getResponsibleOfStationIds()) ? "EMPTY" :
                                            detailResponse.getResponsibleOfStationIds().stream().collect(Collectors.joining(","))
                            ));
                        }

                        /**
                         * 针对大活的管理单位,我们额外可以限制他们可以访问的驻勤点信息
                         * 比如: 文旅局, 体育局，默认是查看所有, 但是也可以限制指定的岗点
                         */
                        if(!CollectionUtils.isEmpty(userDetailResponse.getSelectedStationIds()) &&
                                BusinessUtil.isDaHuoManagedOrg(userDetailResponse.getInstitutional())) {
                            limitedStationIds.addAll(selectedStationIds);
                            sb.append(String.format("大活行业设置selectedStationIds=%s;", selectedStationIds.stream().collect(Collectors.joining(","))));
                        }

                        return TenantEmployeeContext.PermissionModel.create(
                                detailResponse.isDomainAdmin(),
                                limitedStationIds,
                                selectedStationIds,
                                subScopeCompanyIds,
                                sb.toString()
                        );
                    }, 5 * 60, TenantEmployeeContext.PermissionModel.class, false);
            return permissionModel;
        } else {
            /**
             * todo: 周一后可以尝试优化解决掉: 放在token,但是数据刷新的情况
             */
            Set<String> limitedStationIds = new HashSet<>();
            if (!CollectionUtils.isEmpty(limitedStationIds)) {
                limitedStationIds.addAll(userDetailResponse.getResponsibleOfStationIds());
            }

            /**
             * 针对大活的管理单位,我们额外可以限制他们可以访问的驻勤点信息
             * 比如: 文旅局, 体育局，默认是查看所有, 但是也可以限制指定的岗点
             */
            if(!CollectionUtils.isEmpty(userDetailResponse.getSelectedStationIds()) &&
                    BusinessUtil.isDaHuoManagedOrg(userDetailResponse.getInstitutional())) {
                limitedStationIds.addAll(userDetailResponse.getSelectedStationIds());
            }

            return TenantEmployeeContext.PermissionModel.create(
                    userDetailResponse.isDomainAdmin(),
                    limitedStationIds,
                    userDetailResponse.getSelectedStationIds(),
                    subScopeCompanyIds,
                    "APP-非调度台"
            );
        }
    }

    public static TenantSecurityEmployeeCache getSecurityEmployeeCache(HotCacheProvider hotCacheProvider,
                                                                       EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                                                       RdSyncRpcWriterProvider syncRpcWriterProvider,
                                                                       String employeeId) {
        TenantSecurityEmployeeCache employeeCache =
                hotCacheProvider.get(
                        CacheKeyManage.getUserInfoFromAuthFilterKey(employeeId, DeskType.Normal),
                        TenantSecurityEmployeeCache.class);
        if (employeeCache != null) {
            return employeeCache;
        }

        EmployeeDetailResponse employeeDetailResponse =
                employeeReaderRpcProvider.get(employeeId, DeskType.Normal);
        if (employeeDetailResponse == null) {
            syncRpcWriterProvider.sync(
                    SyncParameterWrapperRequest.create(DispatchDataType.Employee,
                            Collections.singleton(employeeId))
            );

            employeeDetailResponse =
                    employeeReaderRpcProvider.get(employeeId, DeskType.Normal);

            logger.error("触发重新从原始数据刷新用户信息:{}; 刷新后的结果为={}", employeeId, (employeeDetailResponse == null ? "仍为空" : "不为空"));
        }

        if (employeeDetailResponse != null) {
            employeeCache = TenantSecurityEmployeeCache.create(
                    employeeDetailResponse.getSecurityStationId(),
                    employeeDetailResponse.getSecurityStationName(),
                    employeeDetailResponse.getSuperviseDepartId(),
                    employeeDetailResponse.getTencentUserId(),
                    employeeDetailResponse.getName(),
                    employeeDetailResponse.getIdCardNo(),
                    employeeDetailResponse.getSuperviseDepartName(),
                    employeeDetailResponse.getResourceTypes()
            );

            hotCacheProvider.put(
                    CacheKeyManage.getUserInfoFromAuthFilterKey(employeeId, DeskType.Normal),
                    24 * 60 * 60, employeeCache, false);

            return employeeCache;
        } else {
            return null;
        }
    }

    public static EmployeeDefaultStationResponse getSecurityStationId(HotCacheProvider hotCacheProvider,
                                                                      EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                                                      RdSyncRpcWriterProvider syncRpcWriterProvider,
                                                                      String employeeId
    ) {
        String stationId = null;
        int countOfStations = 1;
        EmployeeDetailResponse detailResponse = employeeReaderRpcProvider.get(employeeId, DeskType.Normal);
        if (detailResponse == null) {
            if (StringUtil.isEmpty(employeeId)) {
                throw new BadTenantException(String.format("当前团队成员/组织职员(%s)信息无效或者不在任何组织、请选择正确团队/组织后重新进入", employeeId));
            }

            syncRpcWriterProvider.sync(
                    SyncParameterWrapperRequest.create(DispatchDataType.Employee,
                            Collections.singleton(employeeId))
            );

            detailResponse =
                    employeeReaderRpcProvider.get(employeeId, DeskType.Normal);

            logger.error("触发重新从原始数据刷新用户信息:{}; 刷新后的结果为={}", employeeId, (detailResponse == null ? "仍为空" : "不为空"));

            if (detailResponse != null) {
                stationId = detailResponse.getSecurityStationId();
                countOfStations = detailResponse.getCountOfStations();
            }
        } else {
            stationId = detailResponse.getSecurityStationId();
            countOfStations = detailResponse.getCountOfStations();
        }

        if (stationId != null) {

        } else {
            if (StringUtil.isEmpty(employeeId)) {
                throw new BadTenantException(String.format("当前团队成员/组织职员(%s)信息无效或者不在任何组织、请选择正确团队/组织后重新进入", employeeId));
            }

            throw new BadTenantException(String.format("当前的人员(%s)不在任何驻勤点", employeeId));
        }

        return EmployeeDefaultStationResponse.create(
                BusinessConstants.isEmptyValue(stationId),
                stationId, countOfStations);
    }
}
