package com.bcxin.tenant.open.rest.apis.impls;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.open.jdks.SocketPublishRpcProvider;
import com.bcxin.tenant.open.jdks.requests.BroadcastMessageRequest;
import com.bcxin.tenant.open.jdks.requests.RequestAbstract;
import com.bcxin.tenant.open.rest.apis.components.WebSocketServer;
import com.bcxin.tenant.open.rest.apis.dtos.SocketSessionDTO;
import com.bcxin.tenant.open.rest.apis.dtos.SocketSessionMessageResponse;
import jakarta.websocket.Session;
import lombok.Getter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Future;

@Getter
public abstract class AbstractSocketPublishRpcProvider implements SocketPublishRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(SocketPublishRpcProvider.class);

    private final WebSocketServer webSocketServer;
    private final JsonProvider jsonProvider;

    public AbstractSocketPublishRpcProvider(WebSocketServer webSocketServer,
                                        JsonProvider jsonProvider) {
        this.webSocketServer = webSocketServer;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public  void  dispatch(BroadcastMessageRequest request) {
        Collection<SocketSessionDTO> socketSessions = getDestinationSocketSessions(request);
        if (!CollectionUtils.isEmpty(socketSessions)) {
            Collection<SocketSessionDTO> closedSocketSessions = new ArrayList<>();
            for (SocketSessionDTO st : socketSessions) {
                boolean isOpen = true;
                try {
                    Session session = st.getSession();
                    if (session.isOpen()) {
                        SocketSessionMessageResponse sr = getSocketSessionMessageResponse(st,request);
                        String body = this.jsonProvider.getJson(sr);

                        synchronized (session){
                            session.getBasicRemote().sendText(body);
                            logger.warn("websocket的推送数据完毕。session:{}, body:{}", st.getDescription(), body);
                        }
                    } else {
                        logger.error("此socket.session已经关闭连接; {}", st.getDescription());
                        closedSocketSessions.add(st);
                        isOpen = false;
                    }
                } catch (Exception ex) {
                    ex.printStackTrace();

                    logger.error("websocket的推送数据失败:{}", ExceptionUtil.getStackMessage(ex));
                }
                finally {
                    if (isOpen) {
                        logger.error("执行socket操作:内容={}, session={}",
                                this.jsonProvider.getJson(request),
                                st.getDescription()
                        );
                    }
                }
            }

            if (!CollectionUtils.isEmpty(closedSocketSessions)) {
                this.webSocketServer.removeCloseSessions(closedSocketSessions);
            }
        }else {
            logger.error("当前的请求:{}; 未找到对应的socketSession", this.jsonProvider.getJson(request));
        }
    }

    public abstract Collection<SocketSessionDTO> getDestinationSocketSessions(BroadcastMessageRequest request);
    public abstract SocketSessionMessageResponse getSocketSessionMessageResponse(SocketSessionDTO st,BroadcastMessageRequest request);

}
