package com.bcxin.tenant.open.rest.apis.filters;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.*;
import com.bcxin.tenant.open.infrastructures.exceptions.*;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.open.infrastructures.valueTypes.TrafficTagValueType;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.TenantWarnContentRequest;
import com.bcxin.tenant.open.jdks.responses.RdDeviceDeskPermissionResponse;
import com.bcxin.tenant.open.infrastructures.UserDetailResponse;
import com.bcxin.tenant.open.rest.apis.caches.TenantSecurityEmployeeCache;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.responses.ResponseBuilder;
import com.bcxin.tenant.open.rest.apis.utils.DubboCommonUtils;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.open.rest.apis.utils.ServletRequestUtil;
import com.bcxin.tenant.open.rest.apis.utils.UserUtil;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
@Order(Ordered.HIGHEST_PRECEDENCE-1)
public class AuthFilter extends OncePerRequestFilter {
    private static final Logger logger = LoggerFactory.getLogger(AuthFilter.class);
    private final static Collection<String> excludedUrls = new ArrayList<>();
    private final static Collection<String> ignoredDeskCheckUrls = new ArrayList<>();
    private final static Collection<String> ignoredReadStationEmployeeCompanyDetail =
            Stream.of("/security-stations/",
                    "/employees/","/companies/").collect(Collectors.toSet());

    static {
        excludedUrls.add("/identity/");
        //excludedUrls.add("/security-stations/station-types");
        /**
         * API Docs
         */
        excludedUrls.add("/v3/");
        excludedUrls.add("/swagger-ui");
        excludedUrls.add("/tencent/callback");
        excludedUrls.add("/flush");
        excludedUrls.add("/sync-slave");
        excludedUrls.add("/static");
        excludedUrls.add("/html");
        excludedUrls.add("/alive-connect");
        excludedUrls.add("/tenant/users/current/real-location");

        excludedUrls.add("/json");
        excludedUrls.add("/do-confirm");
        excludedUrls.add("/data-query");
        excludedUrls.add("/users");
        excludedUrls.add("/rooms/do-post-to-be-join-rooms");
        excludedUrls.add("/messages/do-get-instants");
        excludedUrls.add("/sync-locations");

        /**
         * 针对开麦申请（handup）
         */
        excludedUrls.add("/rooms/\\w+/actions/\\w+");

        ignoredDeskCheckUrls.add("/dispatch-logs");
        ignoredDeskCheckUrls.add("/attendance");
        ignoredDeskCheckUrls.add("/tenant/users/current");
        ignoredDeskCheckUrls.add("/dispatch-logs/search");
        ignoredDeskCheckUrls.add("/roll-call-employees/search");
        ignoredDeskCheckUrls.add("/policeIncidents");
        ignoredDeskCheckUrls.add("/departs");
        ignoredDeskCheckUrls.add("/flush");
        ignoredDeskCheckUrls.add("/sync-slave");
        /**
         * 可以直接从企业端创建电子围栏
         */
        ignoredDeskCheckUrls.add("/security-station-rails");
        /**
         * 针对确认房间的
         */
        ignoredDeskCheckUrls.add("/do-confirm");
        /**
         * websocket链接
         */
        ignoredDeskCheckUrls.add("/websocket/connect");

        ignoredDeskCheckUrls.add("/roll-call-plans");

        ignoredDeskCheckUrls.add("/roll-call-employees/search");
        ignoredDeskCheckUrls.add("/employees/duty-status");
    }

    /**
     * todo 统一修复序列化方式
     */
    private final JsonProvider jsonProvider;
    private final IdentityRpcProvider identityRpcProvider;
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;
    private final TenantWarnContentRpcProvider warnContentRpcProvider;
    private final DispatchDataScopeRpcProvider dataScopeRpcProvider;
    private final HotCacheProvider hotCacheProvider;
    private final RdSyncRpcWriterProvider syncRpcWriterProvider;
    private final ProjectReaderRpcProvider projectReaderRpcProvider;

    private final String tenantNode;

    public AuthFilter(JsonProvider jsonProvider,
                      IdentityRpcProvider identityRpcProvider,
                      EmployeeReaderRpcProvider employeeReaderRpcProvider,
                      TenantWarnContentRpcProvider warnContentRpcProvider,
                      DispatchDataScopeRpcProvider dataScopeRpcProvider,
                      HotCacheProvider hotCacheProvider,
                      RdSyncRpcWriterProvider syncRpcWriterProvider,
                      ProjectReaderRpcProvider projectReaderRpcProvider, @Value("${tenant.node:null}") String tenantNode) {
        this.jsonProvider = jsonProvider;
        this.identityRpcProvider = identityRpcProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.warnContentRpcProvider = warnContentRpcProvider;
        this.dataScopeRpcProvider = dataScopeRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.syncRpcWriterProvider = syncRpcWriterProvider;
        this.projectReaderRpcProvider = projectReaderRpcProvider;
        this.tenantNode = tenantNode;
    }


    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        boolean isAfterParseRequestToken = false;
        TenantContext tenantContext = TenantContext.getInstance();
        try {
            String requestUri = request.getRequestURI();
            boolean ignoredDeskCheck = ignoredDeskCheckUrls.stream()
                    .anyMatch(ix -> requestUri.startsWith(String.format("%s%s", request.getContextPath(), ix)));

            parseRequestToken(request, ignoredDeskCheck);
            isAfterParseRequestToken = true;


            TenantEmployeeContext userContext = tenantContext.getUserContext();
            if (userContext.get() == null) {
                throw new UnAuthorizedTenantException("当前无有效dispatchToken信息");
            }

            /**
             * 无调度台权限
             */
            TenantEmployeeContext.TenantUserModel userModel = userContext.get();

            /**
             * 对于非调度账号, 只能调用获取当前用户信息
             */
            if (!ignoredDeskCheck && !BusinessUtil.checkIfDesk(userModel.getAccountType())) {
                throw new ForbidTenantException(String.format("非调度账号(当前账号(职员=%s)类型为:%s), 禁止访问该资源",
                        userModel.getEmployeeId(),
                        userModel.getAccountType()
                )
                );
            }

            if (!ignoredDeskCheck &&
                    BusinessUtil.checkIfDesk(userModel.getAccountType()) &&
                    !userModel.hasDeviceDeskPermission()) {
                /**
                 * 可以允许查看详情
                 */
                if (!request.getMethod().equalsIgnoreCase("get")) {
                    if (!CollectionUtils.isEmpty(userModel.getDeviceDeskServiceScopes())) {
                        throw new ForbidTenantException(String.format("调度台(%s)已过期, 请联系客服.",
                                userModel.getDeviceDeskServiceScopes().stream()
                                        .map(ii -> ii.toString())
                                        .collect(Collectors.joining(";"))
                        ));
                    }
                }
            }

            responseWithFlowTag(response,tenantContext);
            filterChain.doFilter(request, response);
        } catch (Exception ex) {
            HttpStatus status = HttpStatus.BAD_REQUEST;
            String trace = ExceptionUtil.getStackMessage(ex);
            String message = "系统异常, 请联系管理员";

            if (ex instanceof TenantExceptionAbstract) {
                status = HttpStatus.UNAUTHORIZED;
                message = ex.getMessage();
                if (ex instanceof UnAuthorizedTenantException atE) {
                    trace = String.format("系统认证异常token =%s", atE.getData());
                } else {
                    trace = message;
                }
            } else if (!isAfterParseRequestToken) {
                message = "当前操作无效V2, 请联系系统管理员" + ex.getMessage() + ";";
            }

            ResponseEntity responseEntity =
                    ResponseBuilder.build(status, trace, message);

            response.setStatus(responseEntity.getStatusCode().value());

            String error = this.jsonProvider.getJson(responseEntity.getBody());
            response.setHeader(HttpHeaders.CONTENT_TYPE, "application/json");
            responseWithFlowTag(response,tenantContext);

            response.getOutputStream().write(error.getBytes(StandardCharsets.UTF_8));
            response.flushBuffer();
        }
    }

    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) throws ServletException {
        /**
         * 重置当前用户上下文信息
         */
        TenantContext tenantContext = TenantContext.getInstance();
        tenantContext.getUserContext().reset();
        String requestUri = request.getRequestURI();

        /**
         * 高并发的地址
         * 签到前验证及签到; 直接跳过, 采用token进行验证
         */
        if (requestUri.endsWith("/attendance") || requestUri.endsWith("/attendance/pre-validation")) {
            return true;
        }

        if (requestUri.endsWith("/attendance/my-records")) {
            return true;
        }

        if (requestUri.startsWith("/tests")) {
            return true;
        }

        boolean matchedResult =
                excludedUrls.stream()
                        .anyMatch(ii -> {
                                    String excludeUrl = String.format("%s%s", request.getContextPath(), ii);
                                    boolean isMatch = requestUri.startsWith(excludeUrl) || requestUri.contains("/do-confirm");
                                    if (!isMatch && ii.contains("\\w+")) {
                                        Pattern pattern = Pattern.compile(excludeUrl);
                                        Matcher matcher = pattern.matcher(requestUri);
                                        isMatch = matcher.matches();
                                    }
                                    return isMatch;
                                }
                        );
        if (matchedResult) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }


        if (requestUri.contains("/rooms") && "get".equalsIgnoreCase(request.getMethod())) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        /**
         * 针对点名的回复, 先按照这种方式进行处理
         */
        if ("PUT".equalsIgnoreCase(request.getMethod()) && requestUri.startsWith(String.format("%s/roll-calls", request.getContextPath()))) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        if (requestUri.contains(String.format("%s/swagger", request.getContextPath())) ||
                requestUri.contains("/download/resources")) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        return false;
    }

    private void parseRequestToken(HttpServletRequest request,
                                   boolean ignoredDeskCheck) {
        String dispatchToken = null;
        String content = null;
        try {
            dispatchToken = ServletRequestUtil.getDispatchToken(request);
            /**
             * 针对移动端; 如果有accessToken也是可用
             * 比如: 警情SOS上报等功能
             */
            if (!StringUtils.hasLength(dispatchToken)) {
                String requestUri = request.getRequestURI().toLowerCase();
                if (requestUri.contains("sos") || requestUri.contains("setting") || requestUri.contains("done-operation-instruction") ||
                        (requestUri.endsWith("/tenant/users/current") && "GET".equalsIgnoreCase(request.getMethod())) ||
                        (requestUri.endsWith("/messages/do-get-instants") && "POST".equalsIgnoreCase(request.getMethod())) ||
                        (requestUri.endsWith("/do-confirm") && "POST".equalsIgnoreCase(request.getMethod())) ||
                        (requestUri.contains("/rooms") && requestUri.endsWith("/action") && "POST".equalsIgnoreCase(request.getMethod())) ||
                        (requestUri.contains("/roll-calls") && requestUri.endsWith("/reply") && "PUT".equalsIgnoreCase(request.getMethod())) ||
                        (requestUri.endsWith("/websocket/connect") || requestUri.contains("/security-station-rails"))
                        || requestUri.contains("/employees/duty-status")
                ) {
                    String accessToken = ServletRequestUtil.getAccessToken(request);
                    try {
                        if (StringUtils.hasLength(accessToken)) {
                            String employeeId = JwtUtil.getUserIdFromToken(accessToken);
                            UserDetailResponse detailResponse = UserUtil.get(hotCacheProvider,identityRpcProvider,employeeId,DispatchAccountType.APP);
                                   // this.identityRpcProvider.getByEmployeeId(employeeId, DispatchAccountType.APP);

                            String userJsonContent = this.jsonProvider.getJson(detailResponse);
                            dispatchToken = JwtUtil.getTokenFromIdentityData(userJsonContent);
                        }
                    } catch (Exception ex) {
                        logger.error("解析AccessToken({})信息发送异常:{}", accessToken, ex);
                        throw new UnAuthorizedTenantException("账号过期, 请重新登入系统.");
                    }
                }
            }

            if (!StringUtils.hasLength(dispatchToken)) {
                throw new UnAuthorizedTenantException("当前无有效dispatchToken(token不能为空)信息");
            }

            content = JwtUtil.getUserIdFromToken(dispatchToken);

            /**
             * 如果通过token的方式来保持权限
             *
             */
            UserDetailResponse userDetailResponse =
                    this.jsonProvider.toObject(UserDetailResponse.class, content);
            if (userDetailResponse == null) {
                logger.error(String.format("当前无有效dispatchToken信息=%s;content=%s", dispatchToken, content));
                throw new UnAuthorizedTenantException("当前无有效dispatchToken信息, 请联系管理员");
            }

            boolean advancePermission = false;
            try{
                advancePermission =
                        "advance".equalsIgnoreCase(request.getHeader("permission"));
            }catch (Exception ex1) {
                //todo:
            }

            boolean isDesk = BusinessUtil.checkIfDesk(userDetailResponse.getAccountType());
            Collection<String> subScopeCompanyIds = new ArrayList<>();

            /**
             * 只有调度台的账号才需要加载权限; 否则他只需要看到自己的数据
             */
            if(isDesk || advancePermission) {
                if (CompanyLevel.contain(userDetailResponse.getCLevel(), CompanyLevel.Premium)) {
                    Collection<String> cmpScopes = this.dataScopeRpcProvider.getCompanyDataScopeById(userDetailResponse.getOrganizationId());
                    if (!CollectionUtils.isEmpty(cmpScopes)) {
                        subScopeCompanyIds.addAll(cmpScopes);
                    }
                }

                if (BusinessUtil.isSecurityAssociation(userDetailResponse.getInstitutional()) ||
                        BusinessUtil.isEventOrganizer(userDetailResponse.getInstitutional())) {
                    Collection<String> cmpEventOrgScopes =
                            this.dataScopeRpcProvider.getDataScopeById(userDetailResponse.getOrganizationId(),
                                    DispatchDataScopeType.EventOrganizer);
                    if (!CollectionUtils.isEmpty(cmpEventOrgScopes)) {
                        subScopeCompanyIds.addAll(cmpEventOrgScopes);
                    }
                }
            }

            /**
             * 初始化当前token和对应的tenant user id
             * 获取当前用户的权限
             */
            TenantEmployeeContext.PermissionModel permissionModel =
                    UserUtil.getPermission(
                            hotCacheProvider,
                            identityRpcProvider,
                            employeeReaderRpcProvider,
                            dataScopeRpcProvider,
                            subScopeCompanyIds,
                            userDetailResponse,
                            projectReaderRpcProvider,
                            advancePermission
                    );

            TenantContext tenantContext = TenantContext.getInstance();
            TenantEmployeeContext.TenantUserModel userModel = tenantContext.getUserContext()
                            .init(userDetailResponse,subScopeCompanyIds,permissionModel);

            executeAddProviderTag(request,tenantContext,userModel.getTrafficTag());

            try {
                /**
                 * changed by lhc:2024-12-15
                 */
                if (isDesk || advancePermission) {
                    RdDeviceDeskPermissionResponse permissionResponse =
                            this.identityRpcProvider.getMergedDeviceDeskPermission(
                                    userDetailResponse.getId(),
                                    userDetailResponse.getEmployeeId(),
                                    userDetailResponse.getSupervise()==null?false:userDetailResponse.getSupervise().booleanValue(),
                                    userModel.getAccountType(),
                                    userModel.getOrgInstitutional(),
                                    userModel.getOrgName(),
                                    userModel.getOrganizationId()
                            );

                    if (permissionResponse != null && !CollectionUtils.isEmpty(permissionResponse.getServiceScopes())) {
                        /**
                         * 分为调度台和被调度者信息
                         */
                        String ipAddress = ServletRequestUtil.getIpAddress(request);
                        Collection<TenantEmployeeContext.TenantDeviceDeskServiceScopeModel>
                                allowedServiceScopes = permissionResponse.getAllowedServiceScopes(ipAddress)
                                .stream().map(ix -> {
                                    return TenantEmployeeContext.TenantDeviceDeskServiceScopeModel.create(
                                            ix.getId(),
                                            ix.getName(),
                                            ix.getNumber(),
                                            ix.getBeginDate(),
                                            ix.getEndDate(),
                                            ix.getBoundEmployeeId(),
                                            ix.getAssignedSuperviseDepartId(),
                                            ix.getIpAddress(),
                                            ix.getOrganizationId(),
                                            ix.getDesc(),
                                            ix.getIr()
                                    );
                                }).collect(Collectors.toList());

                        userModel.assignDeviceDeskServiceScopes(allowedServiceScopes);
                    }

                    /**
                     * 针对可忽略调度检测url, 我们直接忽略, 比如：获取当前用户信息
                     */
                    if (!ignoredDeskCheck && CollectionUtils.isEmpty(userModel.getDeviceDeskServiceScopes())) {
                        /**
                         * 仅针对监管调度台; 我们才进行IP的限制验证
                         */
                        /*
                        changed by XQ20240111902: 禁用IP的调度台限制
                        if (permissionResponse != null && !CollectionUtils.isEmpty(permissionResponse.getServiceScopes())) {
                            String errorMessage =
                                    String.format("当前调度台(姓名=%s,当前ip=%s)不在允许的地点(允许调度的ip=%s)进行调度",
                                            userModel.getName(),
                                            ipAddress,
                                            permissionResponse.getServiceScopes()
                                                    .stream().map(ix -> ix.getIpAddress()).distinct()
                                                    .collect(Collectors.joining(";"))
                                    );
                            throw new ForbidTenantException(errorMessage);
                        }
                         */
                    }
                } else{
                    TenantSecurityEmployeeCache employeeCache =
                            UserUtil.getSecurityEmployeeCache(
                                    hotCacheProvider, employeeReaderRpcProvider,
                                    syncRpcWriterProvider, userDetailResponse.getEmployeeId());
                    if (employeeCache != null) {
                        userModel.assignTenantSecurityEmployee(
                                TenantEmployeeContext.TenantSecurityEmployeeModel.create(
                                        employeeCache.getSecurityStationId(),
                                        employeeCache.getSecurityStationName(),
                                        employeeCache.getSuperviseDepartId(),
                                        employeeCache.getTencentUserId(),
                                        employeeCache.getTenantUserName(),
                                        employeeCache.getTenantUserIdCard(),
                                        employeeCache.getSuperviseDepartName(),
                                        employeeCache.getResourceTypes(),
                                        employeeCache.getCreatedTime()
                                ));
                    } else {
                        /**
                         * 针对找不到该职员的; 系统进行记录
                         */
                        try {
                            this.warnContentRpcProvider.dispatch(TenantWarnContentRequest.create(
                                    ContentType.Employee,
                                    userDetailResponse.getEmployeeId(),
                                    "AuthFilter"));
                        } catch (Exception ex) {
                            logger.error("AuthFilter with accountType={} and employeeId={},error={}",
                                    userDetailResponse.getAccountType(),
                                    userDetailResponse.getEmployeeId(),
                                    ExceptionUtil.getStackMessage(ex)
                            );
                        }
                    }
                }

                TenantContext.getInstance().getUserContext().init(userModel);
            } catch (Exception ex) {
                if (ex.getClass().isAssignableFrom(ForbidTenantException.class) || ex.getClass().isAssignableFrom(UnAuthorizedTenantException.class)) {
                    throw ex;
                }

                throw new BadTenantException("解析用户信息无效:" + ex.getMessage(), ex);
            }
        } catch (UnAuthorizedTenantException ex) {
            logger.error("当前Token异常:token={};content={}", dispatchToken, content);
            throw ex;
        } catch (Exception ex) {
            logger.error("当前无有效dispatchToken信息={};content={};", dispatchToken, content, ex);
            throw ex;
        }
    }

    private void responseWithFlowTag(HttpServletResponse response,TenantContext tenantContext) {
        try {
            if (tenantContext.getAttachValue(TenantContext.PROVIDER_FLOW_TAG_NAME) instanceof TrafficTagValueType tagValueType) {
                response.addHeader(TenantContext.PROVIDER_FLOW_TAG_NAME, tagValueType.getTagStringValue());
            }

            response.addHeader(TenantContext.REST_REQUEST_FLOW_TAG_NAME, String.valueOf(tenantContext.getAttachValue(TenantContext.FLOW_TAG_NAME)));
            response.addHeader(TenantContext.REST_VERSION, "1.5");
            response.addHeader("t_node", this.tenantNode);
        } catch (Exception ex) {
            //todo: 增值的功能, 仅仅只是补充
        }
    }

    private void executeAddProviderTag(ServletRequest request,TenantContext tenantContext, String fallBackTrafficTag){
        DubboCommonUtils.addContextTag(request,tenantContext,fallBackTrafficTag);
    }
}
