package com.bcxin.tenant.open.rest.apis.controllers.responses;

import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.enums.*;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.infrastructures.ResponseAbstract;
import com.fasterxml.jackson.annotation.JsonIgnore;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;
import lombok.Setter;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Schema(name = "UserInfoResponse",title = "UserInfoResponse 当前的用户信息")
@Getter
public class UserInfoResponse extends ResponseAbstract {
    @Schema(name = "id", title = "id")
    private final String id;
    @Schema(name = "employeeId", title = "employeeId")
    private final String employeeId;
    @Schema(name = "name", title = "姓名")
    private final String name;

    @Schema(name = "organizationId", title = "organizationId")
    private final String organizationId;

    @Schema(name = "orgInstitutional", title = "机构类型")
    private final String orgInstitutional;

    /**
     * 最细的腾讯资源信息
     */
    @Schema(name = "tencentUserId", title = "tencentUserId")
    private final String tencentUserId;

    @Schema(name = "accountType", title = "APP/Device硬件设备/Desk调度台")
    private final DispatchAccountType accountType;

    @Schema(name = "supervise", title = "是否为监管身份")
    private final boolean supervise;

    @Schema(name = "dutySignInType", title = "最新签到状态(None=未签到;SignIn=签到;SignOut=签退); 仅支持APP/硬件设备")
    private DutySignInType dutySignInType;

    @Schema(name = "org_lat", title = "org_lat 企业所在y轴坐标")
    private Double orgLatitude;

    @Schema(name = "org_lon", title = "org_lon 企业所在x轴坐标")
    private Double orgLongitude;

    @Schema(name = "role", title = "不在根据此控制调度台类型; Urban 市局; SubUrb 分局;PopulationUrban 人口市局;PopulationSubUrb 人口分局;PopulationPoliceStation 人口派出所; Company 企业")
    private RoleType role;

    @Schema(name = "deviceDeskServiceScopes", title = "调度范围")
    private Collection<TenantDeviceDeskServiceScopeResponse> deviceDeskServiceScopes;

    @Schema(name = "deviceDeskServiceScopeIds", title = "调度范围权限Ids")
    private Collection<String> getDeviceDeskServiceScopeIds() {
        if (this.getDeviceDeskServiceScopes() == null) {
            return Collections.EMPTY_LIST;
        }

        return this.getDeviceDeskServiceScopes().stream()
                .map(ii -> ii.getAssignedSuperviseDepartId())
                .distinct().collect(Collectors.toList());
    }

    @Schema(name = "responsibleOfStationIds", title = "负责的驻勤点列表")
    private String[] responsibleOfStationIds;
    
    @Schema(name = "securityStationId", title = "当前用户所在的驻勤点; 仅非调度台有值;非#1的驻勤为有效驻勤点; 可以使用此值来获取驻勤点的详情信息; 比如: 签到界面")
    private String securityStationId;

    @Schema(name = "superviseRegionCode", title = "所属监管区域编码")
    private String superviseRegionCode;

    @Schema(name = "noSuperviseScopePermissions", title = "非监管的数据权限")
    private String[] noSuperviseScopePermissions;

    @Schema(name = "cLevels", title = "企业拥有的级别")
    private Collection<CompanyLevel> cLevels;

    @Setter
    @Schema(name = "permission", title = "附加权限信息")
    private TenantEmployeeContext.PermissionModel permission;


    @Schema(name = "superviseDesk", title = "是否为监管调度台")
    public boolean isSuperviseDesk() {
        if (CollectionUtils.isEmpty(this.getDeviceDeskServiceScopes())) {
            return false;
        }

        return this.getDeviceDeskServiceScopes().stream()
                .anyMatch(ii -> StringUtils.hasLength(ii.getAssignedSuperviseDepartId()));
    }

    @Schema(name = "deviceName", title = "设备名称")
    public String getDeviceName() {
        if (CollectionUtils.isEmpty(this.getDeviceDeskServiceScopes())) {
            return null;
        }

        /**
         * 优先显示当前的调度台名称
         */
        String deviceName = this.getDeviceDeskServiceScopes()
                .stream().filter(ii -> StringUtil.isEqual(ii.getOrganizationId(), this.getOrganizationId()))
                .map(ii -> ii.getName())
                .findFirst().orElse(null);

        if (!StringUtil.isEmpty(deviceName)) {
            return deviceName;
        }

        return this.getDeviceDeskServiceScopes().stream()
                .map(ii -> ii.getName()).distinct()
                .findFirst().orElse(null);
    }


    @Schema(name = "deskTypes", title = "返回调度台类型;控制可以显式哪些调度台: 指挥调度台/内保调度台/临保调度台")
    public Collection<DeskType> getDeskTypes() {
        if (this.getRole() == RoleType.Proprietor) {
            return Stream.of(DeskType.TemporarySecurity, DeskType.Proprietor).collect(Collectors.toSet());
        } else {
            return Stream.of(DeskType.TemporarySecurity, DeskType.Normal).collect(Collectors.toSet());
        }
    }

    @Schema(name = "dispatchNo", title = "对讲的调度编号;创建群组之后;创建一个子组织并以子组织的身份加入群中;subAccount的规则={{dispatchNo}}_{{gid群组的ID}}")
    public String getDispatchNo() {
        return this.getTencentUserId();
    }

    public UserInfoResponse(String id, String employeeId,
                            String name, String organizationId,
                            String orgInstitutional, String tencentUserId,
                            DispatchAccountType accountType,
                            boolean supervise,
                            String[] responsibleOfStationIds,
                            String superviseRegionCode,
                            String[] noSuperviseScopePermissions,
                            Collection<CompanyLevel> cLevels) {
        this.id = id;
        this.employeeId = employeeId;
        this.name = name;
        this.organizationId = organizationId;
        this.orgInstitutional = orgInstitutional;
        this.tencentUserId = tencentUserId;
        this.accountType = accountType;
        this.supervise = supervise;
        this.responsibleOfStationIds = responsibleOfStationIds;
        this.superviseRegionCode = superviseRegionCode;
        this.noSuperviseScopePermissions = noSuperviseScopePermissions;
        this.cLevels = cLevels;
    }

    public void assignOrgLatLon(Double orgLatitude,Double orgLongitude) {
        if (orgLatitude == null || orgLongitude == null) {
            return;
        }
        if (orgLatitude <= 1.1 || orgLongitude <= 1.1) {
            return;
        }
        this.orgLatitude = orgLatitude;
        this.orgLongitude = orgLongitude;
    }

    public void assignDeviceDeskServiceScopes(Collection<TenantDeviceDeskServiceScopeResponse> deviceDeskServiceScopes) {
        this.deviceDeskServiceScopes = deviceDeskServiceScopes;

        /**
         * 针对晋城, 临时改为派出所调度台
         */
        if (!CollectionUtils.isEmpty(deviceDeskServiceScopes) &&
                (this.getRole() == RoleType.Urban || this.getRole() == RoleType.SubUrb)) {
            if (deviceDeskServiceScopes.stream().anyMatch(ii -> ii.getName().contains("派出所"))) {
                this.role = RoleType.PoliceStation;
            }
        }
    }

    public void assignDutySignInType(DutySignInType dutySignInType) {
        this.dutySignInType = dutySignInType;
    }

    public void assignSecurityStationId(String securityStationId) {
        this.securityStationId = securityStationId;
    }

    public void assignRole(String institutional,String orgName) {

        this.role = RoleType.create(institutional, orgName);
    }

    public static UserInfoResponse create(String id, String employeeId,
                                          String name, String organizationId,
                                          String orgInstitutional,
                                          String tencentUserId,
                                          DispatchAccountType accountType,
                                          boolean supervise,
                                          String[] responsibleOfStationIds,
                                          String superviseRegionCode,
                                          String[] noSuperviseScopePermissions,
                                          Collection<CompanyLevel> cLevels) {
        return new UserInfoResponse(id, employeeId, name, organizationId, orgInstitutional, tencentUserId,
                accountType, supervise,responsibleOfStationIds,superviseRegionCode,noSuperviseScopePermissions,cLevels);
    }

    @Getter
    public static class TenantDeviceDeskServiceScopeResponse extends ResponseAbstract {
        /**
         * 设备Id
         */
        private final String id;
        /**
         * 腾讯的资源号
         */
        private final String number;

        /**
         * 调度台名称
         */
        private final String name;

        /**
         * 有效期开始时间
         */
        private final Timestamp beginDate;
        /**
         * 有效期结束时间
         */
        private final Timestamp endDate;
        /**
         * 被绑定的人
         */
        private final String boundEmployeeId;

        /**
         * 被授权的监管归属的部门Id
         */
        private final String assignedSuperviseDepartId;

        /**
         * ipAddress白名单
         */
        private final String ipAddress;

        private final String organizationId;

        private final byte ir;

        private final ResourceType resourceType;

        private final String desc;

        public TenantDeviceDeskServiceScopeResponse(String id,
                                                    String name,
                                                    String number,
                                                    Timestamp beginDate, Timestamp endDate,
                                                    String boundEmployeeId,
                                                    String assignedSuperviseDepartId,
                                                    String ipAddress,
                                                    String organizationId,
                                                    byte ir, ResourceType resourceType,
                                                    String desc) {
            this.id = id;
            this.number = number;
            this.name = name;
            this.beginDate = beginDate;
            this.endDate = endDate;
            this.boundEmployeeId = boundEmployeeId;
            this.assignedSuperviseDepartId = assignedSuperviseDepartId;
            this.ipAddress = ipAddress;
            this.organizationId = organizationId;
            this.ir = ir;
            this.resourceType = resourceType;
            this.desc = desc;
        }

        public static TenantDeviceDeskServiceScopeResponse create(String id,
                                                                  String name,
                                                                  String number,
                                                                  Timestamp beginDate, Timestamp endDate,
                                                                  String boundEmployeeId,
                                                                  String assignedSuperviseDepartId,
                                                                  String ipAddress,
                                                                  String organizationId,
                                                                  byte ir, ResourceType resourceType,
                                                                  String desc) {
            return new TenantDeviceDeskServiceScopeResponse(id,
                    name,
                    number, beginDate, endDate,
                    boundEmployeeId, assignedSuperviseDepartId, ipAddress,organizationId,ir, resourceType,desc);
        }
    }
}
