package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.UserDetailResponse;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.requests.AttendanceValidationRestRequest;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.open.rest.apis.utils.ServletRequestUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.ServletRequest;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Date;

@Tag(name = "TestController", description = "性能测试")
@RestController
@RequestMapping("/tests")
public class TestController extends ControllerAbstract {
    private final JsonProvider jsonProvider;
    private final HotCacheProvider hotCacheProvider;

    public TestController(JsonProvider jsonProvider, HotCacheProvider hotCacheProvider) {
        this.jsonProvider = jsonProvider;
        this.hotCacheProvider = hotCacheProvider;
    }

    @Operation(
            summary = "validation", description = "验证空方法的速度",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/validation")
    public ResponseEntity doValidate(@RequestBody AttendanceValidationRestRequest restRequest,
                                                                   @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                                                   @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                                                   @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                                                   @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                                                   ServletRequest servletRequest) {
        return this.ok("OK");
    }

    @Operation(
            summary = "validation2", description = "validation2 验证Jwt Access的速度",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/validation2")
    public ResponseEntity doValidate2(@RequestBody AttendanceValidationRestRequest restRequest,
                                     @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                     @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                     @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                     @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                     ServletRequest servletRequest) {
        String accessToken = ServletRequestUtil.getAccessToken(servletRequest);
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);
        return this.ok(employeeId);
    }

    @Operation(
            summary = "doValidate3", description = "doValidate3 验证Jwt dispatchToken的速度",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/validation3")
    public ResponseEntity doValidate3(@RequestBody AttendanceValidationRestRequest restRequest,
                                      @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                      @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                      @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                      @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                      ServletRequest servletRequest) {
        String content = JwtUtil.getUserIdFromToken(ServletRequestUtil.getDispatchToken(servletRequest));
        /**
         * 如果通过token的方式来保持权限
         *
         */
        UserDetailResponse userDetailResponse =
                this.jsonProvider.toObject(UserDetailResponse.class, content);
        return this.ok(userDetailResponse);
    }

    @Operation(
            summary = "validation5", description = "validation5 验证hotCache携带本地的性能",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/validation5")
    public ResponseEntity doValidate5(@RequestBody AttendanceValidationRestRequest restRequest,
                                      @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                      @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                      @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                      @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                      ServletRequest servletRequest) {
       Date value = hotCacheProvider.get("test_validation5", () -> {
            return new Date();
        }, 60 * 60, Date.class, true);

        return this.ok(value);
    }

    @Operation(
            summary = "validation6", description = "validation6 验证hotCache无本地缓存的性能",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/validation6")
    public ResponseEntity doValidate6(@RequestBody AttendanceValidationRestRequest restRequest,
                                      @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                      @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                      @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                      @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                      ServletRequest servletRequest) {
        Date value = hotCacheProvider.get("test_validation6", () -> {
            return new Date();
        }, 60 * 60, Date.class, false);

        return this.ok(value);
    }
}
